#!perl 
#===============================================================================
#
# Copyright (C) 2000 - 2014 by SAP AG,  http://www.sap.com.
# All rights reserved.
#
# This software is the confidential and proprietary information of SAP AG.
# You shall not disclose such confidential information and shall use it only
# in accordance with the terms of the license agreement you entered into with SAP.
#
#       AUTHOR: i072928
#

=head1 NAME

HostsDialog

=head1 DESCRIPTION

Sets hosts values.
If root credentials are required for the hosts connection, the user
will be asked for them.
=cut

#===============================================================================
package LCM::Gui::Dialogs::Rename::HostsDialog;

use strict;

use Wx qw ( wxVERTICAL wxHORIZONTAL wxALIGN_CENTER_VERTICAL wxEXPAND wxGROW wxALL wxDefaultPosition wxDefaultSize wxDefaultValidator
	wxBORDER_SIMPLE wxLC_SINGLE_SEL wxLC_REPORT wxLIST_AUTOSIZE wxLIST_NEXT_ALL wxLIST_STATE_SELECTED wxTE_PASSWORD wxLIST_AUTOSIZE_USEHEADER);
use Wx::Event qw (EVT_BUTTON EVT_LIST_ITEM_SELECTED EVT_LIST_ITEM_DESELECTED);

use SDB::Install::Gui::Control::Label;
use SDB::Install::Gui::Control::Button;
use SDB::Install::Gui::Design;
use LCM::Gui::Dialogs::Rename::AlterHostPropsDialog;
use base qw (SAPDB::Install::Wizard::Dialog::Base);

sub new {
	my ( $self ) = shift->SUPER::new ( @_ );

	my $is_gtk1    = Wx::wxGTK1 ();
	my $app        = Wx::wxTheApp;
	my $instconfig = $app->{instconfig};

	$self->{id}   = 'HostsDialog';
	$self->{name} = "Define Hosts Properties";

	# sizers
	my $outersizer = new Wx::BoxSizer ( wxVERTICAL );
	$self->SetSizer ( $outersizer );

	my $sizer = new Wx::BoxSizer ( wxVERTICAL );
	$self->{'sizer_inner'} = $sizer;
	$outersizer->Add ( $sizer, 1, wxALL | wxEXPAND | wxGROW, 20 );
	$self->SetAutoLayout ( 1 );
	return $self;
}

sub init {
	my ( $self, $config ) = @_;
	if ( $self->{isInitialized} ) {
		return {};
	}
	my $app        = Wx::wxTheApp;
	my $instconfig = $app->{instconfig};
	
	if ( $instconfig->isDistributedSystem (1) ) {
		# display root credentials fields if they are are needed
		$self->_addHostsProperties ( $instconfig, $self->{'sizer_inner'}, Wx::wxGTK1 () );
		$self->_fillHostsList ( $instconfig );
		$self->SetAutoLayout ( 1 );
	}

	$self->{isInitialized} = 1;
	return {};
}

sub _fillHostsList {
	my ( $self, $instconfig ) = @_;
    my $instance = $instconfig->getOwnInstance ();
    my @hosts = sort @{ $instance->get_allhosts () };	
	my $listCtrl = $self->{hostsList};

	foreach my $originHost ( @hosts ) {
		if(defined $originHost){
			my $count = $listCtrl->GetItemCount ();
			my $item = new Wx::ListItem ();
			$item->SetId          ( $count );
			$listCtrl->InsertItem ( $item );
	
			$listCtrl->SetItem ( $count, 0, $originHost );
			$listCtrl->SetItem ( $count, 1, $originHost );
		}
	}

	$self->adjustListCtrlColumnWidths ( $listCtrl );

	return 1;
}

sub _addHostsProperties {
	my ( $self, $instconfig, $sizer, $is_gtk1 ) = @_;

	# title
	my $title = new Wx::StaticText ( $self, -1, 'Enter Hosts Properties', wxDefaultPosition,
		wxDefaultSize );
	$title->SetFont ( SDBFONT_DialogTitle ) if ( !$is_gtk1 );
	$sizer->Add ( $title, 0 );
	$sizer->AddSpacer ( 20 );
	
	if($instconfig->isDistributedSystem ()){
		# display root credentials fields only if needed
		my $fieldsSizer = $self->_createFieldsSizer ( $instconfig );
		$sizer->Add ( $fieldsSizer, 0, wxEXPAND );
		$sizer->AddSpacer ( 10 );
	}

	my $addHostsSizer = $self->_createHostsSizer ( $instconfig );
	$sizer->Add ( $addHostsSizer, 1, wxEXPAND | wxGROW );

	return 1;
}

sub _createHostsSizer {
	my ( $self, $instconfig ) = @_;

	my $sizer = new Wx::BoxSizer ( wxVERTICAL );
	my $list =
		Wx::ListCtrl->new ( $self, 0, wxDefaultPosition, wxDefaultSize,
		wxBORDER_SIMPLE | wxLC_REPORT | wxLC_SINGLE_SEL | wxEXPAND | wxGROW,
		wxDefaultValidator );

	$list->InsertColumn ( 0, "Source Host Name" );
	$list->InsertColumn ( 1, "Target Host Name" );

	$list->SetColumnWidth ( 0, wxLIST_AUTOSIZE_USEHEADER );
	$list->SetColumnWidth ( 1, wxLIST_AUTOSIZE_USEHEADER );

	$list->{minColWidths} = [ 200, 200 ];

	$self->{hostsList} = $list;
	$sizer->Add ( $self->{hostsList}, 1, wxEXPAND | wxGROW );
	$sizer->AddSpacer ( 10 );

	my $editButton =
		new SDB::Install::Gui::Control::Button ( $self, -1, '&Edit Host...', wxDefaultPosition,
		wxDefaultSize );
	$self->{editButton} = $editButton;
	$editButton->Enable ( 0 );
	EVT_BUTTON ( $self, $editButton->GetId, \&_onEditHost );
	$sizer->Add ( $editButton, 0 );
	$sizer->AddSpacer ( 5 );

	EVT_LIST_ITEM_SELECTED (
		$self,
		$list->GetId,
		sub {
			$editButton->Enable ( 1 );
		}
	);
	EVT_LIST_ITEM_DESELECTED (
		$self, $list,
		sub {
			$editButton->Enable ( 0 );
		}
	);

	return $sizer;
}

sub _createFieldsSizer {
	my ( $self, $instconfig ) = @_;

	my $sizer = new Wx::FlexGridSizer ( 2, 2, 5, 10 );
	$sizer->SetFlexibleDirection ( wxHORIZONTAL );
	$sizer->AddGrowableCol ( 1 );
	if($instconfig->isUseSAPHostagent()){
	    $self->createParameterInputFiled($sizer,$instconfig,'HostagentPassword',1,wxTE_PASSWORD);       
    } else {
    if ( defined $instconfig->{params}->{'RootUser'}->{skip}
        && !$instconfig->{params}->{'RootUser'}->{skip} ) {
        $self->createParameterInputFiled($sizer,$instconfig,'RootUser',1,undef);        
    }
        $self->createParameterInputFiled($sizer,$instconfig,'RootPassword',0,wxTE_PASSWORD);
	}
	return $sizer;
}

sub _onEditHost {
	my ( $self ) = @_;

	my $listCtrl       = $self->{hostsList};
	my $itemIndex      = $listCtrl->GetNextItem ( -1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED );
	my $targetHostName = $listCtrl->GetItem ( $itemIndex, 1 )->GetText ();

	my $dlg = new LCM::Gui::Dialogs::Rename::AlterHostPropsDialog ( $self );
	$dlg->SetHost ( $targetHostName );

	my $retCode = $dlg->ShowModal ();
	$dlg->Destroy ();
	if ( $retCode != $dlg->GetAffirmativeId () ) {
		return;
	}

	$targetHostName = $dlg->GetHost ();
	$listCtrl->SetItem ( $itemIndex, 1, $targetHostName );
	$self->adjustListCtrlColumnWidths ( $listCtrl );
}

sub adjustListCtrlColumnWidths() {
	my ( $self, $listCtrl ) = @_;

	my $colCount = $listCtrl->GetColumnCount ();
	for ( my $i = 0; $i < $colCount; $i++ ) {
		$listCtrl->SetColumnWidth ( $i, wxLIST_AUTOSIZE );
		my $colWidth = $listCtrl->GetColumnWidth ( $i );
		if ( $colWidth < $listCtrl->{minColWidths}->[$i] ) {
			$listCtrl->SetColumnWidth ( $i, $listCtrl->{minColWidths}->[$i] );
		}
	}
}

sub checkField {
	my ( $self, $instconfig, $msgLst, $paramId, $fieldItInSelf ) = @_;
    my $value = $self->{$fieldItInSelf}->GetValue ();
	return undef unless $value;
	return $instconfig->setValue ( $paramId, $value, $msgLst );
}

sub Process {

	# NEXT button was pressed on dialog
	my ( $self, $config ) = @_;
	my $app = Wx::wxTheApp ();
	my $instconfig = $app->{instconfig};
	$instconfig->resetMsgLstContext ();
	my $msgLst = new SDB::Install::MsgLst ();
	
	# validate hosts
	if ( $instconfig->isDistributedSystem(1) ) {
		my $listCtrl = $self->{hostsList};
		my $count = $listCtrl->GetItemCount ();
		for ( my $i = 0; $i < $count; $i++ ) {
			my $sourceHostName = $listCtrl->GetItem ( $i, 0 )->GetText ();
			my $targetHostName = $listCtrl->GetItem ( $i, 1 )->GetText ();
			my $errorList = $instconfig->validateHost ( $sourceHostName, $targetHostName );
			if($errorList && !$errorList->isEmpty()){
				$msgLst->appendMsgLst ( $errorList ) ;
				$app->ShowErrorMsg ( 'Configuration error', $msgLst );
                return undef;
			}
		}
		if(!$instconfig->checkAllMapValues('HostMap')){
			$app->ShowErrorMsg('Configuration error', $instconfig->getErrMsgLst());
			return undef;
		}
	}
	
	if(exists $instconfig->{_remoteHosts} && defined $instconfig->{_remoteHosts}){
	    $instconfig->{_remoteHosts}->destroy();
	    $instconfig->{_remoteHosts} = undef if(!$instconfig->isUseSAPHostagent());# set remote hosts to undef only when --remote_execution=ssh
	}
	if ( $instconfig->isDistributedSystem() ) {
		if(!$instconfig->isUseSAPHostagent()){
			if( !$self->checkField ( $instconfig, $msgLst, 'RootUser', 'RootUser' ) ) {
		        $msgLst->addError ( 'Error while trying to verify the root credentials.', $instconfig->getErrMsgLst() );
		        $app->ShowErrorMsg ( 'Configuration error', $msgLst );
		        return undef;
		    }
		
		    if ( !$instconfig->{params}->{RootPassword}->{skip} ) {
		    	unless ( $self->checkField ( $instconfig, $msgLst, 'RootPassword', 'RootPassword' ) ) {
		    		$msgLst->addError ( 'Error while trying to verify the root credentials.', $instconfig->getErrMsgLst() );
		    		$app->ShowErrorMsg ( 'Configuration error', $msgLst );
		    		return undef;
		    	}
		    }
		}
		elsif ( !$instconfig->{params}->{HostagentPassword}->{skip}) {
            unless ( $self->checkField ( $instconfig, $msgLst, 'HostagentPassword', 'HostagentPassword' ) ) {
                $msgLst->addError ( 'Error while trying to verify the sapadm credentials.', $instconfig->getErrMsgLst() );
                $app->ShowErrorMsg ( 'Configuration error', $msgLst );
                return undef;
            }
        }
    }

	# detect internal network configuration and certificates
	if ($instconfig->isDistributedSystem(1) && 
		$instconfig->isCertificatesHostMapNeeded() && 
		!$instconfig->fillCertificatesHostmapDefaultValuesAndLocalData()) {
        $msgLst->addMessage ( 'Cannot detect external host names default values.' );
	}

	if ( !$msgLst->isEmpty () ) {
		$app->ShowErrorMsg ( 'Configuration error', $msgLst );
		return undef;
	}
	
	if(!$instconfig->isCertificatesHostMapNeeded()) {
        my $wizard = $self->GetWizard ();
        $instconfig->AddMessage ( 'Skipping external host names properties dialog.' );
        $wizard->SkipDialog ( 'CertificatesDialog' );
    }

	return {};
}

sub createParameterInputFiled{
	my($self,$parentContainer,$instconfig,$paramID,$isMandatory,$style) = @_;
	my $parameter = $instconfig->{params}->{$paramID};
    my $label = new SDB::Install::Gui::Control::Label ( $self, -1, $parameter->{str}, $isMandatory);
    $parentContainer->Add ( $label, 0, wxALIGN_CENTER_VERTICAL );
    $self->{$paramID} = new Wx::TextCtrl ( $self, -1, $instconfig->getDefault($paramID) ? $instconfig->getDefault($paramID) : "", 
                                           wxDefaultPosition, [ 200, -1 ], $style );
    $parentContainer->Add ( $self->{$paramID}, 0, wxALIGN_CENTER_VERTICAL );
}
1;