package LCM::MediumAnalyzer;

use strict;

use File::Spec;
use LCM::FileUtils;
use File::Basename;
use LCM::Manifests::XS2ApplicationManifest;
use SDB::Install::SysVars qw($currentPlatformName $isWin $path_separator);
use SDB::Install::Globals qw ($gKeynameEngine
                              $gKeynameClient
                              $gKeynameStudio
                              $gKeynameAFL
                              $gFlavourCockpit);
use SDB::Install::System qw (listDir isRegularFile isDirectory);
use LCM::ComponentManager::MediumComponentManager qw(@componentDirNames);
use LCM::Component qw($componentKeynameToBatchKey);
use LCM::Manifests::ManifestFactory;

my $HANA_DIGITAL_DIRECTORY = 'SAP_HANA_DATABASE_EXPRESS'; # Old value was SAP_HANA_DIGITAL_DATABASE
my $HANA_DIGITAL_FOR_COCKPIT_DIRECTORY = 'SAP_HANA_DATABASE_EXPRESS_FOR_COCKPIT';
my $INSTALER_HDB_DIRECTORY = '__installer.HDB';
my $server = $componentKeynameToBatchKey->{$gKeynameEngine};
my $client = $componentKeynameToBatchKey->{$gKeynameClient};
my $studio = $componentKeynameToBatchKey->{$gKeynameStudio};
my $afl    = $componentKeynameToBatchKey->{$gKeynameAFL};
my $lss    = 'lss';
my $epmmds = 'epmmds';
my $eml    = 'eml';

sub new {
    my ($class, $configuration) = @_;
    my $self = bless({}, $class);
    $self->_setConfiguration($configuration);
    return $self;
}

sub analyze {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $defaultDvdPath = $configuration->getDefault('DvdPath');
    my $defaultComponentDirs = $configuration->getDefault('ComponentDirs');

    return 1 if(defined($defaultDvdPath));
    return 1 if(defined($defaultComponentDirs));
    my $installerDir = $self->_getInstallerDir();
    return 1 if ($installerDir =~ /hdblcm_installer_kit/);

    if ($self->_isDVD()) {
        my $dvdDir = $self->_getDvdDir();
        $configuration->setDefault('DvdPath', $dvdDir, 1);
        return 1;
    }

    return 1 if($self->_productionShareRelWeekstone());
    return 1 if($self->_productionShareDevBuild());
    return 1 if($self->_extractedSMPArchives());
    return undef;
}

sub _getConfiguration {
    my ($self) = @_;
    return $self->{_config};
}

sub _setConfiguration {
    my ($self, $configuration) = @_;
    $self->{_config} = $configuration;
}

sub _isDVD {
    my ($self) = @_;
    my $dvdDir = $self->_getDvdDir();

    return defined($dvdDir) && (-f File::Spec->catfile($dvdDir, 'MID.XML'))
}

sub _getDvdDir {
    my ( $self, ) = @_;
    my $installerDir = $self->_getInstallerDir();

    return undef if(!defined($installerDir));
    return MyRealPath(File::Spec->catfile($installerDir, '..', '..'));
}

sub _getInstallerDir {
    my ($self) = @_;

    if (!defined($self->{_installer})) {
        $self->{_installer} = new LCM::Installer();
    }

    my $installer = $self->{_installer};
    my $installerRuntimeDir = $installer->getRuntimeDir();

    return MyRealPath(File::Spec->catfile($installerRuntimeDir, '..'));
}

sub _productionShareRelWeekstone {
    my ($self) = @_;
    my $installerDir = $self->_getInstallerDir();
    my $weekstoneRoot = MyRealPath(File::Spec->catfile($installerDir, '..', '..', '..'));
    my $serverDir = $self->_getServerComponentDirectory($installerDir, $weekstoneRoot);
    my $componentDirCandidates = [
        $serverDir,
        File::Spec->catfile($weekstoneRoot, $client, $currentPlatformName, 'SAP_HANA_CLIENT', $client),
        File::Spec->catfile($weekstoneRoot, $studio, $currentPlatformName, 'SAP_HANA_STUDIO', $studio),
        File::Spec->catfile($weekstoneRoot, $afl, $currentPlatformName, 'SAP_HANA_AFL', 'packages'),
        File::Spec->catfile($weekstoneRoot, $epmmds, $currentPlatformName, 'SAP_HANA_EPM-MDS', 'packages'),
        File::Spec->catfile($weekstoneRoot, $eml, $currentPlatformName, 'SAP_HANA_EML_AFL', 'packages'),
        File::Spec->catfile($weekstoneRoot, $lss, $currentPlatformName, 'SAP_HANA_LSS', 'packages'),
    ];
    my @componentDirs = grep { -f File::Spec->catfile($_, 'manifest') } @{$componentDirCandidates};
# Bug: 158457
    if (@componentDirs) {
        return 1 if ($self->_isCockpitFlavour($serverDir));
    }
    return $self->_handleComponents(\@componentDirs);
}

sub _getServerComponentDirectory {
    my ($self, $installerDir, $weekstoneRoot) = @_;

    if($installerDir =~ m!/\Q$HANA_DIGITAL_FOR_COCKPIT_DIRECTORY\E/?$!){
        return File::Spec->catfile($weekstoneRoot, 'hde_cockpit', $currentPlatformName, $HANA_DIGITAL_FOR_COCKPIT_DIRECTORY, $server);
    }
    if($installerDir =~ m!/\Q$HANA_DIGITAL_DIRECTORY\E/?$!){
        return File::Spec->catfile($weekstoneRoot, 'hde', $currentPlatformName, $HANA_DIGITAL_DIRECTORY, $server);
    }
    return File::Spec->catfile($weekstoneRoot, $server, $currentPlatformName, 'SAP_HANA_DATABASE', $server);
}

sub _isCockpitFlavour {
    my ($self, $serverDir) = @_;
    return 0 if (! -d $serverDir);
    my $pathToServerManifest = File::Spec->catfile($serverDir, 'manifest');
    my $serverManifest = LCM::Manifests::ManifestFactory::createComponentManifest($pathToServerManifest);
    return ($serverManifest->getHANAFlavour() eq $gFlavourCockpit);
}

sub _productionShareDevBuild {
    my ($self) = @_;
    my $installerDir = $self->_getInstallerDir();
    my $pathSeparatorPattern = quotemeta($path_separator);
    if ($installerDir !~ m!${pathSeparatorPattern}\Q$INSTALER_HDB_DIRECTORY\E${pathSeparatorPattern}?$!) {
        my @subDirs = map { File::Spec->catfile($installerDir, $_) } ('server', 'client');
        return 0 if (grep(isDirectory($_), @subDirs) != 2);
    }
    my $installerDirParent = MyRealPath(File::Spec->catfile($installerDir, '..'));
    my $productSpecificAFLsDir = File::Spec->catfile($installerDirParent, 'product_specific_afls');
    my $directoryEntries = listDir($productSpecificAFLsDir) || [];
    my $serverDir = File::Spec->catfile($installerDir, $server);
    my $componentDirCandidates = [
        $serverDir,
        File::Spec->catfile($installerDir, $client),
        File::Spec->catfile($installerDir, $studio),
        # File::Spec->catfile($installerDir, $lss, 'packages'),
        File::Spec->catfile($installerDirParent, 'AFL_installer', 'installer', 'packages'),
        File::Spec->catfile($installerDirParent, 'epmmds_installer', 'installer', 'packages'),
        File::Spec->catfile($installerDirParent, 'EML_installer', 'installer', 'packages'),
    ];

    for my $entry (@{$directoryEntries}){
        push(@{$componentDirCandidates}, File::Spec->catfile($productSpecificAFLsDir, $entry, 'installer', 'packages'));
    }
    my @componentDirs = grep { isRegularFile(File::Spec->catfile($_, 'manifest')) } @{$componentDirCandidates};
# Bug: 158457
    if (@componentDirs) {
        return 1 if ($self->_isCockpitFlavour($serverDir));
    }
    return $self->_handleComponents(\@componentDirs);
}

sub _extractedSMPArchives {
    my ($self) = @_;
    my $installerDir = $self->_getInstallerDir();
    my $smpRoot = MyRealPath(File::Spec->catfile($installerDir, '..'));
    if(!$isWin && $smpRoot eq $path_separator){
    	$smpRoot = MyRealPath($installerDir);
    }
    my $directoryEntries = listDir($smpRoot) || [];
    my $componentDirs = $self->hasXsApps($smpRoot) ? [ $smpRoot ] : [];

    my $installationKitServerPath = undef;
    for my $entry (@{$directoryEntries}){
        for my $dName (@componentDirNames) {
            my $candidateDir = File::Spec->catfile($smpRoot, $entry, $dName);
            my $manifestPath = File::Spec->catfile($candidateDir, 'manifest');
            if (! -f $manifestPath) {
                next;
            }
            my $manifest = LCM::Manifests::ManifestFactory::createComponentManifest($manifestPath);
            if (defined $manifest && $manifest->isServer() && $candidateDir eq File::Spec->catfile($installerDir, $dName)) {
                $installationKitServerPath = $candidateDir;
            }
            push(@{$componentDirs}, $candidateDir);
            last;
        }
        my $entryDir = File::Spec->catfile($smpRoot, $entry);
        push(@{$componentDirs}, $entryDir) if ($self->hasXsApps($entryDir));
    }
# Autodetect cockpit stack manifest
    if (-f File::Spec->catfile($smpRoot, 'manifest')) {
        push(@{$componentDirs}, $smpRoot);
    }
# Autodetect server manifest
    if (-f File::Spec->catfile($smpRoot, 'server', 'manifest')) {
        push(@{$componentDirs}, File::Spec->catfile($smpRoot, 'server'));
    }
    if (defined $installationKitServerPath) {
        $componentDirs = $self->_removeServerComponentsNotInKit($componentDirs, $installationKitServerPath);
    }
    return $self->_handleComponents($componentDirs);
}

sub _removeServerComponentsNotInKit {
    my ($self, $componentDirs, $installationKitServerPath) = @_;
    my @result = ();
    for my $componentDir (@$componentDirs) {
        if ($componentDir eq $installationKitServerPath) {
            push(@result, $componentDir);
            next;
        }
        my $manifest = LCM::Manifests::ManifestFactory::createComponentManifest(File::Spec->catfile($componentDir, 'manifest'));
        if (defined $manifest && !$manifest->isServer()) {
            push(@result, $componentDir);
            next;
        }
    }
    return \@result;
}

sub hasXsApps {
    my ($self, $dir) = @_;
    my $directoryEntries = listDir($dir) || [];
    my @zipArchives = map { File::Spec->catfile($dir, $_) } (grep { $_ =~ /\.ZIP$/i } @{$directoryEntries});
    my $hasXsApps = grep { LCM::Manifests::XS2ApplicationManifest->new($_)->isXS2Application() } @zipArchives;
    return $hasXsApps;
}

sub _handleComponents {
    my ($self, $componentDirs) = @_;
    my $configuration = $self->_getConfiguration();
    my $computedDefault = join(',', @$componentDirs);

    return 0 if(!$computedDefault);

    $configuration->setDefault('DefaultLocation', $computedDefault, 1);
    return 1;
}

1;