package LCM::ProcessExecutor;

use base SDB::Install::Base;
use SAPDB::Install::PipeExec;
use SDB::Install::SysVars qw ($path_separator $isWin);
use SDB::Install::HdbInstallerOutputParser qw (parseHdbInstallerErrorMessages);
use strict;

sub new {
    my $self = shift->SUPER::new(); 
    (
        $self->{'executable'},      # path to the program, absolute, or
                                    # relative to the 'cwd', mandatory.
                                    
        $self->{'argTokens'},       # ref to array of cmdl tokens, optional.
        
        $self->{'stdinLines'},      # ref to array of stdin lines (w/o line terminators), optional.
        
        $self->{'cwd'},             # working dir of program, optional.
        
        $self->{'restoreCwd'},      # cwd to restore after execution, optional.
        
        $self->{'uid'},             # uid to change to, optional, only root can do that.
        
        $self->{'gid'},             # gid to change to, optional, only root can do that.
        
        $self->{'havePtyIo'},       # for programs with io over the pty, e.g. pw prompts, optional.

        $self->{'timeout'}         # timeout in seconds, optional.

    ) = @_;
    $self->{'outputLines'} = undef;
    $self->{'switchedUser'} = 0;
    $self->{quoteCmdLineArgumentsFlag} = 1;
    return $self;
}

#----------------------------------------------------------------------------
# 
# Executes external application.
#
sub execExtProgram {
    my ( $self, $componentManager, $message, $errorMessage, $doNotSetLogLocation, $doNotShowProgress ) = @_;
    my $rc = 1;
    my $progressHandler;
    if($doNotShowProgress) {
        $progressHandler = $componentManager->getMsgLst ()->getProgressHandler();
        $componentManager->getMsgLst ()->setProgressHandler(undef);
    }
    my $msg = $componentManager->getMsgLst ()->addProgressMessage ( $message );
    if($doNotShowProgress) {
        $componentManager->getMsgLst ()->setProgressHandler($progressHandler);
    }
    my $saveCntxt = $componentManager->setMsgLstContext ( [ $msg->getSubMsgLst () ] );
    $componentManager->initProgressHandler ();#also sets identation depth+1 
    $progressHandler = $componentManager->getProgressHandler ();

    $self->setOutputHandler ( $progressHandler );
    my $exitCode = $self->executeProgram ();
    $componentManager->getMsgLst ()->addMessage ( undef, $self->getMsgLst () );
    $componentManager->setLogLocation ( $componentManager->parseLogFileLocation ( $self->getOutputLines () ) ) unless $doNotSetLogLocation;
    
    if ( !defined $exitCode || $exitCode ) {
        my $errMsgLst = $componentManager->getHdbInstallerErrorMessages ( $self->getOutputLines () );
        $componentManager->setErrorMessage ( $errorMessage,
            $errMsgLst->isEmpty ? $self->getErrMsgLst () : $errMsgLst );
        $rc = undef;
    }
    $msg->endMessage ( undef, $message );
    if (defined $progressHandler && $progressHandler->can('decrementIndentationDepth'))  {
        $progressHandler->decrementIndentationDepth();
    }
    $componentManager->setMsgLstContext ( $saveCntxt );
    return $rc;
}

sub execExtProgramLogging{
	my ( $self, $logger, $message, $errorMessage, $doNotShowProgress) = @_;
    my $rc = 1;
    my $progressHandler;
    if($doNotShowProgress) {
        $progressHandler = $logger->getMsgLst ()->getProgressHandler();
        $logger->getMsgLst ()->setProgressHandler(undef);
    }
    else{
    	 my $progressHandler = $logger->getProgressHandler();
        if(defined $progressHandler){
            $progressHandler->InitProgress (undef, 0);
        }
        $self->setOutputHandler ( $progressHandler );
    }
    my $msg = $logger->getMsgLst ()->addProgressMessage ( $message );
    my $saveCntxt = $logger->setMsgLstContext ( [ $msg->getSubMsgLst () ] );
    my $exitCode = $self->executeProgram ();
    $logger->getMsgLst ()->addMessage ( undef, $self->getMsgLst () );
    
    if ( !defined $exitCode || $exitCode ) { 
        $logger->setErrorMessage ( $errorMessage,$self->getErrMsgLst () );
        $rc = undef;
    }
    $msg->endMessage ( undef, $message );
    $logger->setMsgLstContext ( $saveCntxt );
    
    if($doNotShowProgress) {
        $logger->getMsgLst ()->setProgressHandler($progressHandler);
    }
    return $rc;
}


#----------------------------------------------------------------------------

=functionComment

Sets if command line arguments should be placed in quotes.
For example quoted arguments are:
    mycmd "arg1" "arg2"

Unquoted arguments are: 
    mycmd arg1 arg2
    
=cut
sub quoteCmdLineArguments
{
    $_[0]->{quoteCmdLineArgumentsFlag} = $_[1];   
}

#----------------------------------------------------------------------------

=functionComment

set the process environment for the child process
first parameter is the environment as a hash reference

=cut
sub setProcessEnvironment {
    $_[0]->{processEnvironment} = $_[1];
}


#----------------------------------------------------------------------------

=functionComment

set callback function
it's called after each read attempt

=cut
sub setCallback{
    $_[0]->{'fCallback'} = $_[1];
}

#----------------------------------------------------------------------------

=functionComment

set function reference to substitute logged cmdln

=cut
sub setLogCmdFunction{
    $_[0]->{'fLogCmdSubst'} = $_[1];
}

#----------------------------------------------------------------------------

=functionComment

set function reference to substitute logged output

=cut
sub setSubstLoggedOutputFunction{
    $_[0]->{'fSubstLoggedOut'} = $_[1];
}


#----------------------------------------------------------------------------

=functionComment

returns the return code of the program (0 f. success, != 0 f. error),
or undef, if execution was impossible.

=cut
sub executeProgram {
    my (
        $self, $isShellExecutable
    ) = @_;
    return $self->_executeProgram($isShellExecutable);
}

#----------------------------------------------------------------------------

=functionComment

returns combined stdout/stderr output of program as ref to array of lines.

=cut
sub getOutputLines {
    my (
        $self
    ) = @_;
    return $self->{'outputLines'};
}


#----------------------------------------------------------------------------

=functionComment

sets an output handler object

=cut
sub setOutputHandler {
    $_[0]->{'outputHandler'} = $_[1];
}

#----------------------------------------------------------------------------

sub wasATimeout{
   return $_[0]->{_was_a_timeout};
}

#----------------------------------------------------------------------------
# only "private" methods below this line.
#----------------------------------------------------------------------------

sub _executeProgram {
    my ( $self, $isShellExecutable) = @_;
    my $prog = $self->{'executable'};

    if(!$isShellExecutable && not -x $prog) {
        if ($!){
            $self->setErrorMessage ("Cannot execute file '$prog': $!");
        }
        else{
            $self->setErrorMessage ("'$prog' is not an executable program.");
        }
        return undef;
    }
    my $nl = $isWin? "\r\n" : "\n";
    if(not $self->_setCwd()) {
        return undef;
    }
    if(not $self->_switchToUser()) {
        return undef;
    }
    
    my $pipe = SAPDB::Install::PipeExec->new();
    if (defined $self->{timeout}){
        $pipe->Timeout($self->{timeout});
    }
    my $cmd_line = $prog =~ /\s/ ? "\"$prog\"" : $prog;
    if(defined $self->{'argTokens'}) {
        foreach my $arg (@{$self->{'argTokens'}}) {
            if(!defined $arg) {
                next;
            }
            $arg =~ s/"/\\"/g;
            if($arg eq '') {
                $arg = $self->{quoteCmdLineArgumentsFlag} ? '""' : '';
            }
            elsif($arg =~ /\s/) {
                $arg = $self->{quoteCmdLineArgumentsFlag} ? "\"$arg\"" : $arg;
            }
            $cmd_line .= ' '.$arg;
        }
    }
    my $msg = $self->getMsgLst()->addMessage('Starting external program ' . $prog);
    my $log_cmd_line = $cmd_line;
    if (defined $self->{'fLogCmdSubst'}){
        $log_cmd_line = $self->{'fLogCmdSubst'}->($cmd_line);
    }
    $msg->getSubMsgLst()->addMessage('Command line is: ' . $log_cmd_line);
    if (defined $self->{timeout}){
        $msg->getSubMsgLst()->addMessage('Execution timeout: ' . $self->{timeout} . ' s');
    }

    {
        local %ENV = defined $self->{processEnvironment} ? %{$self->{processEnvironment}} : %ENV;

        if ($self->{switchedUser}){
            require SDB::Install::User;
            my $user = new SDB::Install::User ();
            my $home = $user->home ();
            my $name = $user->getname ();
            if (defined $home){
                $ENV{HOME} = $home;
            }
            if (defined $name){
                $ENV{USER} = $name;
            }
        }

        if($self->{'havePtyIo'}) {
            $pipe->OpenPty($cmd_line);
        } else{
            $pipe->Open($cmd_line);
        }
    }
    
    if($pipe->GetError) {
        $self->setErrorMessage ('Cannot execute program '.$prog.': ' . $pipe->GetError);
        $self->_switchBack();
        $self->_restoreCwd();
        return undef;
    }

    if (defined $self->{'fCallback'}){
        $pipe->SetCallback($self->{'fCallback'});
    }
    elsif(%{Wx::} && exists ${Wx::}{wxTheApp}) {
        $pipe->SetCallback(sub {Wx::Yield()});
    }
    
    if($self->{'stdinLines'}) {
        if($self->{'havePtyIo'}) {
            sleep(2);
        }
        my $len;
        foreach my $input_string (@{$self->{'stdinLines'}}) {
            $len = $pipe->Write($input_string . $nl);
            if(!defined $len || $len < 0) {
                $self->setErrorMessage ('Cannot write to stdin pipe of program '.$prog.': ' . $pipe->GetError);
                $self->_switchBack();
                $self->_restoreCwd();
                return undef;
            }
            if (defined $self->{'delayOnSend'}) {
            	sleep($self->{'delayOnSend'});	
            }
        }
        if(!$self->{'havePtyIo'}) {
            if(!defined $pipe->CloseStdin ()) {
                $self->getMsgLst()->addWarning('Cannot close stdin pipe of program '.$prog.': ' . $pipe->GetError);
            }
        }
    }
    my @buffer;
    my $line;
    my $i = 1;
    my $templ = 'Output line %d: %s';
    my $fSubstLoggedOut = $self->{'fSubstLoggedOut'};
    while(my $line = $pipe->Readline()) {
        if($isWin) {
            local $/ = "\r\n";
            chomp $line;
        }
        chomp $line;
        if (defined $self->{outputHandler}){
            $self->{outputHandler}->addLine ($line);
        }
        $msg->getSubMsgLst()->addMessage(sprintf ($templ, $i++,
            defined $fSubstLoggedOut ? $fSubstLoggedOut->($line) : $line));
        push @buffer,$line;
    }
    $self->{'outputLines'} = \@buffer;

    if ($pipe->WasaTimeout()){
        $self->{_was_a_timeout} = 1;
        $msg->getSubMsgLst()->addMessage("Timeout ($self->{timeout} s) occurred.");
    }

    if($pipe->GetError) {
        $self->setErrorMessage ('Error executing program '.$prog.': ' . $pipe->GetError);
        $self->_switchBack();
        $self->_restoreCwd();
        return undef;
    }
    my $rc = $pipe->Close();
    if(defined $rc || $rc) {
        $msg->getSubMsgLst()->addMessage('Program terminated with exit code ' . $rc);
    } else {
        $self->setErrorMessage ('Program '.$prog.' terminated with error: ' . $pipe->GetError);
        $self->_switchBack();
        $self->_restoreCwd();
        return undef;   
    }
    if(not $self->_switchBack()) {
        return undef;
    }
    if(not $self->_restoreCwd()) {
        return undef;
    }

    return $rc;
}

#----------------------------------------------------------------------------

sub _setCwd {
    my (
        $self
    ) = @_;
    if(defined $self->{'cwd'}) {
        my $rc = chdir($self->{'cwd'});
        if(!$rc) {
            $self->getErrMsgLst()->addError("could not chdir to $self->{'cwd'}.");
            return undef;
        }
    }
    return 1;
}

#----------------------------------------------------------------------------

sub _restoreCwd {
    my (
        $self
    ) = @_;
    if(defined $self->{'restoreCwd'}) {
        my $rc = chdir($self->{'restoreCwd'});
        if(!$rc) {
            $self->getErrMsgLst()->addError("could not chdir to $self->{'restoreCwd'}.");
            return undef;
        }
    }
    return 1;
}

#----------------------------------------------------------------------------

sub _switchToUser {
    my (
        $self
    ) = @_;
    if((not $isWin) && (defined $self->{'uid'})) {
        my $gidChange = (defined $self->{'gid'}) ? 1 : 0;
        if($> != $self->{'uid'} || $gidChange ) {
            my $gidMsgStr = $gidChange ? " and group id ".$self->{'gid'}."." : ".";
            my $msg = $self->getMsgLst()->addMessage ("Switching to user id ".$self->{'uid'}.$gidMsgStr);
            require SAPDB::Install::SetUser;
            my ($rc, $errtext) = SAPDB::Install::SetUser::SetUser ($self->{'uid'}, $self->{'gid'});
            if( $rc != 0) {
                $self->getErrMsgLst()->addError ("SetUser() failed: $errtext");
                return undef;
            }
            # why does this not work?
            #$msg->getSubMsgLst()->addMessage("uid = $<, gid = $(");
            #$msg->getSubMsgLst()->addMessage("euid = $>, egid = $)");
            $self->{'switchedUser'} = 1;
        }
    }
    return 1;
}

#----------------------------------------------------------------------------

sub _switchBack {
    my (
        $self
    ) = @_;
    if(not $isWin) {
        if($self->{'switchedUser'}) {
            my $msg = $self->getMsgLst()->addMessage ("Switching back to root user.");
            require SAPDB::Install::SetUser;
            my ($rc, $errtext) = SAPDB::Install::SetUser::SetUser ();
            if($rc) {
                $self->getErrMsgLst()->addError ("SetUser() failed: $errtext");
                return undef;
            }
            # why does this not work?
            #$msg->getSubMsgLst()->addMessage("uid = $<, gid = $(");
            #$msg->getSubMsgLst()->addMessage("euid = $>, egid = $)");
            $self->{'switchedUser'} = 0;
        }
    }
    return 1;
}

#sets delay in seconds when data are send to the called process.
#it is required in SLDregistration scenario to workaround a bug with parsing of provided
#arguments in sldreg ver 8.0
sub setDelayOnWrite() {
	my ($self, $delay) = @_;
	$self->{'delayOnSend'} = $delay;
}

sub genHdbErrMsgLstFromOutput{
    my ($self) = @_;
    return parseHdbInstallerErrorMessages ($self->{outputLines});
}

#----------------------------------------------------------------------------

1;
