package LCM::SHA::DeploymentStrategy::DeployConfiguration;

use strict;
use base qw(LCM::SHA::DeploymentStrategy);
use SDB::Install::Installer;
use SDB::Install::Globals qw($gSignatureManifestName determineSignedManifestPath);
use File::Basename;
use File::Spec;
use LCM::FileUtils;
use LCM::Installer;

sub initialize {
	my ($self) = @_;
	my $hdblcmDir = $self->_getHdblcmDir();
	my $operationsDir = File::Spec->catdir($hdblcmDir, 'operations.d');
	my $signatureFilePath = $self->_determinePathToSignatureFile($hdblcmDir);
	my $signature = $self->_getSignature($signatureFilePath);
	if(!$signature){
		$self->appendErrorMessage("Failed to load signature file '$signatureFilePath'");
		return undef;
	}
	my $isCockpitInMasterSignature = $self->isCockpitInMasterSignature ($signature->getContent ());

	my ($configurations, $cockpitConfigurations) =
	$self->_getConfigurations($operationsDir, !$isCockpitInMasterSignature ? 'hdbcockpit_\S+\.conf$' : ());
	if(! defined($configurations)){
		$self->appendErrorMessage('Failed to load SAP Host Agent configurations from filesystem');
		return undef;
	}
	unshift(@{$configurations}, $signature);
	$self->{_configurations} = $configurations;
	if (defined $cockpitConfigurations && @$cockpitConfigurations){
		my $cockpitSignatureFile = $self->getCockpitSignatureFile ($signatureFilePath);
		if (defined $cockpitSignatureFile){
			my $cockpitSignature = $self->_getSignature($cockpitSignatureFile);
			$self->{_cockpitConfigurations} = [$cockpitSignature, @$cockpitConfigurations];
		}
		else{
			$self->getMsgLst()->addWarning ("HANA Offline Cockpit deployment is skipped.");
		}
	}
	return 1;
}

sub _deployOnHost {
	my ($self, $hostctrl) = @_;
	return $hostctrl->DeployConfiguration($self->{_configurations});
}

sub _deployCockpitOnHost {
    my ($self, $hostctrl) = @_;
    return $hostctrl->DeployConfiguration($self->{_cockpitConfigurations});
}


sub _getHdblcmDir {
	my ($self) = @_;
	if(!exists($self->{_hdblcmDir})){
		$self->{_hdblcmDir} = $self->_determinePathToOperations();
	}
	return $self->{_hdblcmDir};
}

sub _determinePathToOperations {
	my ($self) = @_;
	my $lcmIinstaller = new LCM::Installer();
	my $sdbIinstaller = new SDB::Install::Installer();
	my $runtimeDir = $sdbIinstaller->GetRuntimeDir();
	if($lcmIinstaller->isInstalled()){
# Runtime dir is /<target>/<SID>/hdblcm/instruntime
		return dirname($runtimeDir);
	} elsif($sdbIinstaller->isServerInstaller() && !$sdbIinstaller->isServerHdblcmInstaller()){
# Runtime dir is /<target>/<SID>/global/hdb/install/bin/instruntime
		return dirname(dirname(dirname($runtimeDir)));
	}
# Runtime dir is /<installation_kit_paht>/instruntime/
	return dirname($runtimeDir);
}

# Consider whether the SIGNATURE.SMF file is located in the installer's dir or it's parent dir
sub _determinePathToSignatureFile {
	my ($self, $hdblcmDir) = @_;
	return determineSignedManifestPath($hdblcmDir);
}

sub _getSignature {
	my ($self, $signatureFilePath) = @_;
	my $fileContent = LCM::FileUtils::readFile($signatureFilePath, $self->getErrMsgLst());

	if(defined $fileContent){
		return new LCM::SHA::DeploymentStrategy::Configuration('manifest', $gSignatureManifestName, $fileContent);
	}
	return undef;
}

sub _getConfigurations {
	my ($self, $operationsDir, $cockpit_regex) = @_;

	my $fileList = LCM::FileUtils::listDirectory($operationsDir, $self->getErrMsgLst());
	return undef if(!defined($fileList));

	my @configurationsList = ();
	my @cockpitConfigurationsList = ();

	for my $fileName (@{$fileList}){
		my $filePath = File::Spec->catfile($operationsDir, $fileName);

		next unless(-f $filePath && $fileName =~ /^.*\.conf$/); # skip non '.conf' files

		my $fileContent = LCM::FileUtils::readFile($filePath, $self->getErrMsgLst());

		return undef if(!defined $fileContent);

		my $configuration = new LCM::SHA::DeploymentStrategy::Configuration('operation', $fileName, $fileContent);

		if (defined $cockpit_regex && ($fileName =~ /$cockpit_regex/)){
			push(@cockpitConfigurationsList, $configuration);
		}
		else{
			push(@configurationsList, $configuration);
		}
	}
	return \@configurationsList,\@cockpitConfigurationsList;
}

sub _shallDeployCockpit{
    return defined $_[0]->{_cockpitConfigurations};
}

1;
package LCM::SHA::DeploymentStrategy::Configuration;

use strict;

sub new {
	my ($class, $targetType, $name, $content) = @_;
	
	$content = defined($content) ? $content : [];

	my $self = {
		_name => $name,
		_content => $content,
		_targetType => $targetType,
	};
	
	return bless($self, $class);
}

sub setContent { $_[0]->{_content} = $_[1]; }

sub getContent { return $_[0]->{_content}; }

sub setName { $_[0]->{_name} = $_[1]; }

sub getName { return $_[0]->{_name}; }

sub setTargetType { $_[0]->{_targetType} = $_[1]; }

sub getTargetType { return $_[0]->{_targetType}; }

1;