package LCM::SapHostAgentFunctions;

use strict;
use base qw(SDB::Install::Base);
use SDB::Install::Saphostagent qw( getSHAVersion getActiveSaphostexecDir getSaphostexecPath);
use SDB::Install::SysVars qw($path_separator $isWin);
use SDB::Install::System qw($plain_hostname_regex);
use LCM::ProcessExecutor;
use SDB::Install::Version;
use SDB::Install::Globals qw($gProductNameSHA);

sub new {
	my ($class, $component, $logger, $doNotSetLogLocation, $doNotShowProgress) = @_;
	my $self = {};
	bless( $self, $class );
	if(defined $component){
		$self->{_component} = $component;
	}else{
		$self->{logger} = $logger;
	}
    $self->{doNotSetLogLocation} = $doNotSetLogLocation // 0;
    $self->{doNotShowProgress} = $doNotShowProgress // 0;
    
	return $self;
}

sub isSapHostAgentAvailable
{
    my $self = shift();
	return undef if (! $self->isHostagentInstalled());

	my $cacheLogLocation = $self->getDoNotSetLogLocation();
	my $cacheShowProgress = $self->getDoNotShowProgress();
	$self->setDoNotSetLogLocation(1);
	$self->setDoNotShowProgress(1);
	my $rc = $self->_execSapHostAgentCommand(['-status'], 'Checking for running SAP Host Agent...', 'SAP Host Agent is not running or error occurred while checking.', 'saphostexec');
	$self->setDoNotSetLogLocation($cacheLogLocation);
    $self->setDoNotShowProgress($cacheShowProgress);
	return $rc;
}

#----------------------------------------------------------------------------
#
# Registers SAP Instance service.
#
# $_[0] - self
# $_[1] - SID on which the instance should run
# $_[2] - Instance number
# $_[3] - hostname on which the instance should run
#
sub registerInstanceService {
	my ( $self, $sid, $nr, $hostname ) = @_;

	my @args;
	push( @args, '-function RegisterInstanceService' );
	push( @args, '-sid ' . $sid );
	push( @args, '-nr ' . $nr );
	push( @args, '-saplocalhost ' . $hostname );

	# TODO: check if windows options are needed
	return $self->_execSapHostAgentCommand(\@args, "Registering Instance Service...", "Registration of the Instance Service failed");
}

sub restartSapHostAgent {
    my ( $self, $doNotSetLogLocation, $doNotShowProgress ) = @_;

    my @args;
    push( @args, '-restart' );
    $self->setDoNotSetLogLocation($doNotSetLogLocation);
    $self->setDoNotShowProgress($doNotShowProgress);
    if ($self->_execSapHostAgentCommand(\@args, "Restarting SAP Host Agent...", "Restart of SAP Host Agent failed.", 'saphostexec' )) {
        return $self->_waitForAvailableSHA();
    }
    return 0;
}

sub _execSapHostAgentCommand
{
    my ($self, $argsArray, $startMessage, $failureMessage, $executable ) = @_;
    my $hostAgentExeDir = getActiveSaphostexecDir ();
    my $exe = (defined $executable) ? $executable : 'saphostctrl';
    my $ext = $isWin ? '.exe' : '';
    my $shaExe = $hostAgentExeDir . $path_separator . $exe . $ext;
    my $doNotSetLogLocation = $self->{doNotSetLogLocation};
    my $doNotShowProgress =  $self->{doNotShowProgress}; 
    # using LCM::ProcessExecutor instead of SDB::Install::Saphostagent for consistency
    my $exer = new LCM::ProcessExecutor ( $shaExe, $argsArray );
        $exer->quoteCmdLineArguments ( 0 );    # do not quote arguments as SHA cannot handle them
        my $rc;
    if (defined $self->{logger}) {
    	$rc = $exer->execExtProgramLogging($self->{logger}, $startMessage,
        $failureMessage, $doNotShowProgress );
    } elsif (defined $self->{_component}) {
    	$rc = $exer->execExtProgram ( $self->{_component}, $startMessage,
        $failureMessage, $doNotSetLogLocation, $doNotShowProgress );
    } else {
        $exer->setMsgLstContext($self->getMsgLstContext());
        $rc = ($exer->executeProgram() == 0) ? 1 : 0;
    }


    return undef unless $rc;
    return wantarray() ? ($rc, $exer) : $rc;
}

sub _waitForAvailableSHA {
    my ($self) = @_;
    $self->getMsgLst()->addMessage("Waiting for $gProductNameSHA to start...");
    for (1..20) {
        sleep(3);
        if ($self->isSapHostAgentAvailable()) {
            $self->getMsgLst()->addMessage("$gProductNameSHA started successfully.");
            return 1;
        }
    }
    return 0;
}

sub isHostagentInstalled {
	my ($self) = @_;
	my $saphostexecPath = getSaphostexecPath();
	if (!defined $saphostexecPath) {
		my $hostAgentExeDir = getActiveSaphostexecDir();
		$self->setErrorMessage ("Cannot find installed SAP Host Agent at $hostAgentExeDir");
		return undef;
	}
	return 1;
}

sub setDoNotSetLogLocation{
	$_[0]->{doNotSetLogLocation} = $_[1];
}

sub getDoNotSetLogLocation{
	return $_[0]->{doNotSetLogLocation};
}

sub setDoNotShowProgress{
	$_[0]->{doNotShowProgress} = $_[1];
}

sub getDoNotShowProgress{
	return $_[0]->{doNotShowProgress};
}

sub getFQDN {
	my ($self) = @_;
	if (!$self->isHostagentInstalled()) {
		$self->getMsgLst()->addMessage("Cannot get FQDN because $gProductNameSHA is either not installed");
		return undef;
	}
	my @args = qw(-function GetCIMObject -enuminstances SAP_ITSAMComputerSystem);
	my $startMsg = "Getting host FQDN from CIM Object SAP_ITSAMComputerSystem...";
	my $failMsg = "Failed to get FQDN name through $gProductNameSHA";

	my ($rc, $exer) = $self->_execSapHostAgentCommand(\@args, $startMsg, $failMsg);
	if (!$rc) {
		return undef;
	}
	my $fqdn = undef;
	my $output = $exer->getOutputLines();
	for my $line (@$output) {
		next if ($line !~ /^\W*FQDName/);
		local $/ = ' ';
		chomp($fqdn = (split(' , ', $line))[-1]);
	}
	return $fqdn;
}

1;