package LCM::Slpp::ParameterAdapter;

use strict;

use constant STATUS_OK     	=> "slp.parameter.validation.status.OK";
use constant STATUS_WARNING => "slp.parameter.validation.status.WARNING";
use constant STATUS_ERROR   => "slp.parameter.validation.status.ERROR";

sub new {
	my ($class, $paramId, $hdblcmRepresentation) = @_;
	my $self = bless({}, $class);
	
	$self->{parameterID} = $paramId;
	$self->{hdblcmRepresentation} = $hdblcmRepresentation;
	
	return $self;
}

sub getValidationStatus {
	my ($self) = @_;
	my $validationStatus = $self->{validationStatus} ? [ @{$self->{validationStatus}} ] : [];
	if ($self->hasParameterWarnings()) {
# Each time we want to return this parameter update the warnings because in the time between the setting of
# the value of the parameter and the time of returning it via the REST API, some new warnings may have appearead
		my $parameterWarnings = $self->getParameterWarnings();
		push(@{$validationStatus}, { ValidationStatus => { status => STATUS_WARNING, message => $_ } }) for @{$parameterWarnings};
	}
	return $validationStatus;
}

sub setValidationStatus {
	my($self, $returnCode, $message) = @_;
	my $hasParameterWarnings = $self->hasParameterWarnings();
	my $status = $returnCode ? STATUS_OK : STATUS_ERROR;
	$self->{validationStatus} = [ { ValidationStatus => { status => $status, message => $message } } ];
	$self->_setIsSet(1) if($returnCode); # Mark that this parameter has been set with a user input
}

sub getSlppType {
	my ($self) = @_;
	# no valid values can be presented for this parameter so the WebUI will treat it as scalar
	if($self->{parameterID} eq 'hdbinst_studio_SelectedFeatures') {
		return 'slp.parameter.type.SCALAR';
	}

	my $type = $self->{hdblcmRepresentation}->{type};
	my $hasValidValues = $self->{hdblcmRepresentation}->{valid_values};

	if($type =~ /map|csv/ || $hasValidValues) {
		return 'slp.parameter.type.TABLE';
	}

	return 'slp.parameter.type.SCALAR';
}

sub getHdblcmTag {
	my ($self) = @_;
	my $parameter = $self->{hdblcmRepresentation};
	return {
		label => $parameter->{opt},
		parameter_string => $parameter->{ui_str} // $parameter->{str},
		tooltip => $parameter->{desc} ? $parameter->{desc} : $parameter->{str},
		recommendation => $parameter->{recommendation},
		type => $parameter->{type},
		valid_values => $parameter->{valid_values} ? join(",", @{$parameter->{valid_values}}) : "",
		section => $parameter->{section},
	};
}

sub getDefaultSlppValue {
	#	OVERRIDE IN SUBCLASS
}

sub getRequired {
	my ($self) = @_;
	my $hdblcmParameter = $self->{hdblcmRepresentation};
	if($hdblcmParameter->{mandatory} && ! ($hdblcmParameter->{skip}) ){
		return 'true'
	}
	return 'false';
}

sub getSecure {
	return 'false';
}

sub getSlppValue {
	#	OVERRIDE IN SUBCLASS
}

sub getSlppBatchValue {
	#	OVERRIDE IN SUBCLASS
}

sub getSlppValueTag {
	#	OVERRIDE IN SUBCLASS
}

sub getSlppBatchValueTag {
	#	OVERRIDE IN SUBCLASS
}

sub hasStructure {
	return 0;
}

sub getStructure {
	return undef;
}

sub hasDefault {
	return 0;
}

sub hasDefaultTable {
	return 0;
}

sub hasBatch {
	my ($self) = @_;
	my $hdblcmParameter = $self->{hdblcmRepresentation};
	return defined($hdblcmParameter->{batchValue});
}

sub getSlppParameter {
	my ($self) = @_;

	my $slppValueTag = $self->getSlppValueTag();
	my $slppBatchValueTag = $self->getSlppBatchValueTag();

	return {
		Parameter => {
			id 					=> $self->{parameterID},
			type				=> $self->getSlppType(),
			hdblcm				=> $self->getHdblcmTag(),
			(
				$self->hasDefault() ? (default => $self->getDefaultSlppValue()) : ()
			),
			(
				$self->hasDefaultTable() ? (defaulttablevalue => $self->getDefaultSlppTableValue()) : ()
			),
			required 			=> $self->getRequired(),
			secure 				=> $self->getSecure(),
			$slppValueTag 		=> $self->getSlppValue(),
			(
				$self->hasStructure() ? (structure => $self->getStructure()) : ()
			),
			validationResult	=> $self->getValidationStatus(),
			(
				$self->hasBatch() ? ($slppBatchValueTag => $self->getSlppBatchValue()) : ()
			)
		}
	};
}

sub _getSlppBoolean {
	my ($self, $value) = @_;
	my $bool_false_pattern = '^off$|^false$|^0$|^-$|^no$|^n$';
	my $bool_true_pattern =  '^on$|^true$|^1$|^\+$|^x$|^yes$|^y$';
	if($value =~ /$bool_true_pattern/i){
		return 'true';
	}elsif($value =~ /$bool_false_pattern/i){
		return 'false';
	} 
	return '';
}

sub hasParameterWarnings {
	my ($self) = @_;
	my $hdblcmParameter = $self->{hdblcmRepresentation};
	my $warnings = $hdblcmParameter->{warnings};
	return (defined($warnings) && ref($warnings) eq 'ARRAY' && scalar(@{$warnings}) > 0);
}

sub getParameterWarnings {
	my ($self) = @_;
	
	return [] unless($self->hasParameterWarnings());
	
	my $hdblcmParameter = $self->{hdblcmRepresentation};
	my $result = [];
	for my $warning (@{$hdblcmParameter->{warnings}}){
		push(@{$result}, $warning);
	}
	return $result;
}

# Following two subs are a workaround for some web UI issues concerning default values of parameters with defaultTable
# If a parameter has ever been successfully set with a user input value, we shall not display the defaultTable property anymore 
sub _setIsSet {
	my ($self, $isSet) = @_;
	$self->{_isSet} = $isSet;
}

sub _isSet {
	my ($self) = @_;
	return $self->{_isSet};
}

1;