package LCM::Slpp::ScenarioExecutor;

use strict;
use base qw(SDB::Install::Base);

use threads;
use threads::shared;

use Time::HiRes qw(time);
use LCM::Slpp::SlppProgressHandler;
use LCM::Slpp::HdblcmModelFactory qw(createExecutor);
use LCM::Slpp::SlppSubtaskListener;
use LCM::Slpp::SlppProcessTaskListener;
use LCM::Slpp::ExecutionState qw(STATE_INITIAL STATE_RUNNING STATE_FINISHED STATE_ERROR STATE_DIALOG STATE_ABORTED);
use LCM::Slpp::ErrorState;

# All the shared structures we're going to need.
# They must be accessed and modified only with the getters and setters, which this class provides
my $taskInformation     : shared; # id => {hash} pairs, no need for deep copy, only the warnings property as it is array ref
my $progress            : shared; # simple scalar, no need for deep copy
my $progressMessages    : shared; # array of scalars, no need for deep copy
my $state               : shared; # scalar, no need for deep copy
my $errorMessage        : shared; # scalar, again no need for deep copy
my $executionStepIds    : shared; # array of scalars, no need for deep copy
my $finishedAt          : shared; # scalar, no need for deep copy

# Output delimters
my $startDelimiter = "------------- START -------------";
my $endDelimiter   = "------------- END -------------";

sub new {
    my ($class, $executionHandler, $adapter) = @_;
    my $self = bless({
        executionHandler    => $executionHandler,
        slppAdapter         => $adapter,
        isJSONLoaded        => 0,
    }, $class);

    $self->initSharedData();
    $self->loadJSONModule();
    return $self;
}

sub initSharedData {
    my ($self) = @_;
    $taskInformation    = shared_clone({});
    $progress           = 0;
    $progressMessages   = shared_clone([]);
    $state              = undef;
    $errorMessage       = "";
    $executionStepIds   = shared_clone([]);
    $finishedAt         = undef;
}

sub isJSONLoaded {
    my ($self) = @_;
    return $self->{isJSONLoaded};
}

sub loadJSONModule {
    my ($self) = @_;
    if (!$self->isJSONLoaded()) {
        local %ENV = %ENV;
        $ENV{PERL_JSON_BACKEND} = 'JSON::backportPP';
        eval("require JSON;");
        if ($@) {
            $self->setState(STATE_ERROR, "Failed to load JSON module");
            $self->getExecHandler()->stopTool();
            return;
        }
        $self->{isJSONLoaded} = 1;
    }
}

sub getExecHandler {
    my ($self) = @_;
    return $self->{executionHandler};
}

sub getAdapter {
    my ($self) = @_;
    return $self->{slppAdapter};
}

sub startExecution {
    my ($self) = @_;
    my $pid = $self->{pid} = $self->_createWorkerProcess();
    if (!defined $pid) {
        return undef;
    }
    return 1;
}

# ####################################################################################################################################
# ---- Methods, relevant for the process management ----------------------------------------------------------------------------------
# ####################################################################################################################################

sub _createWorkerProcess {
    my ($self) = @_;
    pipe(my $readOut, my $writeOut);

    my $pid = fork();
    if (!defined($pid)) {
        return undef;
    }

    if ($pid == 0) {
        # Close the read end, as we won't be needing it here
        close($readOut);
        select($writeOut);
        # Autoflush on
        $| = 1;

        $self->{isWorker} = 1;
        $self->_registerWorkerHandlers();
        $self->_executeScenario();
        $self->getExecHandler()->writeLogs();

        sleep(1); # Avoids SIGPIPE when scenario exits really fast
        close ($writeOut);
        exit(0);
    }
    # We don't need the write end in the parent process
    close($writeOut);

    $self->{readOut} = $readOut;
    $self->{isWorker} = 0;
    $self->_startMonitor();

    return $pid;
}

sub isWorker {
    my ($self) = @_;
    return $self->{isWorker} ? 1 : 0;
}

sub getWorkerPid {
    my ($self) = @_;
    return $self->{pid};
}

sub hasExecutionStarted {
    my ($self) = @_;
    return defined($self->getWorkerPid()) ? 1 : 0;
}

sub isWorkerAlive {
    my ($self) = @_;
    if ($self->isWorker()) {
        return 1;
    }
# kill with SIGZERO just checks whether a signal can be sent to the process
# It doesn't send any actual signals
    return $self->hasExecutionStarted() ? kill(0, $self->getWorkerPid()) : 0;
}

sub killWorker {
    my ($self) = @_;
    if ($self->isWorker() || !$self->getWorkerPid()) {
        return undef;
    }
    kill 'KILL', $self->getWorkerPid();
}

sub _registerWorkerHandlers {
    my ($self) = @_;
    my $configuration = $self->getAdapter()->getConfiguration();
    my @signals = qw (INT PIPE ABRT EMT FPE QUIT SYS TRAP TERM KILL);
    return if !$self->isWorker();

    for my $signal (@signals) {
        $SIG{$signal} = sub {
# Add proper message and save the logs if someone or something
# kills the worker process for any reason
            my $message = "Signal '__${signal}__' caught";
            $self->sendState(STATE_ERROR, $message);
            $configuration->getMsgLst()->addError($message);
            $self->getExecHandler()->writeLogs();
            exit(0);
        };
    }
}

# ####################################################################################################################################
# ---- Thread safe getters - can be invoked from the main thread and from the monitor ------------------------------------------------
# ####################################################################################################################################

sub getTasksInfo {
    my ($self) = @_;
    my $taskInformationCopy;
    my $taskIds = $self->getExecutionStepIds();
    for my $taskId (keys %{$taskInformation}) {
        my $taskInfo = $self->getTaskInfo($taskId);
        if ($taskInfo) {
            $taskInformationCopy->{$taskId} = $taskInfo;
        }
    }
    return $taskInformationCopy;
}

sub getTaskInfo {
    my ($self, $id) = @_;
    my $taskInfoCopy;
    {
        lock $taskInformation;
        my $currentTaskInfo = $taskInformation->{$id};
        return undef if !$currentTaskInfo;
        # We need special handling for the warnings - copy the warnings
        my $warnings = [ @{$currentTaskInfo->{warnings}} ];
        # Copy everything else
        $taskInfoCopy = { %{$currentTaskInfo} };
        # Set the copied warnings to the result
        $taskInfoCopy->{warnnings} = $warnings;
    }
    return $taskInfoCopy;
}

sub getProgress {
    my ($self) = @_;
    my $progressCopy;
    {
        lock $progress;
        $progressCopy = $progress;
    }
    return $progressCopy;
}

sub getState {
    my ($self) = @_;
    my $stateCopy;
    {
        lock $state;
        $stateCopy = $state;
    }
    return $stateCopy;
}

sub getProgressMessages {
    my ($self) = @_;
    my $progressMessagesCopy;
    {
        lock $progressMessages;
        $progressMessagesCopy = [ @{$progressMessages} ];
    }
    return $progressMessagesCopy;
}

sub getExecutionStepIds {
    my ($self) = @_;
    my $executionStepIdsCopy;
    {
        lock $executionStepIds;
        $executionStepIdsCopy = [ @{$executionStepIds} ];
    }
    return $executionStepIdsCopy;
}

sub getFinishedAt {
    my ($self) = @_;
    my $finishedAtCopy;
    {
        lock $finishedAt;
        $finishedAtCopy = $finishedAt;
    }
    return $finishedAtCopy;
}

sub getErrorMessage {
    my ($self) = @_;
    my $errorMsgCopy;
    {
        lock $errorMessage;
        $errorMsgCopy = $errorMessage;
    }
    return $errorMsgCopy;
}

sub constructErrorStateObject {
    my ($self) = @_;
    my $errorId = $self->getAdapter()->getScenario()."_error";
    my $message = $self->getErrorMessage();
    return LCM::Slpp::ErrorState->new($errorId, 1, "Execution of hdblcmweb failed", $message);
}

# ####################################################################################################################################
# ---- Thread safe setters - can be invoked from the main thread and from the monitor ------------------------------------------------
# ####################################################################################################################################

sub setTaskInfo {
    my ($self, $taskId, $info) = @_;
    lock $taskInformation;
    $taskInformation->{$taskId} = shared_clone($info);
}

sub setProgress {
    my ($self, $newProgress) = @_;
    lock $progress;
    $progress = shared_clone($newProgress);
}

sub addProgressMessage {
    my ($self, $msg, $prepend) = @_;
    $prepend //= 0;
    lock $progressMessages;
    if ($prepend) {
        $progressMessages->[0] = shared_clone($msg);
    } else {
        push(@{$progressMessages}, shared_clone($msg));
    }
}

sub setFinishedAt {
    my ($self) = @_;
    lock $finishedAt;
    $finishedAt = int(time());
}

sub setState {
    my ($self, $newState, $msg) = @_;
    lock $state;
    lock $errorMessage;
    $state = shared_clone($newState);
    $errorMessage = shared_clone($msg);
    if($state eq STATE_ERROR || $state eq STATE_FINISHED || $state eq STATE_ABORTED) {
        $self->setFinishedAt();
    }
}

sub addStepId {
    my ($self, $id) = @_;
    lock $executionStepIds;
    push(@{$executionStepIds}, shared_clone($id));
}

# ####################################################################################################################################
# ---- Methods, relevant for the monitoring thread -----------------------------------------------------------------------------------
# ####################################################################################################################################

sub _handleJSONObject {
    my ($self, $object) = @_;
    my $id = $object->{structId};
    my $content = $object->{content};
    if ($id eq 'taskInfo') {
        $self->setTaskInfo($content->{id}, $content->{info});
    } elsif ($id eq 'progress') {
        $self->setProgress($content->{progress});
    } elsif ($id eq 'progressMsg') {
        $self->addProgressMessage($content->{message}, $content->{prepend});
    } elsif ($id eq 'state') {
        $self->setState($content->{state}, $content->{message});
    } elsif ($id eq 'stepId') {
        $self->addStepId($content->{id});
    }
}

sub _startMonitor {
    my ($self) = @_;
    local %SIG = %SIG;
    $SIG{USR1} = sub {
        $self->getExecHandler()->stopTool();
    };
    $self->{mainThread} = threads->self();

    async {
        my $readFh = $self->_getReader();
        my $jsonString = "";
        while (my $line = <$readFh>) {
            chomp($line);
            if ($line eq $startDelimiter) {
                next;
            } elsif ($line eq $endDelimiter) {
                eval {
                    $self->_handleJSONObject(JSON::from_json($jsonString));
                };
                $jsonString = "";
            } else {
                $jsonString .= $line;
            }
        }
        # After we're done reading the pipe, we can safely wait for the process to finish
        waitpid $self->getWorkerPid(), 0;
        $self->{mainThread}->kill('USR1');
    }->detach();
}

sub _getReader {
    my ($self) = @_;
    return $self->isWorker() ? undef : $self->{readOut};
}

# ####################################################################################################################################
# ---- Methods which must be called only from the worker process ---------------------------------------------------------------------
# ####################################################################################################################################
# ---- The update* methods just update the local cache of the worker process ---------------------------------------------------------
# ####################################################################################################################################

sub updateTaskInfo {
    my ($self, $id, $newInfo) = @_;
    $self->{taskInformation}->{$id} = $newInfo;
}

sub updateProgress {
    my ($self, $progress) = @_;
    $self->{progress} = $progress
}

sub updateProgressMessages {
    my ($self, $msg, $prepend) = @_;
    if (!defined $self->{progressMessages}) {
        $self->{progressMessages} = [];
    }

    if ($prepend) {
        $self->{progressMessages}->[0] = $msg;
    } else {
        push(@{$self->{progressMessages}}, $msg);
    }
}

sub updateState {
    my ($self, $state) = @_;
    $self->{state} = $state;
}

sub updateExecutionStepIds {
    my ($self, $id) = @_;
    if (!defined $self->{executionStepIds}) {
        $self->{executionStepIds} = [ $id ];
    } else {
        push(@{$self->{executionStepIds}}, $id);
    }
}

# ####################################################################################################################################
# ---- The send* methods send serialized data to the main process --------------------------------------------------------------------
# ####################################################################################################################################

sub sendTaskInfo {
    my ($self, $taskId, $taskInformation) = @_;
    my $structure = {
        structId    => "taskInfo",
        content     => {
            id      => $taskId,
            info    => $taskInformation,
        },
    };
    $self->updateTaskInfo($taskId, $taskInformation);
    $self->_sendSerializedOutput($structure);
}

sub sendProgress {
    my ($self, $progress) = @_;
    my $structure = {
        structId        => "progress",
        content         => {
            progress    => $progress,
        },
    };
    $self->updateProgress($progress);
    $self->_sendSerializedOutput($structure);
}

sub sendProgressMessage {
    my ($self, $progressMsg, $prepend) = @_;
    $prepend //= 0;
    my $structure = {
        structId    => "progressMsg",
        content     => {
            message => $progressMsg,
            prepend => ($prepend ? 1 : 0),
        },
    };
    $self->updateProgressMessages($progressMsg, $prepend);
    $self->_sendSerializedOutput($structure);
}

sub sendState {
    my ($self, $state, $msg) = @_;
    my $structure = {
        structId    => "state",
        content     => {
            state   => $state,
            message => $msg,
        },
    };
    $self->updateState($state);
    $self->_sendSerializedOutput($structure);
}

sub sendExecutionStepId {
    my ($self, $id) = @_;
    my $structure = {
        structId       => "stepId",
        content        => {
            id         => $id,
        },
    };
    $self->updateExecutionStepIds($id);
    $self->_sendSerializedOutput($structure);
}

sub _sendSerializedOutput {
    my ($self, $structure) = @_;
    if (!$self->isWorker()) {
        die("Sending output from the main process");
    }
    eval {
        # This prints directly to the pipe because its the currently selected filehandle for the process
        print "$startDelimiter\n";
        print JSON::to_json($structure)."\n";
        print "$endDelimiter\n";
    };
}

# ####################################################################################################################################
# ----- Below that line are methods, relevant for the execution of the scenario ------------------------------------------------------
# ####################################################################################################################################

sub _executeScenario {
    my ($self) = @_;
# This method should be called only from the child process
# All the execution work is done here
    if (!$self->isWorker()) {
        $self->setState(STATE_ERROR, "Execution from main process");
        return 0;
    }
    eval {
        my $executor = $self->_createExecutor();
        # Dump the file template that contains all the parameters for the newly created executor
        $self->_createConfigFile();
        $executor->execute();
        if ($executor->getStatus()->isInFinishedState()) {
            $self->sendProgress(100);
            $self->sendProgressMessage("Process finished successfully");
            $self->sendState(STATE_FINISHED);
        } else {
            $self->sendProgressMessage("Process failed");
            $self->sendState(STATE_ERROR, ${$self->getAdapter()->getConfiguration()->getErrMsgLst()->getMsgLstString()});
        }
    };
    if (my $error = $@) {
        $self->sendProgressMessage("Process execution failed: $@");
        $self->sendState(STATE_ERROR, "Process execution failed: $error");
    }
}

sub _createExecutor {
    my ( $self ) = @_;
    my $progressHandler = LCM::Slpp::SlppProgressHandler->new($self);
    $self->{progressHandler} = $progressHandler;
    my $adapter = $self->getAdapter();
    my $configuration = $adapter->getConfiguration();
    my $executor = createExecutor($adapter->getScenario(), $configuration, $progressHandler);


    $executor->addListener(new LCM::Slpp::SlppProcessTaskListener($self));
    $progressHandler = $executor->getMsgLst()->getProgressHandler();
    $self->{progressHandler} = $progressHandler;
    $self->_initializeSlppExecutionSteps($executor);

#   Set progressHandler of msgList
#   See BaseApplication::_setConfigurationProgressHandler {
    my $configMsgLst = $configuration->getMsgLst();
    my $configErrMsgLst = $configuration->getErrMsgLst();

    $configMsgLst->setProgressHandler($progressHandler);
    $configErrMsgLst->setProgressHandler($progressHandler);
#    End set progressHandler of msgList

    $executor->setMsgLstContext($configuration->getMsgLstContext());
    my $applicationContext = LCM::App::ApplicationContext::getInstance();
    $applicationContext->setExecutor($executor);
    return $executor;
}

sub _initializeSlppExecutionSteps {
    my ($self, $executor) = @_;
    my $tasks = $executor->getSubtasks();
    $self->{executionSteps} = $tasks;

    for my $task (@{$tasks}){
        $task->addListener(new LCM::Slpp::SlppSubtaskListener($task, $self));
        $self->sendExecutionStepId($task->getId());
    }
}

sub getExecutionSteps {
    my ($self) = @_;
    return $self->{executionSteps};
}

sub _createConfigFile {
    my ( $self ) = @_;
    my $adapter = $self->getAdapter();
    my $configuration = $adapter->getConfiguration();
    $configuration->dumpConfigFile('hdblcmweb');
}

1;
