package LCM::Slpp::SlppExecutionHandler;

use strict;
use base qw(SDB::Install::Base Exporter);

use threads;
use LCM::Slpp::ExecutionState qw(STATE_INITIAL STATE_RUNNING STATE_FINISHED STATE_ERROR STATE_DIALOG STATE_ABORTED getApplicableActions);
use LCM::Slpp::Serializer qw(serialize);
use SDB::Install::Log;
use SDB::Install::Globals qw ($gLogDir gPrepareLogPath);
use SDB::Install::SysVars qw($isWin);
use SDB::Install::SAPSystem;
use File::Spec;
use LCM::Slpp::PersistenceHandler;
use LCM::DevelopmentTrace;
use LCM::Slpp::ScenarioExecutor;
use integer; # important due to the timestamps
use File::Path 'remove_tree';

use Data::Dumper;
our @EXPORT_OK = qw(getInstance $HDBLCM_ERROR_MESSAGE);

our $HDBLCM_ERROR_MESSAGE = "Execution of hdblcmweb failed";

my $OPTIONAL_CONFIG_NAME_MAP = {
	'optional_configuration_v3' => 'add_host_roles',
	'optional_configuration_v4' => 'download_components',
	'optional_configuration_v5' => 'remove_host_roles',
	'optional_configuration_v6' => 'extract_components',
};

my $new = sub { return $_[0]->SUPER::new (); };
my $instance = $new->('LCM::Slpp::SlppExecutionHandler');

# TODO: FIX AFTER REFACTORING END: Remove all logic from getters and setters and depend only on the listener attached to the process task after start of execution
sub getSlppSteps {
	my $self = shift();
	my $scenarioExecutor = $self->getScenarioExecutor();
	my @result = ();
	for my $stepId (@{$scenarioExecutor->getExecutionStepIds()}){
		my $statusHash = $scenarioExecutor->getTaskInfo($stepId);
		my ($status, $progress, $warnings) = ($statusHash->{state}, $statusHash->{progress}, $statusHash->{warnings});
		my ($startedAt, $finishedAt) = ($statusHash->{startedAt}, $statusHash->{finishedAt});
		my $stepObject;
		my ($taskId, $taskType, $taskTechnicalName, $taskDisplayName, $taskDescription) =
			($statusHash->{id}, $statusHash->{type}, $statusHash->{name}, $statusHash->{name}, $statusHash->{description});
		$stepObject = {
			Task => {
				id 				 => $taskId,
				type 			 => $taskType,
				technicalName 	 => $taskTechnicalName,
				status 			 => $status,
				startedAt 		 => $startedAt,
				finishedAt 		 => $finishedAt,
				progress 		 => int($progress),
				refreshRate 	 => '1',
				logs 			 => '',
				error			 => '',
				displayName 	 => $taskDisplayName,
				description 	 => $taskDescription,
				externalInfo 	 => 'help.sap.com',
				warningMessages => _createWarningsStructure($warnings)
			}
		};
		push(@result, $stepObject);
	}
	return @result;
}

sub getSlppFinishedSteps {
	my ($self) = @_;
	return grep {$_->{Task}->{status} =~ m/FINISHED/} $self->getSlppSteps();
}

sub getLogFileURIs {
	my ($self) = @_;
	return $self->{logURIs};
}

sub getInstance {
	return $instance;
}

sub init {
# We rely on the FCGI app to ensure that $stopTimeout is between 1 and 60 inclusive
	my ($self, $adapter, $applicationErrorState, $stopTimeout) = @_;
    my $configuration = $adapter->getConfiguration();
    $self->setMsgLstContext($configuration->getMsgLstContext());
    $self->{adapter} = $adapter;
    $self->{stopTimeout} = $stopTimeout;
    $self->{persistenceHandler} = new LCM::Slpp::PersistenceHandler($adapter, $self->_getWebResourcesDir(), $self);
    $self->{persistenceHandler}->cleanupFilesFromPreviousExecution();
	$self->_defineLogs($configuration);

    $self->{scenarioExecutor} = LCM::Slpp::ScenarioExecutor->new($self, $self->getAdapter());

    if(defined($applicationErrorState)){
    	$self->setState(STATE_ERROR, $applicationErrorState);
		$self->stopTool();
    } else {
		$self->setState(STATE_INITIAL);
    }
    return $self;
}

sub getAdapter {
	my ($self) = @_;
	return $self->{adapter};
}

sub getPersistenceHandler {
	my ($self) = @_;
	return $self->{persistenceHandler};
}

sub getProgressMessages {
	my ($self) = @_;
	return $self->getScenarioExecutor()->getProgressMessages();
}

sub getProgress {
	my ($self) = @_;
	return $self->getScenarioExecutor()->getProgress();
}

sub getErrorState {
	my ($self) = @_;
	my $scenarioExecutor = $self->getScenarioExecutor();
	if (defined $self->{errorStateObject}) {
		return $self->{errorStateObject};
	}
	return $scenarioExecutor->constructErrorStateObject();
}

sub getSlppActions {
	my ($self) = @_;
	my $executionState = $self->getState();
	my $actions = getApplicableActions($executionState);
	my $slppActions = [];

	for my $action (@$actions) {
		push(@$slppActions, $action->{element});
	}
	
	return {actions => $slppActions};
}

sub getSlppAction {
	my ($self, $actionId) = @_;
	my $action = $self->getAction($actionId);
	if(defined $action){
		return $action->{element};
	}
	return undef;
}

sub getAction {
	my ($self, $actionId) = @_;
	my $applicableActions = getApplicableActions($self->getState());
	
	for my $action (@$applicableActions){
		if ($action->{id} eq $actionId) {
			return $action;
		}
	}

	return undef;
}

sub getState {
	my ($self) = @_;
	my $scenarioExecutor = $self->getScenarioExecutor();
	return $scenarioExecutor->getState();
}

sub getFinishedAt {
	my ($self) = @_;
	return $self->getScenarioExecutor()->getFinishedAt();
}

sub setState {
	my ($self, $state, $errorState) = @_;
	my $scenarioExecutor = $self->getScenarioExecutor();
	if ($state eq STATE_ERROR && defined $errorState) {
		$self->{errorStateObject} = $errorState;
	}
	$scenarioExecutor->setState($state);
}

sub executeSlppAction {
	my ($self, $action) = @_;

	my $executionMethod = $self->can($action->{executor});
	if($executionMethod){
		return &$executionMethod($self);
	}
	die("Something went terribly wrong in here my friend!\n");
}

sub start {
    my ( $self ) = @_;
    if($self->getState() ne STATE_INITIAL){
    	# TODO return some kind of message/response code
    	return undef;
    }
    $self->setState(STATE_DIALOG);
}

sub abort {
    my ( $self ) = @_;
    if($self->getState() eq STATE_ABORTED){
    	# TODO return some kind of message/response code
    	return undef;
    }
    $self->setState(STATE_ABORTED);
    $self->stopTool();
}

sub submit {
    my ($self) = @_;
    if ($self->getState() ne STATE_DIALOG) {
        # TODO return some kind of message/response code
        return undef;
    }

    my $config = $self->getAdapter()->getConfiguration();
    my $lastParamID = $config->getParamIds()->[-1];
    $config->setBatchValueOfRequiredParams($lastParamID);

    if (!$config->CheckParams(1)) {
        return;
    }

# Close the remaining timer from the auto-stop monitoring
    alarm(0);

    if(!$self->getScenarioExecutor()->startExecution()) {
    	$self->setErrorMessage("Failed to start worker process.");
    	$self->setState(STATE_ERROR);
    	$self->stopTool();
    }

    $self->setState(STATE_RUNNING);
}

sub getScenarioExecutor {
	my ($self) = @_;
	return $self->{scenarioExecutor};
}

sub _createWarningsStructure {
	my ($resultArray, $warningId, $warnings) = ([], 1, @_);
	for my $warning (@{$warnings}){
		push(@{$resultArray}, { WarningMessage => { id => $warningId++, message => $warning } });
	}
	return $resultArray;
}

sub stopTool {
	my ($self) = @_;
	my $scenarioExecutor = $self->getScenarioExecutor();

	if ($scenarioExecutor->isWorkerAlive()) {
		$scenarioExecutor->killWorker();
	}
	eval {
		$self->_persistSlpState();
	};
	if (my $error = $@) {
		$self->setErrorMessage("Failed to persist SL Protocol state:\n" . $error);
	}
# If it hasn't come to any execution, then the log have to be written by the main process
# Otherwise, we depend on the worker process to write the logs
	if (!$scenarioExecutor->hasExecutionStarted()) {
		$self->writeLogs();
	}
# Workaround to delete the useless log directory and
# trace file of the FileBrowser in the non-resident scenario
    my $scenario = $self->getAdapter()->getScenario();
    if ($scenario eq 'optional_configuration_v1' && $self->getState() ne STATE_ERROR) {
# Goes one directory up otherwise you can't delete the
# process's log directory because it is the CWD
        remove_tree($gLogDir) if chdir "..";
    }

    $self->_setTimerToStopTheProcess($self->{stopTimeout});
}

sub _persistSlpState {
	my ($self, $copyLogs) = @_;
	my $persistenceHandler = $self->getPersistenceHandler();
	my $slppAdapter = $self->getAdapter();
	my $sid = $slppAdapter->getSid();
	$self->{logURIs}->{$slppAdapter->getScenario()} = "../../../lmsl/HDBLCM/$sid/log_files/hdblcmweb.log";

	my $tasksInformation = $self->getScenarioExecutor()->getTasksInfo();
	for my $taskId (keys %{$tasksInformation}) {
		if($tasksInformation->{$taskId}->{logLocation}){
			# If log location is not the empty string
			$self->{logURIs}->{$taskId} = "../../../lmsl/HDBLCM/$sid/log_files/" . $tasksInformation->{$taskId}->{resourcesLogFilename};
		}
	}

	$persistenceHandler->persistSlpState();
	$persistenceHandler->copyLogFilesToResourcesDir($tasksInformation);
}

sub _getWebResourcesDir {
	my ($self) = @_;
	my $systems = CollectSAPSystems ();
	my $sapSystem = $systems->{$self->getAdapter()->getSid()};
	my $hanaSharedSidPath = $sapSystem->get_globalSidDir();
	return File::Spec->catdir($hanaSharedSidPath, 'hdblcm', 'resources');
}

sub _defineLogs {
    my ( $self, $configuration) = @_;
    my $actionName = $self->getAdapter()->getScenario();

# Introduce custom handling for the naming of newly introduced scenarios (see: Bug 107300)
    if($OPTIONAL_CONFIG_NAME_MAP->{$actionName}){
        $actionName = $OPTIONAL_CONFIG_NAME_MAP->{$actionName};
    }
# Introduce custom handling for the naming of newly introduced scenarios (see: Bug 107300)

    gPrepareLogPath($self->getAdapter()->getSid(), "hdblcmweb_$actionName");
    $self->_setLogLocationParams($configuration);

    $self->{traceLog} = SDB::Install::Log->new();
    $self->{errorLog} = SDB::Install::Log->new();
    $self->{traceLog}->setMsgLstContext ([$configuration->getMsgLst()]);
    $self->{errorLog}->setMsgLstContext ([$configuration->getErrMsgLst()]);

    my $name = "hdblcmweb";
    my $errorName = "hdblcmweb_error";
    my $logSuffix = '.log';
    my $msgSuffix = '.msg';
    my $logFile = $name . $logSuffix;
    my $msgFile = $name . $msgSuffix;
    my $errorLogFile = $errorName . $logSuffix;
    my $errorMsgFile = $errorName . $msgSuffix;
    my $webResourcesLogsDir = File::Spec->catdir($self->_getWebResourcesDir(), 'log_files');

    $self->tryCreateLogDir();

    $self->{traceLog}->addLogFileLocation(LOG_FORMAT_PLAIN,  $webResourcesLogsDir, 1, 'hdblcmweb.log', LOG_DONT_INDICATE_LOCATION_ON_STDOUT);
    $self->{traceLog}->addLogFileLocation(LOG_FORMAT_PLAIN,  $gLogDir, 1, $logFile, LOG_DONT_INDICATE_LOCATION_ON_STDOUT);
    $self->{traceLog}->addLogFileLocation(LOG_FORMAT_MSGLIST,  $gLogDir, 1, $msgFile, LOG_DONT_INDICATE_LOCATION_ON_STDOUT);
    $self->{errorLog}->addLogFileLocation(LOG_FORMAT_PLAIN,  $gLogDir, 1, $errorLogFile, LOG_DONT_INDICATE_LOCATION_ON_STDOUT);
    $self->{errorLog}->addLogFileLocation(LOG_FORMAT_MSGLIST,  $gLogDir, 1, $errorMsgFile, LOG_DONT_INDICATE_LOCATION_ON_STDOUT);
}

sub _setLogLocationParams {
    my ($self, $configuration) = @_;
    $configuration->setValue('LogDirectory', $gLogDir);
    $configuration->setValue('TraceFile', LCM::DevelopmentTrace::GetDevelopmentTraceFile());
}

sub tryCreateLogDir {
    my ($self) = @_;
    my $webResourcesDir = $self->_getWebResourcesDir();
    my $logDir = File::Spec->catfile($webResourcesDir, 'log_files');
    return if -d $logDir;

    mkdir($logDir) && chmod(0755, $logDir);
}

sub writeLogs {
    my ( $self ) = @_;
# In case of update_system, when the installer component is updated, the old hdblcm folder is deleted
# and the hdblcmweb.log file cannot be created
    $self->tryCreateLogDir() if $self->getAdapter()->getScenario() eq 'update_system';

    my $isInfoLogWritten = $self->{traceLog}->writeLogs();
    my $isErrorLogWritten = $self->{errorLog}->writeLogs();

    if($isInfoLogWritten && $isErrorLogWritten){
        LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
    }
}

sub _setTimerToStopTheProcess {
	my ($self, $timeout) = @_;

	# default timeout value 3, might need a change (previously 5 seconds)
	$timeout = 3 if(!defined($timeout));
	# alarm(0) does not trigger the $SIG{ALRM} handler (see perldoc documentation on alarm)
	$timeout = 1 if ($timeout < 1);
	alarm(0); # Clean up any not needed signals that might have been scheduled
	$SIG{ALRM} = sub { threads->exit(); };
	alarm($timeout);
}

# Temporary workaround for the Add/Remove hosts and Configure ISC scenarios
# That will ensure the same behavior as the refactored scenarios (Update, SLD)
sub _getDisplayNameByStepId {
	my ($self, $stepId, $state) = @_;
	my $stepName = $stepId;

	return $stepName if($state eq STATE_RUNNING);

	if($stepName =~ m/^Adding\s/){
		$stepName =~ s/^Adding\s/Add /;
	} elsif ($stepName =~ /^Updating\s/) {
		$stepName =~ s/^Updating\s/Update /;
	} elsif ($stepName =~ /^Removing\s/) {
		$stepName =~ s/^Removing\s/Remove /;
	} elsif ($stepName =~ /^Configuring\s/) {
		$stepName =~ s/^Configuring\s/Configure /;
	}

	return $stepName;
}

1;
