package LCM::SummaryTreeBuilder;

use strict;
use LCM::Component;
use LCM::ComponentManager::SystemComponentManager;
use LCM::AdditionalHostHelper;
use SDB::Install::Version;
use LCM::Utils::CommonUtils qw(findAutoConfiguration);
use LCM::Configuration::GenericStackAny qw (INSTALL_ACTION UPDATE_ACTION UPDATE_COMPONENTS_ACTION UNINSTALL_ACTION ADD_HOSTS);
use LCM::DevelopmentTrace;
use SDB::Install::Globals qw ($gKeynameHLM $gKeynameEngine $gProductNameSystem $gProductNameEngine getHumanReadableRolesString $gLogDir);
use SDB::Install::Configuration qw($bool_true_pattern);
use LCM::Configuration::RenameSystemConfiguration qw($SYSTEM_RENAME_ACTION);
use LCM::PersistencyUtils qw (isPendingUpdate);
use LCM::Utils::RemoveHosts qw(IsRemovingRemoteHost);
use LCM::Utils::RemoveRoles qw(IsRemovingLocalRole IsRemovingRemoteRole);
use experimental qw (smartmatch);

our $notUsedParameterMsg     = "Parameter '%s' will not be used.";
our $defaultNotApplicableMsg = 'It is not applicable in the current context.';

sub build{
    my ($self,$instconfig, $component_manager) = @_;
    my $roottext = $self->getRootText($instconfig);

    _addPendingActionWarning($instconfig, $roottext);
    _addDeprecatedPluginsWarnings($instconfig, $component_manager);
    _removeDuplicateWarnings($instconfig);
    _addParamsWarnings($instconfig);

    _addSkippedParamsWarnings($instconfig);

    my @rootChildren = ();
    my $summaryTree  = {
        "text"     => $roottext,
        "children" => \@rootChildren
    };

    if($instconfig->getAction() ne UNINSTALL_ACTION) {
        push( @rootChildren, $self->_buildOperationParamsNode($instconfig) );
    }
    if(defined($component_manager)) {
        push( @rootChildren, _buildSoftwareComponentsNode( $component_manager, $instconfig) );
    }

    if(defined($component_manager) && !$instconfig->isSkipped('SelectedXs2Applications')){
        my $xs2Apps = $component_manager->getXs2ApplicationComponents();
        if(scalar(@{$xs2Apps}) > 0 && $instconfig->getAction() ne UNINSTALL_ACTION){
            push(@rootChildren, _buildXs2ApplicationsNode($component_manager, $instconfig));
        }
    }

    if ( _shouldBuildAdditionalHostsNode($instconfig) ) {
        push( @rootChildren, _buildAdditionalHostsNode( $instconfig ));
    }
    if ( _shouldBuildAutoAddedXs2RolesNode($instconfig) ) {
        push( @rootChildren, _buildAutoAddedXs2RolesNode( $instconfig ));
    }
    if($instconfig->getAction() eq UNINSTALL_ACTION) {
        if(IsRemovingRemoteHost($instconfig)){
            my $hostForRemoval = $instconfig->getValue('RemoveHosts');
            push(@rootChildren, _buildRemoveHostsNode($instconfig));
        }
        if(IsRemovingLocalRole($instconfig) || IsRemovingRemoteRole($instconfig)){
            push(@rootChildren, _buildRemoveRolesNode($instconfig));
        }
    }
    push(@rootChildren, _buildLogLocationInfoNode($instconfig));
    return $summaryTree;

}

sub buildSummaryTree {
    my ($instconfig, $component_manager) = @_;
    my $summaryTreeBuilder;

    my $isInstallCockpitOnPlatform = $instconfig->isa('LCM::Configuration::Cockpit::CockpitOnHANAConfiguration') && $instconfig->isCockpitOnPlatformAction() && !$instconfig->isCockpitStackUpdate();
    if($isInstallCockpitOnPlatform){
        require LCM::Configuration::Cockpit::InstallCockpitOnPlatformSummaryTreeBuilder;
        $summaryTreeBuilder = bless({} , 'LCM::Configuration::Cockpit::InstallCockpitOnPlatformSummaryTreeBuilder');
    } else {
        $summaryTreeBuilder = bless({} , 'LCM::SummaryTreeBuilder');
    }
    return $summaryTreeBuilder->build($instconfig, $component_manager);
}

sub _addPendingActionWarning {
	my ($instconfig, $roottext) = @_;
	my $pendingUpdateSystems = {};
	my $isPendingUpdate = isPendingUpdate($instconfig->{current_sid}, $pendingUpdateSystems);
	my $isPrepareUpdate = $instconfig->can('isPrepareUpdate') && $instconfig->isPrepareUpdate();
	my $isIgnorePendingUpdate = $instconfig->getIgnore('check_pending_upgrade');
	my $isContinueUpdate = $instconfig->getValue('ContinueUpdate');
	my $isResumeUpdate = $isPendingUpdate && !$isPrepareUpdate && !$isIgnorePendingUpdate && $isContinueUpdate;

	if ($isResumeUpdate) {
		if ($instconfig->isSuccessfullyPreparedUpdate()) {
			my $infoMsg = "Execute prepared $gProductNameSystem update";
			_addInfoMsg($instconfig, $roottext, $infoMsg);
		} else {
			my $firstTask = undef;
			my $app = LCM::App::ApplicationContext::getInstance()->getApplication ();
			my $executor = $app->{executor};
			my $subtasks = $executor->getSubtasks();
			for my $task (@$subtasks) {
				next if $task->isSkipped();
				$firstTask = $task;
				last;
			}
			$executor->clearAllSubtasks();
			my $firstTaskName = (defined $firstTask) ? $firstTask->getName() : undef;

			my $action = $instconfig->getAction();
			my $wng = "Resume $gProductNameSystem $action at step '" . $firstTaskName . "'";
			_addWarningMsg($instconfig, $roottext, $wng);
		}
	}
}

sub _addWarningMsg {
	my ($instconfig, $roottext, $wng) = @_;

	if (exists $instconfig->{warnings}) {
		my %warnings = map {$_ => 1} @{$instconfig->{warnings}};
		if (!exists($warnings{$wng})) {
			push (@{$instconfig->{warnings}}, $wng);
		}
	} else {
		$roottext .= " ($wng)";
	}
}

sub _addInfoMsg {
    my ($instconfig, $roottext, $msg) = @_;
    if (! exists $instconfig->{infoMsgs}){
    	$instconfig->{infoMsgs} = [];
    }
    if (exists $instconfig->{infoMsgs}) {
        my %infoMsgs = map {$_ => 1} @{$instconfig->{infoMsgs}};
        if (!exists($infoMsgs{$msg})) {
            push (@{$instconfig->{infoMsgs}}, $msg);
        }
    } else {
        $roottext .= " ($msg)";
    }
}

sub _addSkippedParamsWarnings{
    my ($instconfig) = @_;
    my $autoConfigFile = findAutoConfiguration($instconfig->getAction()) || findAutoConfiguration();
    my $useCustomConfiguration = $autoConfigFile eq $instconfig->{config_file} ? 0 : 1;

    foreach my $param_id ( @{ $instconfig->getParamIds() } ) {
        next if ! $instconfig->isSkipped($param_id);
        next if grep { $_ eq $param_id } ('cmd_line_action', 'Ignore');

        my $configValue  = undef;
        my $defaultValue = $instconfig->getDefault($param_id);
        my $cmdLineValue = $instconfig->getCommandLineValue($param_id);
        if ( $useCustomConfiguration ){
            $configValue = $instconfig->getValueFromConfigfile($param_id);
            if(defined $configValue){
                if($instconfig->getType($param_id) =~ /boolean/){
                    $configValue = ($configValue =~ /$bool_true_pattern/) ? 1 : 0;
                } elsif ($instconfig->getType($param_id) =~ /mapping_list/){
                    $configValue = (scalar @$configValue == 0 || 
                        scalar @$configValue > 0 && ! @$configValue[0]) ? undef : $configValue;
                } elsif ($instconfig->getType($param_id) =~ /path/){
                    $configValue = $instconfig->substituteValues($configValue);
                }
            }
        }
        next if ! defined $cmdLineValue && $useCustomConfiguration && ! defined $configValue;
        if ( defined $cmdLineValue || $useCustomConfiguration && defined $configValue && $configValue ne $defaultValue ){
            my $param = $instconfig->{params}->{$param_id};
            if (not exists $instconfig->{warnings}) {
                $instconfig->{warnings} = [];
            }
            my $warning;
            my $cmdLineOption = defined $instconfig->getOpt($param_id) ? substr($instconfig->getOpt($param_id), 2) : undef;
            my $notUsedMsg = sprintf($notUsedParameterMsg, $cmdLineOption);
            if (exists $param->{skip_warning}){
                $warning = sprintf("%s %s", $notUsedMsg, $param->{skip_warning});
            } else {
                $warning = sprintf("%s %s", $notUsedMsg, $defaultNotApplicableMsg);
            }
            push (@{$instconfig->{warnings}}, $warning);
            $instconfig->getMsgLst()->addWarning($warning);
        }
    }
}

sub _shouldBuildAdditionalHostsNode {
	my ($instconfig) = @_;
	my $isActionValid =  grep { $_ eq $instconfig->getAction() } ( INSTALL_ACTION(), UPDATE_ACTION(), UPDATE_COMPONENTS_ACTION() );
	my $isAddingHosts = $instconfig->getValue('AddHosts') && ! $instconfig->isSkipped('AddHosts');

	return $isActionValid && $isAddingHosts;
}

sub _shouldBuildAutoAddedXs2RolesNode {
	my ($instconfig) = @_;
	return 0 if(!$instconfig->getValue('AutoAddXS2Roles') || $instconfig->isSkipped('AutoAddXS2Roles'));
	my $autoAssignMap = $instconfig->getAutoAssignXs2RolesMap();
	return scalar( grep { length($autoAssignMap->{$_}) > 0 } keys(%{$autoAssignMap}));
}

sub _addParamsWarnings{
    my ($instconfig) = @_;

    my $param;
    foreach my $param_id ( @{ $instconfig->getParamIds() } ) {
        $param = $instconfig->{params}->{$param_id};
        unless($param->{warnings})
        {
            next;
        }
        
        if (not exists $instconfig->{warnings}) {
            $instconfig->{warnings} = [];
        }
                
        foreach my $warning (@{$param->{warnings}})
        {   
            my $alreadyAddedToWarnings = 0;
            foreach my $oldWarning (@{$instconfig->{warnings}})
            {
                if($oldWarning eq $warning)
                {
                    $alreadyAddedToWarnings = 1;
                    last;
                }
                
            }            
            push (@{$instconfig->{warnings}}, $warning) unless $alreadyAddedToWarnings;
        }
    }
    return $instconfig->{warnings};
}

sub getUIValuedNeededParamIds {
	return ['AddLocalRoles', 'AddRoles'];
	
}

sub _buildOperationParamsNode {
	my ($self,$instconfig)           = @_;
	my @operationParams        = ();
	my $operationParamsNodeRef = {
		"text"     => $self->getTitle($instconfig),
		"children" => \@operationParams
	};

	_hideNotNeededParamsFromSummary($instconfig);

    my @params = _sortParams($instconfig);
	foreach my $param_id ( @params ) {
		my $text = $instconfig->getSummaryItem( $param_id, $param_id ~~ getUIValuedNeededParamIds() );
		if (defined $text) {
			my $nodeRef = { "text" => $text };
			push( @operationParams, $nodeRef );
		}
	}

	return $operationParamsNodeRef;
}

sub _sortParams {
    my ($instconfig) = @_;
    my @sortedParams = @{$instconfig->getParamIds()};    # copy
    
    # params without "summaryOrder" propertie will be last in the sorted array
    return sort {
        my $firstElementSummaryOrder = $instconfig->{params}->{$a}->{summaryOrder};
        my $secondElementSummaryOrder = $instconfig->{params}->{$b}->{summaryOrder};
        
        if(defined $firstElementSummaryOrder && defined $secondElementSummaryOrder) {
            # if both elements have summary order
            return $firstElementSummaryOrder <=> $secondElementSummaryOrder;
        }
        
        if(defined $firstElementSummaryOrder) {
            return -1;
        }
        
        if(defined $secondElementSummaryOrder) {            
            return 1;
        }
        
        if(defined $instconfig->{params}->{$a}->{order} && defined $instconfig->{params}->{$b}->{order}) {
            return $instconfig->{params}->{$a}->{order} <=> $instconfig->{params}->{$b}->{order};
        }
        
        return 1;
    } @sortedParams;
}

sub getTitle {
    my ($self,$instconfig) = @_;
    
    if($instconfig->can("getSummaryTitle")) {
        return $instconfig->getSummaryTitle();
    }    
    if($instconfig->isa("LCM::Configuration::SLD::RegistrationConfiguration")) {
        return 'Configuration Parameters';
    }
    
    if ($instconfig->isUpdate()) {
    	return 'Update Parameters';
    }
    return 'Installation Parameters';
}

sub _hideNotNeededParamsFromSummary {
	my ($instconfig) = @_;
	my $notNeededParamIds = _getNotNeededParamIds($instconfig);

	for my $parameterId (@{$notNeededParamIds}) {
		$instconfig->setHidden($parameterId, 1);
	}
}

sub _getNotNeededParamIds {
	my ($configuration) = @_;
	my $notNeededParameters = ['PrintTimezones', 'SelectedComponents', 'AddHosts', 'ContinueUpdate', 'ComponentFsRoot', 'ComponentDirs', 'UseSimpleUI'];

	if($configuration->getDefault('ISCMode') eq $configuration->getValue('ISCMode')){
		push(@{$notNeededParameters}, 'ISCMode');
	}
	return $notNeededParameters;
}

sub _buildSoftwareComponentsNode {
	my ( $component_manager, $instconfig ) = @_;
	my $action = $instconfig->getAction();
	my @swComponentsChildren = ();
	my $swComponentsNodeRef  = {
		"text"     => "Software Components",
		"children" => \@swComponentsChildren
	};

	my $system_component_manager = $component_manager->getSystemComponentManager ();
	if ($action eq UNINSTALL_ACTION) {
	   $component_manager = $system_component_manager;
	}
	
    my $allComponents = $component_manager->getAllComponents();
       
	foreach my $component ( @$allComponents ) {
		if ($component->isInternal()) {
			next;
		}
        my $installed_component;
        if (defined $system_component_manager){
            $installed_component = $system_component_manager->getComponentByKeyName( $component->getComponentKeyName() );
        }
        my $text = _getComponentNodeText($installed_component, $component, $instconfig );
        my @children = ({ "text" => $text });
        if ($component->isa("LCM::Component::Installable") and $component->isComponentSelected()) {
        	my $location = $component->getPath();
        	push (@children, {"text" => "Location: $location"}) unless not $location;
        }

		my $swComponentNode = {
			"text"     => $component->getComponentName(),
			"children" => \@children
		};
		push( @swComponentsChildren, $swComponentNode );
	}

	my $serverComponent = $component_manager->getComponentByKeyName($gKeynameEngine);
	if ((($action eq UPDATE_ACTION) || ($action eq UPDATE_COMPONENTS_ACTION)) &&
		defined $serverComponent && $serverComponent->isComponentSelected() && $serverComponent->isUpdate()) {
		for my $component (@{$instconfig->getDeprecatedComponentsForUninstall()}) {
	        my $version = $component->getManifest()->getVersion();
	        my @children = ({ "text" => "Uninstall version $version" });
			my $swComponentNode = {
				"text"     => $component->getComponentName(),
				"children" => \@children
			};
			push( @swComponentsChildren, $swComponentNode );
		}
	}

	return $swComponentsNodeRef;
}

sub _buildXs2ApplicationsNode {
	my ($componentManager, $instconfig) = @_;
	my $allComponents = $componentManager->getXs2ApplicationComponents();
	my @swComponentsChildren = ();
	my $swComponentsNodeRef  = {
		"text"     => "XS Advanced Components",
		"children" => \@swComponentsChildren
	};

	for my $component(@{$allComponents}) {
		my $text = _getXsComponentNodeText($instconfig, $component);
		my @children = ({ "text" => $text });
		my $swComponentNode = {
			"text"     => $component->getComponentName(),
			"children" => \@children
		};

		if ($component->isComponentSelected()) {
			my $location = $component->getPath();
			push (@children, {"text" => "Location: $location"});
		}
		push(@swComponentsChildren, $swComponentNode);
	}
	return $swComponentsNodeRef;
}

sub shouldAddHLMToSummary {
	my ( $components, $instconfig ) = @_;
	
    my $scm = new LCM::ComponentManager::SystemComponentManager();
    my $sapmnt = $instconfig->getValue('Target');
    my $sid = $instconfig->getValue('SID');
    $scm->detectComponentsBySapmntSid( $sapmnt, $sid );
    
    my %componentsHash = ();
    for ( @$components ) {
        $componentsHash{$_->getComponentKeyName()} = $_;
	}
	my $isHLMInComponentManager = ( defined $componentsHash{$gKeynameHLM} );

    $scm->{shouldAddHLMToSummary} = ( $scm->getComponentByKeyName($gKeynameHLM) && ! $isHLMInComponentManager) ; 
	return $scm;
}

sub _getXsComponentNodeText {
	my ($instconfig, $component) = @_;
	my $scm = $instconfig->getSystemComponentManager();
	my $componentKey = $component->getManifest()->getComponentKey();
	my $isComponentInstalled = defined($scm) && defined($scm->getComponentByKeyName($componentKey));
	my $isSelected = $component->isComponentSelected();

	if ($isComponentInstalled) {
		my $existingVersion = $scm->getComponentByKeyName($componentKey)->getVersion();
		return $isSelected ? sprintf('Update from version %s to %s', $existingVersion, $component->getVersion()) :
							 sprintf("Version %s , do not update", $existingVersion);
	}
	return $isSelected ? sprintf('Install version %s', $component->getVersion()) : 'Do not install';
}

sub _getComponentNodeText() {
	my ( $installed_component, $available_component, $instconfig ) = @_;
	my $action = $instconfig->getAction();
	if ($action eq UNINSTALL_ACTION) {
		if ( defined $installed_component ) {
			my $installed_version = $installed_component->getManifest()->getVersion();
			if ( $installed_component->isComponentSelected() ) {
				return 'Uninstall version ' . $installed_version;
			} else {
				return 'Version ' . $installed_version . ', do not uninstall';
			}
		} 
		return 'Not installed';
	}

	my $isUpdateWithCheckPending = $instconfig->isUpdate() && ! $instconfig->getIgnore('check_pending_upgrade');
	if ($isUpdateWithCheckPending && $available_component->isComponentSelected()) {
		my $hasPersistenceXml = $available_component->hasPersistenceFile($instconfig);
		my $xml = $hasPersistenceXml ? $available_component->getPersistenceXMLObject($instconfig) : undef;

		if (defined $xml) {
			return _getPendingActionTextNode($xml, $available_component, $installed_component, $instconfig->isPrepareUpdate());
		}
	}

	my $isUpdateScenario = $instconfig->isUpdate();
	my $isSelected = $available_component->isComponentSelected();
	if (defined($installed_component) && $isSelected && $isUpdateScenario) {
		return sprintf('Update from version %s to %s', $installed_component->getVersion(), $available_component->getVersion());
	}
	if (defined($installed_component) && $isUpdateScenario) {
		my $installed_version = $installed_component->getVersion();
		my $installedManifest = $installed_component->getManifest();
		my $targetManifest = $available_component->getManifest();
		my $componentName = $available_component->getComponentName();
		my $canUpgrade = $instconfig->canUpgradeComponent($installedManifest, $targetManifest, $componentName);

		return $canUpgrade ? "Version $installed_version , do not update" : 
							 "Version $installed_version , no update available. ". $instconfig->getErrorString();
	}
	return $isSelected ? sprintf('Install version %s', $available_component->getVersion()) : 'Do not install';
}

sub _getPendingActionTextNode {
	my ($pendingXmlObject, $availableComponent, $installedComponent, $isPrepareUpdate) = @_;
	my $sourceVersionString = $pendingXmlObject->getSourceVersionString();
	my $targetVersionString = $pendingXmlObject->getTargetVersionString();
	my $targetVersionObject = new SDB::Install::Version(split('\.', $targetVersionString));
	my $availableVersionObject = new SDB::Install::Version(split('\.', $availableComponent->getVersion()));
	my $isUpdate = ($pendingXmlObject->getActionString() =~ /update/);
	my $canResume = $availableVersionObject->isEqual($targetVersionObject);

	if($canResume){
		if (!$isUpdate) {
			return sprintf("Resume pending installation of version %s", $targetVersionString);
		}
		if (!$isPrepareUpdate) {
			return sprintf("Resume pending update from version %s to %s", $sourceVersionString, $targetVersionString);
		}
	}
	return $isUpdate ? sprintf("Update from version %s to %s", $sourceVersionString, $availableComponent->getVersion()) :
					   sprintf("Install version %s", $availableComponent->getVersion());
}

sub _buildAdditionalHostsNode {
	require SDB::Install::Configuration::AddHostsParser;
	require LCM::AdditionalHostHelper;

	my ($instconfig) = @_;
	my @hostsChildren = ();
	my $parser = new SDB::Install::Configuration::AddHostsParser ();
	my $hostsNodeRef  = {
		"text"     => "Additional Hosts",
		"children" => \@hostsChildren
	};

	$parser->parse($instconfig->getValue('AddHosts'));

	foreach my $host (@{$parser->getHosts()}){
		my $workerGroups = $parser->getValue($host, 'workergroup') || [];
		my $workerGroupsString = scalar(@{$workerGroups}) > 0 ? join(',', @{$workerGroups}) : undef;
		my $hostObj = new LCM::AdditionalHostHelper({
			host => $host,
			workergroup => $workerGroupsString,
			group => $parser->getValue($host, 'group'),
			role => join( ',', @{$parser->getRoles($host)}),
			storage_partition => $parser->getValue($host, 'storage_partition'),
		});

		my @hostChildren = ();
		push (@hostChildren, {text => "Role: " . $instconfig->getCSVRoleDescriptiveNames($hostObj->getRoleDisplayValue(), ', ')}) if $hostObj->getRole();
		push (@hostChildren, {text => "High-Availability Group: " . $hostObj->getGroupDisplayValue()}) if $hostObj->getGroup();
		push (@hostChildren, {text => "Worker Group: " . $hostObj->getWorkerGroupDisplayValue()}) if($hostObj->getWorkerGroupDisplayValue());
		push (@hostChildren, {text => "Storage Partition: " . $hostObj->getStoragePartitionDisplayValue()});

		my $hostNode = {
			"text"     => $hostObj->getHost(),
			"children" => \@hostChildren
		};

		push (@hostsChildren, $hostNode);
	}

	return $hostsNodeRef;
}

sub _buildAutoAddedXs2RolesNode {
	my ($instconfig) = @_;
	my @hostsChildren = ();
	my $hostsNodeRef  = {
		"text"     => "Automatically Assigned Roles",
		"children" => \@hostsChildren
	};
	my $autoAddedRolesMap = $instconfig->getAutoAssignXs2RolesMap();

	foreach my $host (keys(%{$autoAddedRolesMap})){
		my $xs2Role = $autoAddedRolesMap->{$host};

		if(length($xs2Role) > 0){
			push (@hostsChildren, { "text" => sprintf('Host: %s Role: %s', $host, getHumanReadableRolesString($xs2Role)) });
		}
	}
	return $hostsNodeRef;
}

sub _removeDuplicateWarnings{
	my ($instconfig) = @_;
	my $warningsMultiSet = $instconfig->{warnings};
	$instconfig->{warnings} = [];
	foreach my $warning (@$warningsMultiSet){   
        my %warnings = map {$_ => 1} @{$instconfig->{warnings}};
        if(!exists($warnings{$warning})){
        	push( @{$instconfig->{warnings}}, $warning );
        }
	}
}

sub _buildRemoveHostsNode {
    my ($instconfig) = @_;
    my $hostForRemoval = $instconfig->getValue('RemoveHosts');
    return {
        "text"     => "Remove Host Properties",
        "children" => [
            { 'text' => sprintf('Host Name(s): %s', $hostForRemoval) },
        ]
    };
}

sub _buildRemoveRolesNode {
    my ($instconfig) = @_;

    my @hostsChildren = ();
    my $hostsNodeRef  = {
        "text"     => "Remove Host Roles",
        "children" => \@hostsChildren
    };
    my $ownInstance = $instconfig->getOwnInstance();
    my $mHostRolesInfo = $ownInstance->getHostRolesInfo();
    my $mRemoveRolesValue = $instconfig->getValue('RemoveRoles');

    foreach my $host (keys %$mRemoveRolesValue){
        my @hostChildren = ();
        my $additionalRoles = $mRemoveRolesValue->{$host};
        next if (! defined $additionalRoles);

        push( @hostChildren, {text => "Remove Role(s): " . $additionalRoles});
        my $hostNode = {
            "text"     => $host,
            "children" => \@hostChildren
        };
        push(@hostsChildren, $hostNode);
    }
    return $hostsNodeRef;
}

sub _addDeprecatedPluginsWarnings {
	my ($instconfig, $component_manager) = @_;

	return if (!defined $component_manager);
	my $serverComponent = $component_manager->getComponentByKeyName($gKeynameEngine);
	if (defined $serverComponent && $serverComponent->isComponentSelected() && $serverComponent->isUpdate()) {
		$instconfig->_addDeprecatedPluginsWarnings();
	}
}

sub _buildLogLocationInfoNode {
    my ($instconfig) = @_;
    my $traceFile = LCM::DevelopmentTrace::GetDevelopmentTraceFile();
    my $logLocationNodes = [
        {text => "Log directory: $gLogDir"},
        {text => "Trace location: $traceFile"},
    ];
    return {
        text => "Log File Locations",
        children => $logLocationNodes,
    };
}

sub getRootText{
    my($self,$instconfig) = @_;
    my $roottext;
    my $flavourProductName = LCM::App::ApplicationContext::getFlavourProductName();
    if ($instconfig->getAction() eq UNINSTALL_ACTION) {
        $roottext = "$flavourProductName System Uninstallation";
    } elsif ($instconfig->getAction() eq INSTALL_ACTION) {
        $roottext = "$flavourProductName System Installation";
    } else {
        $roottext = $flavourProductName;
    }
    return $roottext;
}
return 1;
