package LCM::Task::CommonTask::AssignAdditionalRemoteRolesTask;

use strict;
use parent qw (LCM::Task::SHASupportedTask);
use SDB::Install::Globals qw ($gHostRoleAcceleratorWorker $gHostRoleAcceleratorStandby $gLogDir);
use SDB::Install::RemoteHostctrlHosts;
use LCM::HostsParser;
use SDB::Install::SysVars qw($path_separator);
use SDB::Install::System  qw (copySelectedFiles);
use SDB::Install::SAPInstance::TrexInstance;
use SAPDB::Install::Hostname;
use LCM::Utils::CommonUtils qw(getHdbmodifyTimeoutValues buildTimeoutsString);
use LCM::Task::TaskProgressHandler::AddHostRolesTaskProgressHandlerFactory;

# These constants are here because the GSI configuration
# does not have 'system_user' and 'add_roles' parameters
my $PROGRESS_MESSAGE = 'Assigning Additional Roles to the Remote Hosts';
my $XS_PROGRESS_MESSAGE = 'Assigning Role to the XS Advanced Controller Host';
my $HOSTCTRL_OPERATION = 'hdblcm_assign_roles_v8';
my $PASSWORD_KEYS = ['Password', 'AcceleratorPassword', 'OrgManagerPassword', 'SQLSysPasswd', 'SQLTenantUserPassword'];
my $SYSTEM_USER_OPTION = 'system_user';
my $ADD_ROLES_OPTION = 'add_roles';

our @EXPORT = qw ($PROGRESS_MESSAGE);

sub getId {
    return 'assign_additional_remote_roles';
}

sub getName {
    my $self = shift();    
    return ($self->isXsControllerHostTask) ? "Assign Role to the XS Advanced Controller Host" : "Assign Additional Roles to the Remote Hosts";    
}

sub getExecutionName {
    my $self = shift();    
    return ($self->isXsControllerHostTask) ? $XS_PROGRESS_MESSAGE : $PROGRESS_MESSAGE;
}

sub _getNumberOfExpectedOutputLines {
    return 100;
}

sub _executeInternal {
    my $self = shift();
    my $remoteHostsObject = $self->_getConfiguration()->getRemoteHosts();
    my $message = $self->getMsgLst()->addProgressMessage($self->getExecutionName() . '...');
    my $saveContext = $self->setMsgLstContext([$message->getSubMsgLst()]);
    $self->getMsgLst()->getProgressHandler()->incrementIndentationDepth();

    my $returnCode = $remoteHostsObject->isHostctrl() ? $self->_executeSHAOperation() : $self->_executeAsRoot();

    if($returnCode) {
        $self->getStatus()->_setFinishedState();
        $message->endMessage (undef, sprintf("%s finished", $self->getName()));
    } else {
        $self->getStatus()->_setErrorState();
        $message->endMessage (undef, sprintf("%s failed", $self->getName()));
    }
    $self->getMsgLst()->getProgressHandler()->decrementIndentationDepth();
    $self->setMsgLstContext($saveContext);
}

sub _executeSHAOperation {
    my ($self) = @_;
    my $localHost = $self->_createRemoteHostsHostctrlObject();
    my $configuration = $self->_getConfiguration();

    $localHost->setMsgLstContext( $self->getMsgLstContext() );
    $localHost->useSidadm();

    my $rc = 1;
    my $optionsMap = $self->buildSHAOptionsMap();
    my $hostsArray = $self->_getExecutionHosts();

    my $exitCode = $localHost->executeHostctrlParallel($HOSTCTRL_OPERATION, $configuration, undef, $PASSWORD_KEYS, undef, undef, undef,
                                                         undef, undef, $optionsMap, undef, $hostsArray, undef, 0);
    my $outputBuffer = $localHost->getOutputBuffer();
    my $executionOutput = [ split('\n', $outputBuffer->[0]) ];
    $self->_setLogLocation($self->_parseLogFileLocation($executionOutput));

    if (!defined $exitCode || ($exitCode != 0)) {
        $self->setErrorMessage($self->getFailedExecutionMessage(), $localHost->getErrMsgLst());
        $rc = undef;
    }
    return $rc;
}

sub _createRemoteHostsHostctrlObject {
    my ($self) = @_;
    my $hostsArray = $self->_getExecutionHosts();
    my $remoteHostsObject = new SDB::Install::RemoteHostctrlHosts(@{$hostsArray});
    my $progressHandler  = $self->getMsgLst()->getProgressHandler();

    for my $host (@{$hostsArray}){
        $remoteHostsObject->setOutputHandler($host, $progressHandler);
    }

    return $remoteHostsObject;
}

sub setAutoAsignedRolesMap {
    my ($self, $autoAsignedRolesMap) = @_;
    $self->{auto_asigned_roles_map} = $autoAsignedRolesMap;
}

sub _getAutoAsignedRolesMap {
    my ($self, $configuration) = @_;
    return $self->{auto_asigned_roles_map} if defined $self->{auto_asigned_roles_map};

    my $isAutoAssignEnabled = $configuration->getValue('AutoAddXS2Roles') && ! $configuration->isSkipped('AutoAddXS2Roles');
    return $isAutoAssignEnabled ? $configuration->getAutoAssignXs2RolesMap() : {};
}

sub setRemoteRolesMap {
    my ($self, $remoteRolesMap) = @_;
    $self->{remote_roles_map} = $remoteRolesMap;
}

sub _getRemoteRolesMap {
    my ($self, $configuration) = @_;
    return $self->{remote_roles_map} if defined $self->{remote_roles_map};
    return $configuration->{params}->{AddRoles}->{value};
}

sub _getRemoteRolesString {
    my ($self, $configuration) = @_;

    my $result = [];
    my $mapValues = $self->_getRemoteRolesMap($configuration);
    my $autoAssignedRolesMap = $self->_getAutoAsignedRolesMap($configuration);
    my $originValues = defined $self->getHosts() ? $self->getHosts() : $configuration->{params}->{AddRoles}->{origin_values};

    for my $host (@{$originValues}) {
        next if(LCM::HostsParser::IsLocalHost($host));

        my $autoAssignedRole = $autoAssignedRolesMap->{$host};
        my $autoAssignedRoleString = length($autoAssignedRole) > 0 ? ":role=$autoAssignedRole" : '';
        my $isAddingRoles = defined($mapValues->{$host}) && length($mapValues->{$host}) > 0;
        my $userSpecifiedRolesString = $isAddingRoles ? sprintf(':role=%s', join(':role=', split(',', $mapValues->{$host}))) : '';

        next if(! $isAddingRoles && length($autoAssignedRoleString) <= 0);

        push(@{$result}, sprintf('%s%s%s', $host, $autoAssignedRoleString, $userSpecifiedRolesString));
    }
    return join(',', @{$result});
}

sub _getExecutionHosts {
    my ($self) = @_;
    return [ hostname() ];
}

sub getParameterToOptionMapping {
    return {
        'Target' => 'SAPMNT',
        'SID' => 'SID',
        'SkipModifySudoers' => 'SUD',
    };
}

sub _executeAsRoot {
    my ($self) = @_;
    my $executor = $self->_createProcessExecutor();
     my $saveCntxt = $self->getMsgLstContext();
    $executor->setMsgLstContext( $self->getMsgLstContext() );
    my $rc = $executor->executeProgram(1);
    $self->_setLogLocation($self->_parseLogFileLocation($executor->getOutputLines()));
    $self->setMsgLstContext($saveCntxt);

    return 1 if (defined $rc && $rc == 0);

    $self->getErrMsgLst()->addError($self->getFailedExecutionMessage());
    return undef;
}

sub _getExecutable {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    
    my $hdbExecutable =
           join($path_separator,
                $configuration->getSAPSystem()->get_globalTrexInstallDir(),
                'bin',
                'hdbmodify');
    
    return $hdbExecutable;
}

sub getSlppLogFileName {
    return 'assign_remote_roles.log';
}

sub buildSHAOptionsMap {
    my $self = shift();
    my $result = $self->SUPER::buildSHAOptionsMap(@_);
    my $configuration = $self->_getConfiguration();
    my $remoteRolesString = $self->_getRemoteRolesString($configuration);

    $result->{SID} = $self->getSID();
    $result->{ROLES} = $self->_getRemoteRolesString($configuration);
    $result->{INSTLOG_DIR} = $gLogDir;

    if($remoteRolesString =~ /$gHostRoleAcceleratorWorker|$gHostRoleAcceleratorStandby/){
        $result->{ASE_USER} = $configuration->getValue('AcceleratorUser');
    }
    $result->{SYSTEM_USER} = $configuration->getValue('SystemUser') || 'SYSTEM';
    if (!$configuration->isSkipped('AutoInitializeServices')) {
        $result->{AIS} = $configuration->getValue('AutoInitializeServices');
    }
    if (!$configuration->isSkipped('TenantUser')) {
        $result->{TENANT_USER} = $configuration->getValue('TenantUser');
    }

    if(!$configuration->isSkipped('OrgManagerUser')) {
        $result->{ORG_MANAGER_USER} = $configuration->getValue('OrgManagerUser');
    }
    my $importContentXS2 = $configuration->getValue('ImportContentXS2');
    if (defined $importContentXS2) {
        my $key         = $configuration->getHostctrlOpt('ImportContentXS2');
        $result->{$key} = ($importContentXS2) ? 'on' : 'off';
    }
    my $timeoutString = buildTimeoutsString($configuration, getHdbmodifyTimeoutValues(), 0); # 0 - without CLI option
    $result->{TIMEOUTS} = $timeoutString if $timeoutString;

    return $result;
}

sub _isPasswordArgsExists {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    return 1 if($configuration->getValue("Password"));
    return 1 if($configuration->getValue("AcceleratorPassword"));
    return 1 if($configuration->getValue("SQLSysPasswd"));
    return 1 if($configuration->getValue("OrgManagerPassword"));
    return 1 if(!$configuration->isUseSAPHostagent() && $configuration->getValue("RootPassword"));

    return 0;
}

sub _buildArgs {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $remoteRolesString = $self->_getRemoteRolesString($configuration);
    my $remoteRolesArgument = "--${ADD_ROLES_OPTION}=$remoteRolesString";
    my $args = ['-b', $remoteRolesArgument, '--read_password_from_stdin=xml',"--instlog_dir=$gLogDir"];

    if(!$configuration->isUseSAPHostagent()){
        my $rootUser = $configuration->getValue('RootUser');
        my $rootUserOpt =$configuration->getOpt('RootUser');
        my $rootUserArgument = "$rootUserOpt=$rootUser";
        push(@{$args}, $rootUserArgument);
    }

    if($remoteRolesString =~ /$gHostRoleAcceleratorWorker|$gHostRoleAcceleratorStandby/){
        my $acceleratorUser = $configuration->getValue('AcceleratorUser');
        my $acceleratorUserOpt = $configuration->getOpt('AcceleratorUser');
        my $acceleratorUserArgument = "$acceleratorUserOpt=$acceleratorUser";
        push(@{$args}, $acceleratorUserArgument);
    }
    my $systemUser = $configuration->getValue('SystemUser') || 'SYSTEM';
    my $systemUserArgument = "--${SYSTEM_USER_OPTION}=$systemUser";
    push(@{$args}, $systemUserArgument);
    if (!$configuration->isSkipped('AutoInitializeServices')) {
        my $autoInitOpt = $configuration->getOpt('AutoInitializeServices');
        my $autoInit = $configuration->getValue('AutoInitializeServices');
        my $autoInitArgument = "${autoInitOpt}=$autoInit";
        push(@{$args}, $autoInitArgument);
    }
    if (!$configuration->isSkipped('TenantUser')) {
        my $tenantUserOpt = $configuration->getOpt('TenantUser');
        my $tenantUser = $configuration->getValue('TenantUser');
        my $tenantUserArgument = "${tenantUserOpt}=$tenantUser";
        push(@{$args}, $tenantUserArgument);
    }
    if(!$configuration->isSkipped('OrgManagerUser')) {
        my $orgManagerUserOpt = $configuration->getOpt('OrgManagerUser');
        my $orgManagerUser = $configuration->getValue('OrgManagerUser');
        my $orgManagerUserArgument = "$orgManagerUserOpt=$orgManagerUser";
        push(@{$args}, $orgManagerUserArgument);
    }
    my $importContentXS2 = $configuration->getValue('ImportContentXS2');
    if (defined($importContentXS2)) {
        my $option = $configuration->getOpt('ImportContentXS2');
        my $value = $importContentXS2 ? 'on' : 'off';
        push(@{$args}, sprintf('%s=%s', $option, $value));
    }
    if ($configuration->hasValue('SkipModifySudoers')) {
        my $value = $configuration->getValue('SkipModifySudoers') ? 'on' : 'off';
        my $option = $configuration->getOpt('SkipModifySudoers');
        push(@{$args}, sprintf('%s=%s', $option, $value));
    }
    my $timeoutString = buildTimeoutsString($configuration, getHdbmodifyTimeoutValues());
    push(@{$args}, $timeoutString) if $timeoutString;

    return $args;
}

sub setHosts {
    my ($self, $hosts) = @_;
    $self->{_hosts} = $hosts;
}

sub getHosts {
    my ($self) = @_;
    return $self->{_hosts};
}

sub setXsControllerHostTask {
    my ($self, $value) = @_;
    $self->{_isXsControllerHostTask} = $value;
}

sub isXsControllerHostTask {
    my ($self) = @_;
    return $self->{_isXsControllerHostTask};
}

sub _createProcessExecutor {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $command = $self->_getExecutable();
    my $args = $self->_buildArgs();
    my $executor;
    if ($self->_isPasswordArgsExists()){
        my $xmlPasswordStream = $configuration->getXmlPasswordStream();
        $executor = new LCM::ProcessExecutor($command, $args, $xmlPasswordStream);
    } else {
        $executor = LCM::ProcessExecutor($command, $args);
    }
    $executor->setOutputHandler( $self->getMsgLst()->getProgressHandler() );
    $executor->setProcessEnvironment (LCM::TraceLoggingEnvironment::PrepareProcessExecutorHdbEnvironment($self->_getExecutable()));
    return $executor;
}

sub initProgressHandler {
    my ($self) = @_;
    my $progressHandler = LCM::Task::TaskProgressHandler::AddHostRolesTaskProgressHandlerFactory::createAddHostRolesTaskProgressHandler($self);
    $self->getMsgLst()->setProgressHandler($progressHandler);
}

sub shouldPersistFinishedStep {
    return 1;
}

1;