package LCM::Task::CommonTask::UpdateComponentListTask;

use warnings;
use strict;
use File::Spec;
use File::stat;
use LCM::Installer;
use LCM::Landscape::LandscapeDescription;
use SDB::Install::Globals qw($gProductNameEngine);
use parent qw(LCM::Task::SHASupportedTask);

# Override
sub getId {
    return 'update_component_list';
}

# Override
sub getName {
    my ($self) = @_;
    my $scenario = $self->_isUpdatingLMStructure() ? 'Update' : 'Creation';
    return "$scenario of Component List";
}

# Override
sub getExecutionName {
    my ($self) = @_;
    my $scenario = $self->_isUpdatingLMStructure() ? 'Updating' : 'Creating';
    return "$scenario Component List";
}

# Override
sub _getNumberOfExpectedOutputLines {
    return 100;
}

sub isUpdateLMStructureTask {
    return 1;
}

# Override
sub _executeInternal {
    my $self = shift();

    my $isUpdate = $self->_isUpdatingLMStructure();
    my $progressMessage = ($isUpdate ? 'Updating ' : 'Creating ') . 'Component List...';
    my $failedMessage = ($isUpdate ? 'Updating ' : 'Creating ') . 'Component List failed';

    my $message = $self->getMsgLst()->addProgressMessage($progressMessage);

    my $landscapeDescription = $self->_getLandscapeDescription($message->getSubMsgLst());
    if (! defined $landscapeDescription) {
        $message->getSubMsgLst()->addProgressMessage('Skipping ' . $self->getName() . " because the PD.XML of the $gProductNameEngine could not be retrieved");
        $message->endMessage();
        $self->getStatus()->_setFinishedState();
        return;
    }
    if(!$landscapeDescription->getErrMsgLst()->isEmpty()){
        $message->endMessage($failedMessage);
        $self->setErrorMessage(sprintf('%s failed', $self->getName()), $landscapeDescription->getErrMsgLst());
        $self->getStatus()->_setErrorState();
        return;
    }
    if(!$landscapeDescription->store()){
        $message->endMessage($failedMessage);
        $self->setErrorMessage(sprintf('%s failed', $self->getName()), $self->getErrMsgLst());
        $self->getStatus()->_setErrorState();
        return;
    }
    $message->endMessage();
    $self->getStatus()->_setFinishedState();
}

sub _getLandscapeDescription {
    my ($self, $msgLst) = @_;
    my $configuration = $self->_getConfiguration();
    my $instance = $configuration->getOwnInstance(1);

    return undef if (!defined $instance || !defined File::stat::stat($instance->getPDXMLPath()));

    my $sid = $configuration->getSID();
    return new LCM::Landscape::LandscapeDescription([$msgLst], $sid);
}

# caching the result because when building
# the Execution Summary Table this is called again
# and the landscapeDescription.xml always exists.
sub _isUpdatingLMStructure {
    my ($self) = @_;
    if (exists $self->{_isUpdatingLMStructure}) {
        return $self->{_isUpdatingLMStructure};
    }
    my $installer = new LCM::Installer();
    my $runtimeDir = $installer->getRuntimeDir();
    my $configuration = $self->_getConfiguration();
    my $sid = $configuration->getSID();
    my $target = $installer->isInstalled()  ? File::Spec->catfile($runtimeDir, '..', '..')
                                            : File::Spec->catfile($configuration->getValue('Target'), $sid);

    $self->{_isUpdatingLMStructure} = (-e File::Spec->catfile($target, 'lm_structure', 'landscapeDescription.xml')) ? 1 : undef;
    return $self->{_isUpdatingLMStructure};
}

1;
