package LCM::Task::CommonTask::UpdateRemoteHostsTask;

use strict;
use parent qw (LCM::Task::SHASupportedTask Exporter);

use SDB::Install::SysVars qw($isWin $path_separator);
use SAPDB::Install::Hostname;
use LCM::Configuration::Hosts::UpdateHosts::UpdateLocalHostConfiguration qw($PARAMETER_TO_OPTION_MAPPING);
use LCM::Task qw($gFlavourProductName);
use SDB::Install::Tools qw(escapeCommandLineArgument);

sub new{
    my($class, $hostconfig) = @_;
    my $self = $class->SUPER::new($hostconfig);
    $self->_initTaskMessages();
    return $self;
}

sub getId {
    return 'update_remote_hosts';
}

sub getName {
	my $self = shift;
    return $self->{_name};
}

sub getExecutionName {
    my $self = shift;
    return $self->{_executionName};
}

sub isUpdateHostTask {
    return 1;
}

sub _initTaskMessages{
    my($self) = @_;
    $self->{_executionName} = $self->_formatMessage("Updating $gFlavourProductName Instance Integration on Remote Hosts");
    $self->{_name} = $self->_formatMessage("Update $gFlavourProductName Instance Integration on Remote Hosts");
    $self->{_taskFailedMessage} = $self->_formatMessage("Update of $gFlavourProductName instance integration on remote hosts failed ");
    $self->{_hostProgressMessageStart} = $self->_formatMessage("Updating $gFlavourProductName instance integration on host '%s'...");
    $self->{_hostProgressMessageSuccess} = $self->_formatMessage("$gFlavourProductName instance integration updated on  host '%s'.");
    $self->{_hostProgressMessageFail} = $self->_formatMessage("Updating $gFlavourProductName instance integration on host '%s' failed!");
    my $configuration = $self->_getConfiguration();
    if($configuration->shallRegisterComponents() && !$configuration->shallRegenerateCertificates() && !$configuration->shallRegenerateCertificates() 
         && !$configuration->shallUpdateSLD() && !$configuration->shallDeployConfigurations()){
            my $componentsRegistration = $configuration->getValue('ComponentsRegistration');
            if($componentsRegistration =~ /^register/){
                $self->{_executionName} = $self->_formatMessage("Registering $gFlavourProductName Components on Remote Hosts");
                $self->{_name} = $self->_formatMessage("Register $gFlavourProductName Components on Remote Hosts");
                $self->{_taskFailedMessage} = $self->_formatMessage("Registering $gFlavourProductName components on remote hosts failed");
                $self->{_hostProgressMessageStart} = $self->_formatMessage("Registering $gFlavourProductName components on host '%s' ...");
                $self->{_hostProgressMessageSuccess} = $self->_formatMessage("$gFlavourProductName components successfully registered on  host '%s'.");
                $self->{_hostProgressMessageFail} = $self->_formatMessage("Registering $gFlavourProductName components on host '%s' failed!");
            } else {
                $self->{_executionName} = $self->_formatMessage("Unregistering $gFlavourProductName Components on Remote Hosts");
                $self->{_name} = $self->_formatMessage("Unregister $gFlavourProductName Components on Remote Hosts");
                $self->{_taskFailedMessage} = $self->_formatMessage("Unregistering $gFlavourProductName components on remote hosts failed");
                $self->{_hostProgressMessageStart} = $self->_formatMessage("Unregistering $gFlavourProductName components on host '%s' ...");
                $self->{_hostProgressMessageSuccess} = $self->_formatMessage("$gFlavourProductName components successfully unregistered on  host '%s'.");
                $self->{_hostProgressMessageFail} = $self->_formatMessage("Unregistering $gFlavourProductName components on host '%s' failed!");
            }
    }
    
    $self->getStatus()->_setActionDoneMessage($self->{_name}." finished successfully");
    $self->getStatus()->_setActionFailedMessage($self->{_taskFailedMessage});
    $self->getStatus()->_setActionAbortedMessage($self->{_name}." aborted");
}

sub _executeInternal {
    my $self = shift;
    my $configuration = $self->_getConfiguration();
    my $remoteHostsObjects = $configuration->{_remoteHosts};
    if(!defined($remoteHostsObjects) || scalar(@{$remoteHostsObjects}) == 0){
        $self->setErrorMessage( "Illegal call: updating remote hosts on a single host system.");
        $self->getStatus()->_setErrorState();
        return;
    }
    my $executionResult = 1;
    my $msg = $self->getMsgLst ()->addProgressMessage ($self->getExecutionName(). "...");
    my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst ()]);
    $self->getMsgLst()->getProgressHandler()->incrementIndentationDepth();

    for my $remoteHostsObject (@{$remoteHostsObjects}){
        my $saveCntxt = $remoteHostsObject->setMsgLstContext ($self->getMsgLstContext());
        if ($configuration->isUseSAPHostagent()) {
            $executionResult = $self->_executeUpdateHostThroughSHA($remoteHostsObject, $configuration);
        } else {
            $executionResult = $self->_executeUpdateHostThroughSSHRoot($remoteHostsObject, $configuration);
        }
        $remoteHostsObject->setMsgLstContext ($saveCntxt);
    }
    if(!$executionResult){
    	$self->getStatus()->_setErrorState();
    	$self->setErrorMessage($self->{_taskFailedMessage},$self->getErrMsgLst());
    } else {
        $self->getStatus()->_setFinishedState();
    }
    $msg->endMessage (undef, $self->getStatus()->getMessage());
    $self->getMsgLst()->getProgressHandler()->decrementIndentationDepth();
    $self->setMsgLstContext($saveCntxt);

    $self->persistStep();
}

sub _getNumberOfExpectedOutputLines{
    return 5;
}

sub _executeUpdateHostThroughSHA {
    my ($self, $remoteHostsObject) = @_;
    my $configuration = $self->_getConfiguration();
    my $password = $configuration->getValue('Password');
    if (defined $password) {
        $remoteHostsObject->useSidadm();
    }
    my $hostctrlOperation = 'hdblcm_update_host_v5';
    my $remoteHostnames = $self->_getFilteredHosts($remoteHostsObject->getHostNames());

    return 1 if(scalar(@{$remoteHostnames}) < 1); # No remote hosts to be updated

    my $hostMapOptionMap = $self->_getHostSpecificSHAOptions($remoteHostnames);
    my $optionsMap = $self->buildSHAOptionsMap();

    my $exitCode = $remoteHostsObject->executeHostctrlParallel($hostctrlOperation, $configuration, undef, ['Password','HostagentPassword'], undef, undef, $self->{_hostProgressMessageStart},
                                                         $self->{_hostProgressMessageSuccess}, $self->{_hostProgressMessageFail}, $optionsMap, $hostMapOptionMap, $remoteHostnames, undef, 0);

    if (!defined $exitCode || ($exitCode != 0)) {
        $self->setErrorMessage( "Updating remote hosts through SAP Host Agent failed.", $remoteHostsObject->getErrMsgLst());
        return 0;
    }
    return 1;
}

sub _getFilteredHosts {
    my ($self, $remoteHosts) = @_;
    my $configuration = $self->_getConfiguration();
    my $ownInstance = $configuration->getOwnInstance(1); # No cache
    my $systemHosts = [ grep { $self->_isHostPartOfSystem($_, $ownInstance) } @{$remoteHosts} ];
    my $specificHosts = $configuration->getValue('SpecificHosts');

    return $systemHosts unless(defined($specificHosts));

    my $filteredHosts = [];
    for my $host (@{$systemHosts}){
        next unless(grep { $_ eq $host } @{$specificHosts});
        push(@{$filteredHosts}, $host);
    }
    return $filteredHosts;
}

sub _isHostPartOfSystem {
    my ($self, $hostName, $ownInstance) = @_;
    my $currentHanaHosts = defined($ownInstance) ? $ownInstance->get_allhosts() : [];
    my $isPartOfSystem = grep { lc($_) eq lc($hostName) } @{$currentHanaHosts};

    return $isPartOfSystem;
}

sub _getHostSpecificSHAOptions {
    my ($self, $hosts) = @_;
    my $configuration = $self->_getConfiguration();
    return undef if($configuration->isSkipped('CertificatesHostmap'));
    
    my $result = {};
    my $certificatesHostmap = $configuration->{params}->{CertificatesHostmap}->{value};
    for my $host(@{$hosts}){
        my $value = $certificatesHostmap->{$host};
        if(length($value)){
            $result->{$host} = { CH => $value };
        }
    }
    return $result;
}

sub buildSHAOptionsMap {
    my ($self) = @_;
    my $config = $self->_createUpdateLocalHostConfiguration();
    return $self->_exportUpdateHostConfigToOptionsMap($config,$self->getParameterToOptionMapping());
}

sub _createUpdateLocalHostConfiguration {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $updateLocalHostConfiguration = new LCM::Configuration::Hosts::UpdateHosts::UpdateLocalHostConfiguration();
    $updateLocalHostConfiguration->setMsgLstContext($self->getMsgLstContext());
    $updateLocalHostConfiguration->setValue('SkipHostagentCalls',$configuration->getValue('SkipHostagentCalls'));
    $updateLocalHostConfiguration->setValue('RemoteExecution',$configuration->getValue('RemoteExecution'));
    $updateLocalHostConfiguration->setValue('ComponentsRegistration',$configuration->getValue('ComponentsRegistration'));
    $updateLocalHostConfiguration->setValue('GenerateCertificates',$configuration->getValue('GenerateCertificates'));
    $updateLocalHostConfiguration->setValue('DeploySHAConfigurations',$configuration->getValue('DeploySHAConfigurations'));
    $updateLocalHostConfiguration->setValue('UpdateComponentListLocation',$configuration->getValue('UpdateComponentListLocation'));
    $updateLocalHostConfiguration->setValue('SkippedComponents',$configuration->getValue('SkippedComponents'));
    $updateLocalHostConfiguration->setValue('SkipModifySudoers',$configuration->getValue('SkipModifySudoers'));
    return $updateLocalHostConfiguration;
}

sub _exportUpdateHostConfigToOptionsMap {
    my ($self, $updateHostConfig,$parameterToOptionMapping) = @_;
    
    my $optionsMap = {};
    my $params = $updateHostConfig->{params};
    for my $param (keys %$params) {
        next if(!exists($parameterToOptionMapping->{$param}) || $param eq 'CertificateHostname');
        my $value = $params->{$param}->{value};
        if (defined $value) {
            my $opt = $parameterToOptionMapping->{$param};
            $optionsMap->{$opt} = $value;
        }
    }
    $optionsMap->{'SID'} = $self->_getConfiguration()->getSID();
    return $optionsMap;
}

sub _executeUpdateHostThroughSSHRoot {
    my ($self, $remoteHostsObject ) = @_;
    my $configuration = $self->_getConfiguration();
    my $hdblcmPath = $self->_getHdblcmExecutable();
    my $remoteHostnames = $self->_getFilteredHosts($remoteHostsObject->getHostNames());

    return 1 if(scalar(@{$remoteHostnames}) < 1); # No remote hosts to be updated

    my $cmdTmplRpl = $self->_getSSHCommandTemplate($remoteHostnames);
    my $args = $self->buildUpdateHostArgumentsArray();
    my $xml  = $configuration->getXmlPasswordStream(['Password']);

    push @$args, '--read_password_from_stdin=xml' if (defined $xml);
# Temporary solution until we implement some sort of strategy, which passes different parameters
# based on the version of the kit and/or the installed syste. Also changed in hdblcm_update_host_v5.conf
    push @$args, '--ignore_unknown_option';

    my $argsStr = join ' ', @$args;
    my $command = join (' ', $hdblcmPath, $argsStr);

    my $exitCode =  $remoteHostsObject->executeParallel( $command, $self->{_hostProgressMessageStart}, $self->{_hostProgressMessageSuccess}, $self->{_hostProgressMessageFail}, $xml, $cmdTmplRpl, $remoteHostnames, undef, 0);

    if ($exitCode != 0) {
        $self->setErrorMessage( undef, $remoteHostsObject->getErrMsgLst());
        return 0;
    }
    return 1;
}

sub _getHdblcmExecutable {
    my ($self) = @_;
    
    my $config = $self->_getConfiguration();
    my $hdblcmExecutable = join($path_separator, $config->getValue('Target'), $config->getValue('SID'), 'hdblcm', 'hdblcm');
    return $hdblcmExecutable;
}

sub _getSSHCommandTemplate {
    my ($self, $remoteHosts) = @_;
    my $configuration = $self->_getConfiguration();
    return undef if($configuration->isSkipped('CertificatesHostmap'));
    
    my $result = [];
    my $certificatesHostmap = $configuration->{params}->{CertificatesHostmap}->{value};
    for my $host(@{$remoteHosts}){
        my ($value, $templateValue) = ($certificatesHostmap->{$host}, '');
        if(length($value)){
            $templateValue = '--certificate_hostname=' . escapeCommandLineArgument($value);
        }
        push(@{$result}, [ $templateValue ]);
    }
    return $result;
}

sub buildUpdateHostArgumentsArray {
    my ($self, $host) = @_;
    my $updateLocalHostCfg = $self->_createUpdateLocalHostConfiguration($host);
    return $self->_exportUpdateHostConfigToArgumentsArray($updateLocalHostCfg);
}

sub _exportUpdateHostConfigToArgumentsArray {
    my ($self, $updateLocalHostConfig) = @_;
    
    my $configuration = $self->_getConfiguration();
    my $args = [];
    push @$args, "--action=update_host";
    push @$args, "--batch";

    my $params = $updateLocalHostConfig->{params};
    for my $param (keys %$params) {
        my $value = $params->{$param}->{value};
        my $opt = $params->{$param}->{opt};

        next if($params->{$param}->{type} =~ /passwd|password/);
        next unless(defined($value));

        if ($params->{$param}->{type} =~ /map/){
            foreach my $paramKey(keys %{$value}){
                my $mapValue = $paramKey."=".$value->{$paramKey};
                $mapValue = escapeCommandLineArgument($mapValue); 
                push @$args, "--$opt=$mapValue";
            }
        } else {
            $value = escapeCommandLineArgument($value); 
            push @$args, "--$opt=$value";
        }
    }
    
    if(!$configuration->isSkipped('CertificatesHostmap')){
        push @$args, "%s";
    }
    
    return $args;
}

sub getParameterToOptionMapping{
	return $PARAMETER_TO_OPTION_MAPPING;
}

sub getSlppLogFileName {
	return 'update_remote_hosts.log';
}

sub shouldBeShownOnSummaryPage {
	return 0;
}

sub shouldPersistFinishedStep {
    return 1;
}

1;
