package LCM::Task::DeployXSAComponentsTask;

use strict;
use parent 'LCM::Task::GenericStackTask::ComponentTask';
use SDB::Install::MsgLst;
use LCM::Utils::ComponentsDependencySort qw(sortComponentsTopologically);
use SDB::Install::Globals qw($gShortProductNameXS2 $gLogDir $gXSASpaceSAP);

my $PROGRESS_MESSAGE = 'Installing XS Advanced Components';

our @EXPORT = qw ($PROGRESS_MESSAGE);

sub new{
    my($class,$configuration,$component) = @_;
    my $self = shift->SUPER::new($configuration);
    $self->_addListeners();
    return $self;
}

sub getId {
    return 'install_xsa_components_task';
}

sub getName {
    return "Install XS Advanced Components";
}

sub getExecutionName {
    return $PROGRESS_MESSAGE;
}

sub getSlppLogFileName {
    return 'xs_install_components.log';
}

sub _getNumberOfExpectedOutputLines {
    my $self = shift();
    my $configuration = $self->_getConfiguration();
    my $componentManager = $configuration->getComponentManager();
    my @xs2Components = grep { $_->isComponentSelected() } @{$componentManager->getXs2ApplicationComponents()};

    return 30 * scalar(@xs2Components);
}

sub _targetXSSpace {
    my ($self, $configuration, $xsClient) = @_;
    if ($configuration->isCockpitOnPlatformAction()) {
        my $spaceGuid = $configuration->getValue('CockpitSpaceGUID');
        my $spaceName = $configuration->getValue('CockpitXSSpace');
        return $spaceGuid ? $xsClient->targetSpaceByGUID($spaceGuid) : $xsClient->targetSpace($spaceName);
    } else {
        return $xsClient->targetSpace($gXSASpaceSAP);
    }
}

sub _executeInternal {
    my ($self) = @_;
    my $message = $self->getMsgLst()->addProgressMessage($self->getExecutionName() . '...');
    my $saveContext = $self->setMsgLstContext([$message->getSubMsgLst()]);
    LCM::ExecutionWarningsObservable->getInstance()->registerListener($self);
    $self->getMsgLst()->getProgressHandler()->incrementIndentationDepth();

    my $isDeploymentNeeded = $self->_isDeploymentNeeded();
    my $rc = $isDeploymentNeeded ? $self->_executeDeployment($message) : 1;
    if (!$isDeploymentNeeded) {
        $self->getMsgLst()->addMessage("All selected xs applications are up-to-date.");
        $message->endMessage(undef, sprintf("%s finished", $self->getName()));
    }

    if ($rc) {
        $self->getStatus()->_setFinishedState();
    } else {
        $self->getStatus()->_setErrorState();
    }

    $self->persistStep();
    $self->getMsgLst()->getProgressHandler()->decrementIndentationDepth();
    $self->setMsgLstContext($saveContext);
    LCM::ExecutionWarningsObservable->getInstance()->unregisterListener($self);
}

sub _executeDeployment {
    my ($self, $message) = @_;
    my $configuration = $self->_getConfiguration();

    my $instance = $configuration->getOwnInstance();
    my $msg = $self->getMsgLst()->addMessage("Performing check if this is a secondary system");
    my $isSecondarySystem = (defined $instance) ? $instance->isSecondarySystem($msg->getSubMsgLst(), $msg->getSubMsgLst())
                                                : undef;
    if (!defined $isSecondarySystem) {
        $self->appendErrorMessage("Failed to determine if this is a secondary system before deployment of $gShortProductNameXS2 applications");
        return 0;
    }
    if ($isSecondarySystem) {
        $self->getMsgLst()->addProgressMessage("Installation of $gShortProductNameXS2 applications is skipped because this is a secondary system");
        return 1;
    }

    my $xsClient = $configuration->getXSClient();
    $xsClient->setMsgLstContext($self->getMsgLstContext());

    if(!$xsClient->login() || !$self->_targetXSSpace($configuration, $xsClient)) {
        LCM::ExecutionWarningsObservable->getInstance()->notifyWarning("Installation of XS Advanced Components failed");
        $message->endMessage(undef, "Failed to target the $gShortProductNameXS2 space.");
        return 0;
    }

    my $failedDeployment = undef;
    my $deployedXs2Components;
    eval{
        ($failedDeployment, $deployedXs2Components) = $self->_deployXs2Components($configuration);
        $self->_generateProductInstallerLogFile($configuration, $xsClient);
    };

    if(!$xsClient->logout()) {
        $self->appendErrorMessage("Failed to logout from $gShortProductNameXS2");
        LCM::ExecutionWarningsObservable->getInstance()->notifyWarning("Failed to logout from $gShortProductNameXS2");
    }

    if($@ || defined($failedDeployment)) {
        my $middle = length($@) > 0 ? '' : ' the following';
        my $suffix = length($@) > 0 ? "$@" : $failedDeployment->getComponentName();
        my $endMessage = sprintf('Installation of%s XS Advanced Components failed: %s', $middle, $suffix);
        LCM::ExecutionWarningsObservable->getInstance()->notifyWarning($endMessage);
        $message->endMessage(undef, $endMessage);

        if ( defined $failedDeployment ){
            my $topologyOrderedXs2Components = $self->getTopologyOrderedXs2Components($configuration, $deployedXs2Components);
            my $deployedXs2Applications = join (',', map { $_->getComponentBatchKey() } @{$topologyOrderedXs2Components});
            $configuration->setDeployedXs2Applications($deployedXs2Applications);
        }

        return 0;
    }

    $configuration->setDeployedXs2Applications();
    $message->endMessage(undef, sprintf("%s finished", $self->getName()));
    return 1;
}

sub _isDeploymentNeeded {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    return 1 if $configuration->getIgnore('check_version');

    my $xsApps = $self->getTopologyOrderedXs2Components($configuration);
    return grep { !$_->isAlreadyDeployed($configuration) } @{$xsApps};
}

# Override
sub persistFinishedStep {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    $configuration->addToPersistedFinishedSteps('xs_components');
}

# Override
sub isSkipped {
    my $self = shift;
    my $configuration = $self->_getConfiguration();
    return 0 if ! defined $configuration;

    return 1 if $self->SUPER::isSkipped();
    return 0 if ! $configuration->can('getPersistedSteps');

    my $persistedSteps = $configuration->getPersistedSteps();
    if( defined $persistedSteps ) {
        return 1 if ( grep( /^xs_components/, split(',', $persistedSteps) ) );
    }

    return 0;
}

sub _generateProductInstallerLogFile {
    my ($self, $configuration, $xsClient) = @_;
    my $logFileName = File::Spec->catfile($gLogDir, $self->getSlppLogFileName());
    my $startedAt = $self->getStartedAt();
    my $message = $self->getMsgLst()->addMessage('Getting Product Installer logs...');
    my $oldContext = $xsClient->resetMsgLstContext();

    if ($xsClient->generateProductInstallerLogs($logFileName, $startedAt)) {
        $self->_setLogLocation($logFileName);
        $self->getMsgLst()->addMessage("Log file written to '$logFileName'");
    } else {
        $self->getMsgLst()->addWarning('Failed to get the Product Installer logs');
    }

    $xsClient->setMsgLstContext($oldContext);
}

sub getTopologyOrderedXs2Components{
    my ($self, $configuration, $xs2ApplicationComponents) = @_;
    my $componentManager = $configuration->getComponentManager();
    my $xs2Components = $xs2ApplicationComponents;
    if( ! defined $xs2Components) {
        $xs2Components = [ grep { $_->isComponentSelected() } @{$componentManager->getXs2ApplicationComponents()} ];
    }

    return sortComponentsTopologically($componentManager, $xs2Components);
}

sub _deployXs2Components {
    my ($self, $configuration) = @_;
    my $errorList = new SDB::Install::MsgLst();
    my $persDeployedXs2Components = $configuration->getDeployedXs2Applications();
    my @deployedXs2Components = ();
    for my $component (@{$self->getTopologyOrderedXs2Components($configuration)}){
        $component->setMsgLstContext([$self->getMsgLst(), $errorList]);
        $component->setProgressHandler($self->getMsgLst()->getProgressHandler());

        my $componentKey = $component->getComponentBatchKey();
        if( $persDeployedXs2Components =~ /$componentKey/){
            push @deployedXs2Components, $component;
            next;
        }
        
        my $executionResult = 1;
        if ($component->isUpdate()) {
            $executionResult = $component->updateComponent($configuration);
        } else {
            $executionResult = $component->installComponent($configuration);
        }
# The indentation depth has been incremented by the Installble/Xs2Appliaction.pm
        $self->getMsgLst()->getProgressHandler()->decrementIndentationDepth();
        if(!$executionResult){
            $self->getMsgLst()->addProgressMessage(sprintf("Installation of %s failed ", $component->getComponentName()));
            my $message = 'Installation of XS Advanced components failed';
            $self->appendErrorMessage($message, $errorList->isEmpty() ? undef : $errorList);
             return ($component, \@deployedXs2Components);
        }
        push @deployedXs2Components, $component;
        my $topologyOrderedXs2Components = $self->getTopologyOrderedXs2Components($configuration, \@deployedXs2Components);
        my $deployedXs2Applications = join (',', map { $_->getComponentBatchKey() } @{$topologyOrderedXs2Components});
        $configuration->setDeployedXs2Applications($deployedXs2Applications);
        $configuration->setStep(1);
    }
    return (undef, \@deployedXs2Components);
}

#implementing required interface by LCM::ExecutionWarningsObservable
sub onWarning {
	my ($self,$message) = @_;
	$self->_addWarningMessage($message);
}

sub isServerComponentTask {
    return 0;
}

sub getComponentKeyName{
    my $self = shift();
    return "XSApplication";
}

sub shouldPersistFinishedStep { return 0; }
sub getLogLocation{ return undef; }
sub getPhase{ return undef; }

1;