package LCM::Task::DownloadComponents::CleanupDownloadDirectoryTask;

use strict;
use File::Spec;
use File::Path;
use parent 'LCM::Task';
use SDB::Install::System qw(loadSSLRequiringPackage);
use LCM::FileUtils qw(listDirectory);
use File::stat;

my $PROGRESS_MESSAGE = "Cleaning Up Directory...";
my $SUCCESS_END_MESSAGE = "Finished cleaning up directory";
my $FAIL_END_MESSAGE = "Cleaning up directory failed";

# Override
sub new {
    my ($class, $targetDownloadFiles) = (shift(), shift());
    my $self = $class->SUPER::new(@_);

    $self->_setTargetDownloadFiles($targetDownloadFiles);

    return $self;
}

# Override
sub getId {
    return 'cleanup_download_dir';
}

# Override
sub getName {
    my $self = shift();
    my $configuration = $self->_getConfiguration();
    my $downloadTarget = $configuration->getValue('DownloadDir');

    return sprintf('Cleanup Download Directory \'%s\'', $downloadTarget);
}

# Override
sub getExecutionName {
    my $self = shift();
    my $configuration = $self->_getConfiguration();
    my $downloadTarget = $configuration->getValue('DownloadDir');

    return sprintf('Cleaning Up Download Directory \'%s\'', $downloadTarget);
}

# Override
sub _getNumberOfExpectedOutputLines{
    return 10;
}

# Override
sub _executeInternal {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $message = $self->getMsgLst()->addProgressMessage($PROGRESS_MESSAGE);
    my $saveContext = $self->setMsgLstContext([$message->getSubMsgLst()]);
    my $isSuccess = $self->_cleanUpDownloadDirectory();
    my $endMessage = $isSuccess ?  $SUCCESS_END_MESSAGE : $FAIL_END_MESSAGE;

    if($isSuccess){
        $self->getStatus()->_setFinishedState();
    } else {
        $self->getStatus()->_setErrorState();
        $self->setErrorMessage($FAIL_END_MESSAGE, $self->getErrMsgLst());
    }

    $message ->endMessage(undef, $endMessage);
    $self->setMsgLstContext($saveContext);
}

sub _setTargetDownloadFiles {
    my ($self, $targetDownloadFiles) = @_;
    $self->{_targetDownloadFiles} = $targetDownloadFiles;
}

sub _getTargetDownloadFiles {
    my ($self) = @_;
    return $self->{_targetDownloadFiles};
}

sub _cleanUpDownloadDirectory {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $downloadTarget = $configuration->getValue('DownloadDir');
    my $fileList = listDirectory($downloadTarget, $self->getErrMsgLst());

    return undef if(!defined($fileList));

    my @fileNameList = grep { !$self->_shallKeepFile($downloadTarget, $_) } @{$fileList};
    my @filePathList = map { File::Spec->catfile($downloadTarget, $_) } @fileNameList;

    for my $filePath (@filePathList){
        my $fileStatObject = File::stat::stat($filePath);
        next if !$fileStatObject;

        my $rc = (-d $fileStatObject) ? $self->_deleteDir($filePath) : $self->_deleteFile($filePath);
        return undef if !$rc;
    }
    $self->getMsgLst()->addMessage('Unnecessary files deleted.');
    return 1;
}

sub _deleteFile {
    my ($self, $path) = @_;
    my $msg = $self->getMsgLst()->addMessage("Deleting file '$path'...");
    if(!unlink($path)){
        my $saveContext = $self->setMsgLstContext([$msg->getSubMsgLst()]);
        $self->getMsgLst()->addError("Deleting file '$path' failed: $!");
        $self->setMsgLstContext($saveContext);
        return 0;
    }
    return 1;
}

sub _deleteDir {
    my ($self, $path) = @_;
    my $msg = $self->getMsgLst()->addMessage("Deleting directory '$path'...");
    if (! File::Path::remove_tree($path, {error => \my $err})) {
        my $saveContext = $self->setMsgLstContext([$msg->getSubMsgLst()]);
        my $failedToDeleteMsg = $self->getMsgLst()->addError("Deleting directory '$path' failed.");
        $self->setMsgLstContext([$failedToDeleteMsg->getSubMsgLst()]);
        for my $errorEntry (@$err) {
            my ($file, $errMsg) = %$errorEntry;
            if ($file eq '') {
                $self->getMsgLst()->addError("Error: $errMsg.");
            } else {
                $self->getMsgLst()->addError("Problem unlinking '$file': $errMsg.");
            }
        }
        $self->setMsgLstContext($saveContext);
        return 0;
    }
    return 1;
}

sub _shallKeepFile {
    my ($self, $downloadDir, $fileName) = @_;
    my $filesToDownload = $self->_getTargetDownloadFiles();
    my $fileNameStripped = ($fileName =~ s/\.(PART|META)$//r); # strip .PART and .META

    my $fileNameMeta = sprintf('%s.META', $fileNameStripped);
    my $filePathMeta = File::Spec->catfile($downloadDir, $fileNameMeta);
    my $metaFileObject = File::stat::stat($filePathMeta);

    if($metaFileObject && -r $metaFileObject){
        for my $fileToDownload (@{$filesToDownload}){
            my $targetFileMeta = sprintf('%s.META', $self->_getTargetArchiveName($fileToDownload));

            if($targetFileMeta eq $fileNameMeta){
                $self->getMsgLst()->addMessage("Deletion of file '$fileName' skipped. File will be used to resume the download of archive '$fileNameStripped'");
                return 1;
            }
        }
    }
    return ($fileName =~ /^\.{1,2}$/ ) ? 1 : 0;
}

# Important - this sub must always match the implementation of
# sub _getArchiveName in DownloadSingleComponentTask.pm
sub _getTargetArchiveName {
    my ($self, $fileToDownload) = @_;
    my $fileName = $fileToDownload->{'fileName'};
    return $fileName ? $fileName : 'unknown';
}

1;