package LCM::Task::GenericStackTask::ComponentTasklistFactory::ComponentTasklistFactoryUpdate;

use strict;
use parent 'LCM::Task::GenericStackTask::ComponentTasklistFactory::ComponentTasklistFactoryBase';
use experimental qw (smartmatch);
use LCM::Component qw(PREPARE_PHASE ONLINE_PHASE OFFLINE_PHASE CONFIGURE_PHASE SYSTEM_ONLINE_PHASE ALL);
use LCM::Task::StartStopHostsTask::DisableAndStopSpecificServicesTask;
use LCM::Task::StartStopHostsTask::StopSystemTask;
use LCM::Task::GenericStackTask::DisableAutoFailoverTask;
use LCM::Task::GenericStackTask::EnableAutoFailoverTask;
use LCM::Task::StartStopHostsTask::StopSpecificHostsTask;
use LCM::Task::ConvertToMultiDBTask::DisableXSAServicesTask;
use LCM::Task::GenericStackTask::PostOnlinePhaseTaskGenerator;
use LCM::Task::GenericStackTask::PostSystemOnlinePhaseTaskGenerator;
use LCM::Utils::CommonUtils;
use LCM::Hosts::HostsDetector;
use LCM::Installer;
use SDB::Install::Globals qw($gKeynameXS2 $gKeynameEngine $gKeynameClient);


sub new {
    my $class = shift();
    my $self = $class->SUPER::new(@_);
    $self->addPostPhaseTaskGenerator(ONLINE_PHASE, LCM::Task::GenericStackTask::PostOnlinePhaseTaskGenerator->new());
    $self->addPostPhaseTaskGenerator(SYSTEM_ONLINE_PHASE, LCM::Task::GenericStackTask::PostSystemOnlinePhaseTaskGenerator->new());

    return $self;
}

sub createComponentTasklist {
    my ($self) = @_;
    my $configuration = $self->getConfiguration();
    my $components = $configuration->getComponentManager()->getSelectedComponents();
    my $isPrepareUpdate = $configuration->isPrepareUpdate();

    if (!$configuration->isOptimizedExecution() && !$isPrepareUpdate) {
        return $self->_createDefaultTasklist($components);
    }

    my $phase = $configuration->getPhase();
    my $isResumeUpdateIgnored = $configuration->getIgnore('check_pending_upgrade');
    my $isPendingUpdate = $configuration->isPendingUpdate();
    my $isPhasedResumeUpdate = $configuration->isOptimizedExecution() && $isPendingUpdate && !$isResumeUpdateIgnored;
    my $isIgnoredPhasedResumeUpdate = $configuration->isOptimizedExecution() && $isResumeUpdateIgnored;

    if (!$isPrepareUpdate && ($isPhasedResumeUpdate || $isIgnoredPhasedResumeUpdate)) {
        $phase = ALL;
    }

    if($phase eq PREPARE_PHASE){
        return $self->_createPrepareTasklist($components);
    } elsif($phase eq OFFLINE_PHASE){
        return $self->_createOfflineTasklist($components);
    } elsif($phase eq CONFIGURE_PHASE) {
        return $self->_createConfigureTasklist($components);
    } elsif($phase eq ONLINE_PHASE){
        return $self->_createOnlineTasklist($components);
    } elsif($phase eq SYSTEM_ONLINE_PHASE){
        return $self->_createSystemOnlineTasklist($components);
    }

    return $self->_createPhasedTasklist($components);
}

sub _createDefaultTasklist {
    my ($self, $components) = @_;
    $self->getConfiguration()->setWillExecuteAllComponentsInAllPhases(1);

    return $self->SUPER::_createDefaultTasklist($components);
}

sub _createPrepareTasklist {
    my ($self, $components) = @_;
    $self->getConfiguration()->setWillExecuteAllComponentsInAllPhases(0);

    return $self->SUPER::_createPrepareTasklist($components);
}

sub _createOfflineTasklist {
    my ($self, $components) = @_;
    $self->getConfiguration()->setWillExecuteAllComponentsInAllPhases(0);

    return $self->SUPER::_createOfflineTasklist($components);
}

sub _createConfigureTasklist {
    my ($self, $components) = @_;
    $self->getConfiguration()->setWillExecuteAllComponentsInAllPhases(0);

    return $self->SUPER::_createConfigureTasklist($components);
}

sub _createOnlineTasklist {
    my ($self, $components) = @_;
    $self->getConfiguration()->setWillExecuteAllComponentsInAllPhases(1);

    return $self->SUPER::_createOnlineTasklist($components);
}

sub _createOnlinePrerequisiteTasks {
    my ($self, $components) = @_;
    my $tasks = $self->SUPER::_createOnlinePrerequisiteTasks($components);

    if (@{$tasks}) {
        my $configuration = $self->getConfiguration();
        my $mcm = $configuration->getComponentManager();
        my $scm = $configuration->getSystemComponentManager();
        my $isSystemReplication = LCM::Utils::CommonUtils::isSystemReplication($configuration);
        my $server = $scm->getComponentByKeyName($gKeynameEngine);
        my $isConvertToMDCStarted = $server ? $server->isConvertToMDCStarted() : 0;
        my $isConvertToMDC = $configuration->isConvertToMultiDbRequired() || $isConvertToMDCStarted;
        my $isXs2Selected = $mcm->isComponentSelected($gKeynameXS2);
        my $isXs2Available = $scm->isComponentAvailable($gKeynameXS2);
        if(!$isSystemReplication && $isConvertToMDC && $isXs2Available && !$isXs2Selected) {
            push(@{$tasks}, LCM::Task::ConvertToMultiDBTask::DisableXSAServicesTask->new($configuration));
        }
    }

    return $tasks;
}

sub _createSystemOnlineTasklist {
    my ($self, $components) = @_;
    $self->getConfiguration()->setWillExecuteAllComponentsInAllPhases(1);

    return $self->SUPER::_createSystemOnlineTasklist($components);
}

sub _createSystemOnlinePrerequisiteTasks {
    my ($self, $components) = @_;
    my $configuration = $self->getConfiguration();
    my $tasks = $self->SUPER::_createSystemOnlinePrerequisiteTasks($components);

    my $isLeftDisabledFailoverFromPreviousRun = $configuration->getOwnInstance()->isAutoFailoverDisabled();
    if ($configuration->{disableAutoFailoverTaskAdded} || $isLeftDisabledFailoverFromPreviousRun) {
        push(@{$tasks}, LCM::Task::GenericStackTask::EnableAutoFailoverTask->new($configuration));
    }

    return $tasks;
}

sub _getHostsWithRoles {
    my ($self, $roles) = @_;
    my $configuration = $self->getConfiguration();
    my $hostDetector = LCM::Hosts::HostsDetector->new(LCM::Installer->new(), $configuration->getMsgLstContext(), $configuration->getOwnInstance());

    if ($hostDetector->hasParsingErrors()) {
        $configuration->PushError("Cannot detect hosts.");
        return [];
    }

    return $hostDetector->getHostsWithRoles($roles);
}

# Start everything except hosts that have a component role that requires
# restart of the instance
sub _getInstancesToStartBeforeOnlinePhase {
    my ($self, $components) = @_;
    my $configuration = $self->getConfiguration();
    my $allHosts = $configuration->getSystemHosts();
    if ($configuration->isConvertToMultiDbRequired()) {
        return $allHosts;
    }

    my $instanceRestartHosts = $self->_getInstancesToStart($components);
    my @result = grep { !($_ ~~ @{$instanceRestartHosts}) } @{$allHosts};
    return \@result;
}

sub _recomputePhases {
    my ($self, $components) = @_;
    return undef if (!$self->SUPER::_recomputePhases($components));
    return $self->_recomputePhasesForNoStartUpdate();
}

sub _recomputePhasesForNoStartUpdate {
    my ($self) = @_;
    my $configuration  = $self->getConfiguration();
    my $mcm            = $configuration->getComponentManager();
    my $isServerUpdate = $mcm->isHDBServerComponentSelected();
    my $noStart        = $configuration->isServerNoStart();

    if (!$isServerUpdate || !$noStart) {
        return 1;
    }
    my $client = $mcm->getComponentByKeyName($gKeynameClient);

    if (defined $client && $client->isComponentSelected()) {
        $client->setPhase(OFFLINE_PHASE);
    }
    return 1;
}

1;