package LCM::Utils::ComponentsDependencySort;

use strict;

use base qw ( Exporter );

our @EXPORT_OK = qw ( sortComponentsTopologically constructComponentDependencyGraph);

# Clasical implementation of Kahn's topo sort algorithm.
# Original order of the components is preserved as much as possible.
#   <A, B> is a an edge in the ORIGINAL graph if A is dependant of B
# The dependency graph is reverse of the original graph so
#   <A, B> is edge in the dependency graph if <B, A> is edge in the original
# We assume that cyclic dependency between components are impossible
sub sortComponentsTopologically {
    my ($componentManager, $components, $isReversed) = @_;
    my $orderIndex = 0;
    my %orderMap = map { $_->getComponentKeyName() => $orderIndex++ } @{$components};
    my $graph = constructComponentDependencyGraph($componentManager, $components, $isReversed);
    my @withoutDependants = grep { scalar(@{$graph->{$_->getComponentKeyName()}}) == 0 } @{$components};
    my @withDependants = grep { scalar(@{$graph->{$_->getComponentKeyName()}}) > 0 } @{$components};
    my $sortedComponents = [];

    while(scalar(@withoutDependants)){
        my $componentWithoutDependants = shift(@withoutDependants);
        push(@{$sortedComponents}, $componentWithoutDependants);

# remove edges starting from '$componentWithoutDependants'
        for my $componentWithDependants (@withDependants) {
            my $keynameWithout = $componentWithoutDependants->getComponentKeyName();
            my $keynameWith = $componentWithDependants->getComponentKeyName();
            $graph->{$keynameWith} = [ grep { $keynameWithout ne $_ } @{$graph->{$keynameWith}} ];
        }
# add all components with 0 incoming edges to the list of components for addition
        my @forAddition = grep { scalar(@{$graph->{$_->getComponentKeyName()}}) == 0 } @withDependants;
        @withDependants = grep { scalar(@{$graph->{$_->getComponentKeyName()}}) > 0 } @withDependants;

        if(scalar(@forAddition) > 0){
            push(@withoutDependants, @forAddition);
            @withoutDependants = sort { $orderMap{$a->getComponentKeyName()} <=> $orderMap{$b->getComponentKeyName()} } @withoutDependants;
        }
    }
    return $sortedComponents;
}

sub constructComponentDependencyGraph {
    my ($componentManager, $components, $isReversed) = @_;
    my %dependencyGraph = map { $_->getComponentKeyName() => [] } @{$components};
    for my $component (@{$components}) {
        for my $candidateDependant (@{$components}) {
            next if(! $component->isDependencyForComponent($candidateDependant));
            if($isReversed){
                push(@{$dependencyGraph{$component->getComponentKeyName()}}, $candidateDependant->getComponentKeyName());
            }else{
                push(@{$dependencyGraph{$candidateDependant->getComponentKeyName()}}, $component->getComponentKeyName());
            }
        }
    }
    return \%dependencyGraph;
}

1;
