package LCM::Utils::RemoveRoles;

use strict;
use parent qw(Exporter);

our @EXPORT_OK = qw(
					IsRemovingRolesFromHost
					IsRemovingLocalRole
					IsRemovingRemoteRole
					IsRemovingRole
					IsRemovingRoleFromHost
					canRemoveHostRole
					shouldCheckRoleRemovalStatus);

sub IsRemovingRole {
	my ($configuration, $hostRole) = @_;
	my $rolesForRemove = $configuration->getValue('RemoveRoles');

	return 0 if($configuration->isSkipped('RemoveRoles'));
	return 0 if(!$configuration->hasValue('RemoveRoles'));

	for my $host (keys(%{$rolesForRemove})){
		my @roles = split(',', $rolesForRemove->{$host});
		if(scalar(grep { $_ =~ /^$hostRole$/} @roles)){
			return 1;
		}
	}
	return 0;
}

sub IsRemovingRoleFromHost {
	my ($configuration, $host, $hostRole) = @_;
	my $rolesForRemove = $configuration->getValue('RemoveRoles');

	return 0 if($configuration->isSkipped('RemoveRoles'));
	return 0 if(!$configuration->hasValue('RemoveRoles'));
	return 0 if(!defined($rolesForRemove->{$host}));

	my @roles = split(',', $rolesForRemove->{$host});
	my $isRemoving = scalar(grep { $_ =~ /^$hostRole$/} @roles);
	return $isRemoving ? 1 : 0;
}

sub IsRemovingRolesFromHost {
	my ($configuration, $host) = @_;
	my $removeRolesMap = $configuration->getValue('RemoveRoles');

	return 0 if($configuration->isSkipped('RemoveRoles'));
	return 0 if(!$configuration->hasValue('RemoveRoles'));
	return 0 if(!defined($removeRolesMap->{$host}));
	return 0 if(!length($removeRolesMap->{$host}));
	return 1;
}

sub IsRemovingLocalRole {
	my ($configuration) = @_;
	my $trexInstance = $configuration->getOwnInstance();
	my $localHost = $trexInstance->get_host();

	return IsRemovingRolesFromHost($configuration, $localHost);
}

sub IsRemovingRemoteRole {
	my ($configuration) = @_;

	return 0 if(!$configuration->isDistributedSystem());

	my $trexInstance = $configuration->getOwnInstance();
	my $remoteHosts = $trexInstance->get_hosts();

	for my $host (@{$remoteHosts}){
		return 1 if(IsRemovingRolesFromHost($configuration, $host));
	}
	return 0;
}

sub canRemoveHostRole {
    my ($configuration, $host, $role) = @_;
    my $instance = $configuration->getOwnInstance();
    my $saveCtx = $instance->setMsgLstContext([ SDB::Install::MsgLst->new(), SDB::Install::MsgLst->new() ]);
    my $canSafelyRemoveRole = $instance->getHostRoleRemoveStatus($host, $role);
    my $rc = 1;
    $configuration->getMsgLst()->appendMsgLst($instance->getMsgLst());

    if (! $canSafelyRemoveRole) {
        $configuration->appendErrorMessage("Cannot remove host role '$role'", $instance->getErrMsgLst());
        $rc = 0;
    }

    $instance->setMsgLstContext($saveCtx);
    return $rc;
}

sub shouldCheckRoleRemovalStatus {
    my($configuration, $role) = @_;
    my $autoInitServices = $configuration->getValue('AutoInitializeServices');
    my $isSingleTenant = $configuration->isSystemInCompatibilityMode();
    my $isRoleServicePersistenceless = ($role =~ /(streaming|rdsync)/);
    my $shouldCheck = !($autoInitServices && $isSingleTenant && $isRoleServicePersistenceless);
    return $shouldCheck;
}

1;