package LCM::XSAClient;

use strict;
use parent 'SDB::Install::Base';
use LCM::ProcessExecutor;
use LCM::Utils::PrintToFileHandler;
use SDB::Install::NewDBUser;
use SDB::Install::System qw(isAdmin);
use SDB::Install::Globals qw($gXSASpaceSAP);
use File::Spec;
use File::Basename;
use IO::File;

my $info = <<EOF;
=========== How to read the log file ===============

<date> <time> [APP/1-1] ERR => Means that the info was written over the err stream which is expected procedure. Does not indicate an error has occurred.

<date> <time> [APP/1-1] ERR #2.0#2018 08 17 10:26:55.267#+02:00#ERROR ... => Indicates that an error has occurred.

====================================================
EOF

sub new {
    my ($class, $instPath) = @_;
    my $self = $class->SUPER::new();
    $self->setInstallationPath($instPath);
    return $self;
}

# --- Trying to implement kind of builder inside the class. The add* methods are part of the builder api ---

sub addUsername {
    my ($self, $username) = @_;
    $self->setUsername($username);
    return $self;
}

sub addPassword {
    my ($self, $password) = @_;
    $self->setPassword($password);
    return $self;
}

sub addOrganisation {
    my ($self, $orgname) = @_;
    $self->setOrganisation($orgname);
    return $self;
}

sub addSpace {
    my ($self, $space) = @_;
    $self->setSpace($space);
    return $self;
}

sub addApiUrl {
    my ($self, $url) = @_;
    $self->setApiUrl($url);
    return $self;
}

sub executeAs {
    my ($self, $user) = @_;
    $self->setExecutionOSUser($user);
    return $self;
}

# ------------------------------------------------------------------------------------------------------------

sub setUsername {
    my ($self, $username) = @_;
    $self->{username} = $username;
}

sub getUsername {
    my ($self) = @_;
    return $self->{username};
}

sub setPassword {
    my ($self, $password) = @_;
    $self->{password} = $password;
}

sub getPassword {
    my ($self) = @_;
    return $self->{password};
}

sub setOrganisation {
    my ($self, $orgname) = @_;
    $self->{orgname} = $orgname;
}

sub getOrganisation {
    my ($self) = @_;
    return $self->{orgname};
}

sub setSpace {
    my ($self, $space) = @_;
    $self->{space} = $space;
}

sub getSpace {
    my ($self) = @_;
    return $self->{space} // $gXSASpaceSAP;
}

sub setApiUrl {
    my ($self, $apiUrl) = @_;
    $self->{apiUrl} = $apiUrl;
}

sub getApiUrl {
    my ($self) = @_;
    return $self->{apiUrl};
}

sub setInstallationPath {
    my ($self, $clientPath) = @_;
    $self->{instpath} = $clientPath;
}

sub getInstallationPath {
    my ($self) = @_;
    return $self->{instpath};
}

sub setIsLogged {
    my ($self, $isLogged) = @_;
    $self->{isLogged} = $isLogged;
}

sub isLogged {
    my ($self) = @_;
    return $self->{isLogged};
}

sub setOutputLines {
    my ($self, $buffer) = @_;
    $self->{outputLines} = $buffer;
}

sub getOutputLines {
    my ($self) = @_;
    return $self->{outputLines} // [];
}

sub setExecutionOSUser {
    my ($self, $user) = @_;
    $self->{executionUser} = $user;
}

sub getExecutionOSUser {
    my ($self) = @_;
    return $self->{executionUser};
}

# ------------------------------------------------------------------------------------------------------------

sub login {
    my ($self, $redoLogin) = @_;
    if ($self->isLogged && !$redoLogin) {
        return 1;
    }

    my $apiUrl = $self->getApiUrl();
    my $username = $self->getUsername();
    my $password = $self->getPassword();
    my $orgname = $self->getOrganisation();
    my $space = $self->getSpace();
    my $arguments = ['-a', $apiUrl,'-u', $username, '--stdin', '-o', $orgname, '-s', $space];

    if($apiUrl =~ /^https/){
        my $certificateFilePath = $self->_getCertificatesPath();
        push(@{$arguments}, '--cacert');
        push(@{$arguments}, $certificateFilePath);
    }

    my $rc = $self->_executeXSCommand('login', $arguments, [$password]);
    if ($rc) {
        $self->setIsLogged(1);
    }
    return $rc;
}

sub logout {
    my ($self) = @_;
    if (!$self->isLogged()) {
        return 1;
    }

    my $rc = $self->_executeXSCommand('logout');
    if ($rc) {
        $self->setIsLogged(0);
    }
    return $rc;
}

sub listTenantDatabases {
    my ($self) = @_;
    return $self->_executeXSCommand('tenant-databases');
}

sub spaceExists {
    my ($self, $space) = @_;
    my $args = [ $space ];
    return $self->_executeXSCommand('space', $args);
}

sub retrieveSpaceGUID {
    my ($self, $space) = @_;
    my $args = [ $space, '--guid' ];
    if (!$self->_executeXSCommand('space', $args)) {
        return undef;
    }

    my @lines = grep /\S/, @{$self->getOutputLines()};
# Take the last element because of the multi-line env variables problem.
# See the implementation of retrieveXsAdminLoginParam in XS2Configuration.pm
    return $lines[-1];
}

sub retrieveSpaceNameByGUID {
    my ($self, $guid) = @_;
    if (!$self->targetSpaceByGUID($guid)) {
        $self->getErrMsgLst()->addError("Failed to target space with guid '$guid'");
        return undef;
    }

    my $output = $self->getOutputLines();
    for my $line (@{$output}) {
        my ($space) = $line =~ /^\s*Space\s*:\s*(\S+)\s*$/i;
        return $space if $space;
    }

    return undef;
}

sub createSpace {
    my ($self, $space, $tenant) = @_;
    my $args = [ $space ];
    if ($tenant) {
        push @{$args}, '--tenant';
        push @{$args}, $tenant;
    }

    return $self->_executeXSCommand('create-space', $args);
}

sub stopSpace {
    my ($self, $space) = @_;
    my $args = [ '-s', $space ];
    return $self->_executeXSCommand('stop', $args);
}

sub deleteSpace {
    my ($self, $space) = @_;
    my $args = [ '-f', $space ]; # -f => Force, no confirmation
    return $self->_executeXSCommand('delete-space', $args);
}

sub enableTenantDB {
    my ($self,
        $tenantDB,
        $tenantUsername,
        $tenantPassword,
        $systemUsername,
        $systemPasswd)
    = @_;
# TODO: Add the tenant database name in the $args array after XSA implement this parameter
    my $args = [ $tenantDB, '-u', $systemUsername, '--stdin' ];
    my $stdin = [ $tenantPassword, $systemPasswd ];
    return $self->_executeXSCommand('enable-tenant-database', $args, $stdin);
}

sub deployApplication {
    my ($self,
        $appPath,
        $shouldUseNoStart,
        $extensionDescriptors)
    = @_;

    my $args = [ $appPath, '-o', 'ALLOW_SC_SAME_VERSION' ];
    if ($extensionDescriptors) {
        push @{$args}, '-e';
        push @{$args}, $extensionDescriptors;
    }
    push @{$args}, '--no-start' if $shouldUseNoStart;
    push @{$args}, '--ignore-lock';
    return $self->_executeXSCommand('install', $args);
}

sub targetSpace {
    my ($self, $space) = @_;
    my $args = [ '-s', $space ];
    return $self->_executeXSCommand('target', $args);
}

sub targetSpaceByGUID {
    my ($self, $guid) = @_;
    my $args = [ '-g', $guid ];
    return $self->_executeXSCommand('target', $args);
}

sub generateProductInstallerLogs {
    my ($self, $logFilename, $startedAt) = @_;
    if (!$self->targetSpace($gXSASpaceSAP)) {
        $self->getErrMsgLst()->addError("Failed to target space '$gXSASpaceSAP'.");
        return undef;
    }

    my ($sec, $min, $hour, $day, $month, $year) = localtime($startedAt);
    my $startDate = sprintf("%d-%02d-%02d %02d:%02d:%02d", $year + 1900, $month + 1, $day, $hour, $min, $sec);
    my $args = [ 'product-installer', '--since', $startDate ];
    my $fileHandle = IO::File->new(">$logFilename");
    if (!$fileHandle) {
        $self->getErrMsgLst()->addError("Failed to open file '$logFilename': $!");
        return undef;
    }

    my $outputHandler = LCM::Utils::PrintToFileHandler->new($fileHandle);

    $outputHandler->addLine($self->getInfo());
    my $rc = $self->_executeXSCommand('logs', $args, undef, $outputHandler);
    $outputHandler->addLine($self->getInfo());

    $fileHandle->close();
    return $rc;
}

sub getInfo {
    return $info;
}

sub _getCertificatesPath {
    my ($self) = @_;
    my $instPath = $self->getInstallationPath();
    return File::Spec->catfile($instPath, 'controller_data', 'controller', 'ssl-pub', 'router', 'default.root.crt.pem');
}

sub _getExecutable {
    my ($self) = @_;
    my $instPath = $self->getInstallationPath();
    return File::Spec->catfile($instPath, 'bin', 'xs');
}

sub _createXsProcessExecutor {
    my ($self, $command, $args, $stdin) = @_;
    my $executable = $self->_getExecutable();
    my ($uid, $gid) = (undef, undef);
    $args //= [];

    if (isAdmin()) {
        my $user = $self->getExecutionOSUser();
        ($uid, $gid) = ($user->uid(), $user->gid());
    }

    unshift @{$args}, $command if(defined $command); # Add the specific xs command to the args
    return LCM::ProcessExecutor->new($executable, $args, $stdin, dirname($executable), undef, $uid, $gid);
}

sub _executeXSCommand {
    my ($self, $command, $args, $stdin, $outputHandler) = @_;
    my $processExecutor = $self->_createXsProcessExecutor($command, $args, $stdin);
    $processExecutor->setMsgLstContext($self->getMsgLstContext());

    if (defined($outputHandler)) {
        $processExecutor->setOutputHandler($outputHandler);
    } else {
        $processExecutor->setOutputHandler($self->getMsgLst()->getProgressHandler());
    }

    my $rc = $processExecutor->executeProgram();
    $self->setOutputLines($processExecutor->getOutputLines());
    return (!defined $rc || $rc) ? 0 : 1;
}

1;
