
#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: base class of installation applications

package SDB::Install::App;

use SDB::Install::SysInfo;
use SDB::Install::Installer;
use Getopt::Long;
use SDB::Install::Log;
use SDB::Install::Globals qw (gPrepareLogPath
                              $gLogDir
                              $gLogTime
                              $gProductName
                              $gProductNameInstaller
                              $gSapsysGroupName
                              $gTmpDir);
use SDB::Install::Tools qw (printSortedTableToArrayOfLines print_callstack callstack getFilenameTimeStamp);
use SDB::Install::SAPSystem;
use SDB::Install::Configuration::Upgrade;
use File::Spec;
use Carp qw(longmess);
use File::Basename qw (dirname basename);
use SDB::Install::SysVars qw ($isWin $path_separator $isLinux);
use LCM::DevelopmentTrace;
use strict;

use base qw (SDB::Install::Base);

our $sysinfo;
our $installer = SDB::Install::Installer->new ();

our $USAGE_COL1_LIMIT = 30;
our $USAGE_COL2_LIMIT = 15;
our $USAGE_COL3_LIMIT = 85;

our $PASSWORD_COL1_LIMIT = 30;
our $PASSWORD_COL2_LIMIT = 80;

sub signalHandlerDie {
    my ($sig) = @_;
    die ("__SIG${sig}__");
}

our @signals = qw (INT PIPE ABRT EMT FPE QUIT SYS TRAP TERM);

sub setSignalHandler ($$);

if (!$isWin){
    binmode(STDOUT, ":utf8");
}
if ($isLinux){
    require POSIX;
    *setSignalHandler = \&setSignalHandlerPosix;
}
else{
    *setSignalHandler = \&setSignalHandlerNonPosix;
}

sub setSignalHandlerPosix ($$){
    my $sigset = new POSIX::SigSet ();
    $sigset->emptyset;
    my $action = new POSIX::SigAction ($_[1],
                                     $sigset,
                                     &POSIX::SA_SIGINFO);
    no strict 'refs';
    POSIX::sigaction (&{"POSIX::SIG$_[0]"}, $action);
    use strict 'refs';
}

sub setSignalHandlerNonPosix ($$){
    $SIG{$_[0]} = $_[1];
}


sub getSignalInfoText{
    my $siginfo = $_[0]->{siginfo};
    if (!defined $siginfo){
        return '';
    }

    my $text;
    if ($siginfo->{code} < 1){
        $text = "signal $siginfo->{signo} was sent by a user process";
        if (defined $siginfo->{pid}){
            $text .= sprintf (" (pid=%d, uid=%d)", $siginfo->{pid}, $siginfo->{uid});
        }
    }
    else{
        $text = "signal $siginfo->{signo} was sent by the system";
    }
    return $text;
}


sub InitLimits{
    my ($self) = @_;

    if (!$isLinux){
        return 1;
    }

    if ($>){
        return 1;
    }

    my $msglst = $self->getMsgLst ();

    my $msg = $msglst->AddMessage ("Checking process limits");
    $msglst = $msg->getSubMsgLst ();
    require SAPDB::Install::System::Unix;
    import SAPDB::Install::System::Unix qw (getrlimit setrlimit RLIMIT_NPROC RLIM_INFINITY);
    my ($rlimit_rc,$soft,$hard) = getrlimit (RLIMIT_NPROC ());
    if ($rlimit_rc){
        $msglst->addError ("getrlimit (RLIMIT_NPROC) failed: $!");
        return undef;
    }
    else{
        my $str_hard = $hard;
        if ($hard == RLIM_INFINITY()){
            $str_hard = 'unlimited';
        }
        my $str_soft = $soft;

        if ($soft == RLIM_INFINITY()){
            $str_soft = 'unlimited';
        }

        $msglst->addMessage ("nproc limit: hard = $str_hard, soft = $str_soft");
        if ($hard == RLIM_INFINITY() && $soft == RLIM_INFINITY()){
            return 1;
        }
        $msglst->addMessage ("Setting limit 'nproc' to 'unlimited'");
        $rlimit_rc = setrlimit (RLIMIT_NPROC(), RLIM_INFINITY(), RLIM_INFINITY());
        if ($rlimit_rc){
            $msglst->addError ("setrlimit (RLIMIT_NPROC) failed: $!");
            return undef;
        }
    }
    return 1;
}


sub InitSignalHandler{
    my ($self) = @_;
    my $msg = $self->getMsgLst()->addMessage ("Installing signal handler");
    my $submsglst = $msg->getSubMsgLst();
    foreach my $sig (@signals){
        if (exists $SIG{$sig}){
            $submsglst->addMessage ("Handler for signal $sig");
            setSignalHandler ($sig, sub{ $self->{siginfo} = $_[1]; signalHandlerDie($sig);});
        }
    }
    my $stackTraceFilePath = "${gTmpDir}${path_separator}hdb_core_stack_" . getFilenameTimeStamp($gLogTime) . '_' . $$ . '.trc';
    setSignalHandler ('SEGV', sub {
        $self->{siginfo} = $_[1];
        $SIG{SEGV} = 'DEFAULT'; # prevents infinite recursion
        my $fh;
        open($fh, '>', $stackTraceFilePath) or die ("__SIGSEGV__");
        print $fh longmess();
        close($fh);
        die ("__SIGSEGV__");
    });
    $SIG{__DIE__} = sub{
        if (${^GLOBAL_PHASE} eq 'DESTRUCT'){
            return 1;
        }
        if (defined $self->{stackBacktraceMsglst}){
            $self->{stackBacktraceMsglst}->initMsgLst();
            $self->{stackBacktraceMsglst}->addMessage($_[0]);
            callstack ($self->{stackBacktraceMsglst},1);
        }
        elsif (exists $ENV{'HDB_INSTALLER_STACK_BACKTRACE'}){
            print "Call stack due to $_[0]";
            print_callstack ();
        }
    };
    return 1;
}

sub ResetSignalHandler{
    my ($self) = @_;
    my $msg = $self->getMsgLst()->addMessage ("Resetting signal handler");
    my $submsglst = $msg->getSubMsgLst();
    foreach my $sig (@signals){
        if (exists $SIG{$sig}){
            $submsglst->addMessage ("Handler for signal $sig");
            $SIG{$sig} = 'DEFAULT';
        }
    }
    return 1;
}


sub new {
    my $self = shift->SUPER::new(@_);
    $self->InitApp ();
    return $self;
}


#-------------------------------------------------------------------------------
# Prepares the log/tmp path and defines the log/msg locations.
#
# Parameters:  string  $pathName   # is used to create specific log/temp path
#              string  $logName    # is used for the name of log/msg file
#              boolean $checkOnly  # may be undef
#              string  $sid        # may be undef
#
# Returns undef the log/tmp path could not be initialized

sub defineLog {

    my ($self, $pathName, $logName, $checkOnly, $sid, $isGui, $copyDir) = @_;

    $checkOnly = ($checkOnly) ? 1 : 0;
    my $errlst = new SDB::Install::MsgLst();
    my $path   = ($checkOnly) ? $pathName . '_check' : $pathName;
    my $oldLog = $self->{oldLog};

    if (defined $oldLog) {

        my $oldSID = $oldLog->{SID};

        if (($oldLog->{PathName} eq $pathName)
            &&
            ($oldLog->{LogName}  eq $logName)
            &&
            ($oldLog->{CheckOnly} == $checkOnly)
            &&
            ((!defined $sid && !defined $oldSID)
             || (defined $sid && defined $oldSID && ($oldSID eq $sid)))) {

            return 1; # log file locations already created
        }
    }

	if($self->{oldLog} && $isGui){
		$self->{log}->removeLogFileLocation($gLogDir);
	}

    if (!gPrepareLogPath($sid, $path, $errlst, $self->{options}->{instlog_dir},
                                                                      $isGui)) {
        $self->getErrMsgLst()->addError('Cannot get log directory', $errlst);
        return undef;
    }

    $self->{oldLog} = {'PathName'  => $pathName,
                       'LogName'   => $logName,
                       'CheckOnly' => $checkOnly};
    $self->{oldLog}->{SID} = $sid if (defined $sid);

    my $name    = ($checkOnly) ? $logName . '.check' : $logName;
    my $logFile = $name . '.log';
    my $msgFile = $name . '.msg';
    my $uid = $self->{options}->{instlog_uid};
    my $gid = $self->{options}->{instlog_gid};
    $gid    = getgrnam ($gSapsysGroupName) if (!$isWin && !defined $gid);

    $copyDir = $copyDir // $ENV{HDBLCM_LOGDIR_COPY};
    $self->{log}->addLogFileLocation(LOG_FORMAT_PLAIN,  $gLogDir, 1, $logFile, LOG_INDICATE_LOCATION_ON_STDOUT, $uid, $gid, undef, undef, $copyDir);
    $self->{log}->addLogFileLocation(LOG_FORMAT_MSGLIST,$gLogDir, 1, $msgFile, LOG_DONT_INDICATE_LOCATION_ON_STDOUT, $uid, $gid);
    $self->{log}->addLogFileLocation(LOG_FORMAT_PLAIN,  $gTmpDir, 1, $logFile, LOG_DONT_INDICATE_LOCATION_ON_STDOUT, $uid, $gid);
    $self->{log}->addLogFileLocation(LOG_FORMAT_MSGLIST,$gTmpDir, 1, $msgFile, LOG_DONT_INDICATE_LOCATION_ON_STDOUT, $uid, $gid);
    return 1;
}


#-------------------------------------------------------------------------------

sub InitApp{
    my ($self) = @_;
    $self->InitTrace();
    $self->InitSignalHandler();
    $self->InitLimits($self->getMsgLst());
    my $msg = $self->getMsgLst()->addMessage('System Information');
    $sysinfo = SDB::Install::SysInfo->new ();
    $msg->getSubMsgLst()->appendMsgLst ($sysinfo->getMsgLst ());
    $msg->endMessage (1, 'System Info');
    $self->{log} = SDB::Install::Log->new();
    $self->{log}->setMsgLstContext ([$self->getMsgLst()]);
    $self->{options} = {};
    # we have to take care whether we are in the gui case:
    if ($self->isa ('Wx::App')){
        $self->{log}->setIndicateLocationOnStdout(LOG_DONT_INDICATE_LOCATION_ON_STDOUT);
    }
    if (exists $ENV{SDBINSTLOGFILE}){
        my $logfile = $ENV{SDBINSTLOGFILE};
        if ($logfile =~ /[\\\/]/){
            my ($dir,$filename) = ($logfile =~ /(.*)[\\\/]+([^\\\/]+)$/);
            $self->{log}->addLogFileLocation(LOG_FORMAT_PLAIN, $dir, LOG_HISTSIZE_UNLIMITED, $filename, LOG_INDICATE_LOCATION_ON_STDOUT);
        }
        else{
            $self->{log}->addLogFileLocation(LOG_FORMAT_PLAIN, $gTmpDir, LOG_HISTSIZE_UNLIMITED, $logfile, LOG_INDICATE_LOCATION_ON_STDOUT);
        }
    }
}

sub InitTrace {
	my ($self) = @_;
	my $logdir;
	my $filename;
	if ($ENV{HDB_INSTALLER_TRACE_FILE} && -d dirname($ENV{HDB_INSTALLER_TRACE_FILE})) {
		$logdir = dirname($ENV{HDB_INSTALLER_TRACE_FILE});
		$filename = basename($ENV{HDB_INSTALLER_TRACE_FILE});
	} else {
		$logdir = $gTmpDir;
		$filename = $self->getProgramName() . '_' . getFilenameTimeStamp($gLogTime) . '_' . $$ . '.trc';
	}

	if (!LCM::DevelopmentTrace::EnableDevelopmentTrace($logdir, $filename)) {
		die ("Cannot create trace file $logdir$path_separator$filename : $!");
	}
}


sub GetSysInfo{
    return $sysinfo;
}

sub GetInstaller{
    return $installer;
}

sub getProgramName{
    return 'BaseApp';
}

#-------------------------------------------------------------------------------
# Returns a reference to an array containing program options
# without help/info options.

sub GetSwitches{
    return [];
}


#-------------------------------------------------------------------------------
# eturns a reference to an array containing batch options of the program.

sub GetBatchSwitches{
    return ['--batch', '--configfile=<filename>','--dump_configfile_template=<filename>'];

}


#-------------------------------------------------------------------------------
# Returns a reference to an array containing help/info options of the program.

sub GetHelpSwitches {
    return ['--help', '--version'];
}


#-------------------------------------------------------------------------------
# Returns the sorted help switches as a string.

sub getHelpSwitchesString {

    my ($self, $indent) = @_;

    my $switches    = $self->GetHelpSwitches($indent);
    my $line        = ' ';
    my $lineLimit   = 100 - length($indent);
    my $currLineLen = 0;
    my $isFirst     = 1;

    foreach my $currSwitch (sort @$switches) {

        if ($isFirst) {
            $line       .= $currSwitch;
            $currLineLen = length($line);
            $isFirst     = 0;
            next;
        }

        my $len = length($currSwitch);

        if ($currLineLen + $len > $lineLimit) {
            $line       .= "\n$indent";
            $currLineLen = 0;
        }
		$line        .= " | $currSwitch";
        $currLineLen += $len;
    }
    return $line;
}


#-------------------------------------------------------------------------------
# Returns a reference to an array of arrays
# containg the description of batch and help options.

sub GetBatchHelpUsage {
    return [['--batch',   '-b',  'Runs the program in batch mode using default values for unspecified parameters'],
            ['--configfile=<filename>', undef, 'Reads parameters from the specified configuration file'
                                               .' (parameters in command line take precedence)'],
            ['--dump_configfile_template=<filename>', undef, 'Creates a configuration file with default values'],
            ['--help',    '-h',  'Displays the help information'],
            ['--version', '-v',  "Displays the version of $SAPDB::Install::Config{ProgramName}"]];
}


#-------------------------------------------------------------------------------
# Returns a reference to an array of arrays
# containg the description of program options.

sub GetUsage {
    return undef;
}


#-------------------------------------------------------------------------------
# Returns a reference to an array of arrays
# containg the description of password tags.

sub GetPasswordUsage {
    return undef;
}


#-------------------------------------------------------------------------------
# Returns the sorted switches as a string consisting of several lines.

sub getSwitchesString {

    my ($self, $indent) = @_;

    my $batchSwitches = $self->GetBatchSwitches();
    my $switches      = $self->GetSwitches();
    my $lines         = '';
    my $lineLimit     = 100 - length($indent);
    my $currLineLen   = 0;

    $switches = [@$batchSwitches, @$switches] if (defined $batchSwitches);

    foreach my $currSwitch (sort @$switches) {

        my $entry = " [$currSwitch]";
        my $len   = length($entry);

        if (($currLineLen > 0) && $currLineLen + $len > $lineLimit) {

            $lines       .= "\n$indent";
            $currLineLen  = 0;
        }

        $lines       .= $entry;
        $currLineLen += $len;
    }

    return $lines;
}


#-------------------------------------------------------------------------------
# Returns the complete help information as a string consisting of several lines.

sub getUsageString{
    my ($self, $withEnvVarCopyLogDir) = @_;

    my $program      = $SAPDB::Install::Config{ProgramName};
    my $helpPrefix   = "Help:  $program";
    my $usagePrefix  = "Usage: $program";
    my $helpIndent   = (' ' x length($helpPrefix));
    my $usageIndent  = (' ' x length($usagePrefix));

    my $result  = "$gProductNameInstaller ($program)\n"
                . "$SAPDB::Install::Version::Copyright\n\n"
                . $helpPrefix . $self->getHelpSwitchesString($helpIndent) . "\n\n"
                . $usagePrefix. $self->getSwitchesString($usageIndent) . "\n\n";

   my $usage          = $self->GetUsage();
   my $batchHelpUsage = $self->GetBatchHelpUsage();

   $usage = (defined $usage) ? [@$batchHelpUsage, @$usage] : $batchHelpUsage;

    my $outputLines =
         printSortedTableToArrayOfLines($usage, 0, ' ',
              [$USAGE_COL1_LIMIT,  $USAGE_COL2_LIMIT, $USAGE_COL3_LIMIT], '  ', 3);

    foreach my $line (@$outputLines){
        $result .= "$line\n";
    }

    my $passwordUsage = $self->GetPasswordUsage();
    my $passwordCnt   = (defined $passwordUsage) ? scalar @{$passwordUsage} : 0;

    if ($passwordCnt > 0) {

        push @$passwordUsage, ['Passwords', 'Main tag of password section'];

        $result     .= "\nXML tags for password input stream:\n";
        $outputLines = printSortedTableToArrayOfLines($passwordUsage, 0, ' - ',
                        [$PASSWORD_COL1_LIMIT, $PASSWORD_COL2_LIMIT], '  ');

        foreach my $line (@$outputLines){
            $result .= "$line\n";
        }
    }
    
    $result .= "\nThe environment variable 'HDB_INSTALLER_TRACE_FILE=<file>' enables the trace.\n";
    if ($withEnvVarCopyLogDir) {
        $result .= "The environment variable 'HDBLCM_LOGDIR_COPY=<target_dir>' creates a copy of the log directory.\n";
    }
	
	if($self->can("getHelpNotes")){
		my $notes=$self->getHelpNotes();
		if(defined $notes && $notes ne ''){
			$result .= $notes."\n";
		}
	}
	
    return $result;
}

#-------------------------------------------------------------------------------
sub PrintUsage{
    print $_[0]->getUsageString ($_[1]);
}


sub WrongUsage{
    print STDERR "\nWrong usage!\n\n";
    if ($_[1]){
        print STDERR ${$_[1]->getMsgLstString()} . "\n";
    }
    $_[0]->setErrorMessage ("Wrong usage!");
    $_[0]->appendErrorMessage (${$_[1]->getMsgLstString()});
    $_[0]->PrintUsage ();
}

sub PrintVersion{
    my ($self) = @_;
    require SAPDB::Install::Version;

    my $versionString =
            "$gProductNameInstaller ($SAPDB::Install::Config{ProgramName})\n".
            "$SAPDB::Install::Version::Copyright\n".
            "Version: $SAPDB::Install::Version::Buildstring\n".
            "GitHash: $SAPDB::Install::Version::GitHash\n".
            "Perl: $^V\n\n";
    $self->PrintText ($versionString);
}


sub ShowErrorMsg{
    my ($self,$text,$list) = @_;
    my $msg = $self->setErrorMessage ($text,$list,4);
    print STDERR (${$self->getErrMsgLst()->getMsgLstString ()} . "\n");
    return $msg;
}


sub getOptions{
    my ($self, $opt_ctrl, $pass_through) = @_;
    if (!defined $pass_through){
        $pass_through = 1;
    }
    if ($pass_through){
        Getopt::Long::Configure ('pass_through');
    }
    else{
        Getopt::Long::Configure ('nopass_through');
    }
    my $errlst        = new SDB::Install::MsgLst ();
    my $ignoreUnknown = $self->{options}->{ignore_unknown_option};

    local $SIG{__WARN__} = sub {
        my ($text) = @_;
        $text =~ s/\n$//m;
        if ($ignoreUnknown) {
            $self->PrintText($text."\n");
        }
        else {
            $errlst->addError ($text);
        }
    };

    my $error = 0;
    if (! GetOptions  (%$opt_ctrl)){
        $error = 1 if (!$ignoreUnknown);
    }
    if (!$pass_through && @ARGV){
        $error = 1 if (!$ignoreUnknown);
        if (@ARGV){
            foreach my $arg (@ARGV){
                my $errMsg = "Unknown argument: $arg";
                if ($ignoreUnknown) {
                    $self->PrintText($errMsg."\n");
                }
                else {
                    $errlst->addError($errMsg);
                }
            }
        }
     }
    if ($error){
        $self->WrongUsage($errlst);
        return undef;
    }
    return 1;
}

sub isHelp {
    my ($self) = @_;
    return (exists($self->{options}) && $self->{options}->{help}) ? 1 : 0;
}

sub isDumpTemplate {
    my ($self) = @_;
    return (defined($self->{dump_configfile_template})) ? 1 : 0;
}

sub InitCmdLineArgs {
    my ($self,$arguments) = @_;
    local @ARGV = @$arguments;
    my $progname = $SAPDB::Install::Config{ProgramName};
    my $instmsg = $self->getMsgLst()->addMessage('Installer Information');
    $instmsg->setEndTag ('Installer Information');

    my $instsubmsglst = $instmsg->getSubMsgLst();
    $self->GetInstaller ()->appendInstallerInfo ($instsubmsglst);

    if ($self->isa ('SDB::Install::App::Console')){
        $instsubmsglst->addMessage ('Running as console application');
        if ($SAPDB::Install::Config{SilentMode}){
            $instsubmsglst->addMessage ('Running in background mode');
        }
    }
    elsif ($self->isa ('SDB::Install::App::Gui')){
        $instsubmsglst->addMessage ('Running as gui application');
        $instsubmsglst->addMessage ('Using ' . Wx::wxVERSION_STRING ());
    }
    
    my $progmsg = $instsubmsglst->addMessage ('Program Call Information');
    my $progsubmsglst = $progmsg->getSubMsgLst();
    $progsubmsglst->addMessage ('Caller program name is '. $progname);
    my $args = join (' ',@ARGV);
    
    
    #
    # remove passwords in log file
    #
    
    $args =~ s/(-u\s\S+,)\S+/$1\*\*\*/;
    $args =~ s/(-{1,2}(?!use_)[\w_]*password=)\S+/$1\*\*\*/g;
    $args =~ s/(-{1,2}(?!use_)[\w_]*password\s+)\S+/$1\*\*\*/g;
    $args =~ s/(-p\s+)\S+/$1\*\*\*/;
    $args =~ s/(-p=)\S+/$1\*\*\*/;
    
    $progsubmsglst->addMessage ('Command line arguments: '. $args);
    $progsubmsglst->addMessage ('Current directory: '. $SAPDB::Install::Config{CallerDir});
    $progsubmsglst->addMessage ('Installer directory: '. $SAPDB::Install::Config{RuntimeDir});
    $progsubmsglst->addMessage ('Pid is '. $$);
    if ($^O !~ /mswin/i){
        $progsubmsglst->addMessage ('Effective uid '. $>);
        $progsubmsglst->addMessage ('Real uid '. $<);
        $progsubmsglst->addMessage ('Effective gid '. $));
        $progsubmsglst->addMessage ('Real gid ' . $();
    }
    $progmsg->endMessage (1);

    my $envmsg = $instsubmsglst->addMessage ('Environment Dump');
    my $envmsglst = $envmsg->getSubMsgLst (); 
    foreach my $var (sort keys (%ENV)){
        $envmsglst->addMessage ($var . ' = ' . $ENV{$var});
    }
    $envmsg->endMessage (1);
    $instmsg->endMessage (1);
    Getopt::Long::Configure ('noauto_abbrev');
    Getopt::Long::Configure ('gnu_compat');
    Getopt::Long::Configure ('no_ignore_case');
    Getopt::Long::Configure ('no_bundling_values');
    Getopt::Long::Configure ('long_prefix_pattern=--|-');

    if (!defined $self->getOptions (
        {'version'   => \$self->{options}->{version},
        'v'     => \$self->{options}->{version},
        'help'      => \$self->{options}->{help},
        'h'     => \$self->{options}->{help},
        'called_by_hostagent' => \$self->{options}->{called_by_hostagent},
        'configfile=s'  => \$self->{configfile},
        'dump_configfile_template=s'  => \$self->{dump_configfile_template},
        'dump_config_xml'  => \$self->{dump_config_xml},
        'batch'     => \$self->{batch_mode},
        'b'     => \$self->{batch_mode},
        'noprompt' => \$self->{options}->{noprompt}, # skips 'press any key' when finishing under Windows
        'ignore_unknown_option' => \$self->{options}->{ignore_unknown_option},
        ($isWin ? () :
            ( 'instlog_uid=i', \$self->{options}->{instlog_uid}, # used for writing files into $gLogDir
              'instlog_gid=i', \$self->{options}->{instlog_gid}) # used for writing files into $gLogDir
        ),
        'instlog_dir=s'              => \$self->{options}->{instlog_dir}, # SAPHostctrl param: 'INSTLOG_DIR'
                                                                          # Uses this directory for writing log files ($gLogDir = <paramValue>)
        'action_id=s'                => \$self->{options}->{action_id},   # SAPHostctrl param: 'ACTION_ID'
                                                                          # Format: '<programName>_<timestamp>_<pid>'
                                                                          # ID is inserted into filenames when log files are copied to trace dir
        'read_password_from_stdin:s' => \$self->{options}->{read_password_from_stdin}},
        1
    )){
        return undef;
    }

    if (defined $self->{options}->{read_password_from_stdin}){
        if (!$self->{batch_mode}){
            $self->appendErrorMessage ("Wrong usage: --read_password_from_stdin is allowed in batch mode only (use -b|--batch)");
            return undef;
        }
        if ($self->{options}->{read_password_from_stdin} =~ /\D/){
            if ($self->{options}->{read_password_from_stdin} ne 'xml'){
                $self->appendErrorMessage ("Wrong usage: --read_password_from_stdin just accepts optional argument 'xml'");
                return undef;
            }
        }
    }

    if (!$self->{_two_step_config}){
        if (defined $self->{configfile}){
            $self->{configfile} = $self->getAbsPath ($self->{configfile});
            if (!-f $self->{configfile}){
                $self->appendErrorMessage ("Wrong usage: --configfile just accepts an existing file as argument: $!");
                return undef;
            }
        }

        if (!defined $self->{hdblcm} && !$self->isa ('SDB::Install::App::Installation') && $self->isHelp()){
            $self->PrintUsage ();
            $self->{return} = 0;
        }
    }

    if ($self->{options}->{version}){
        $self->PrintVersion ();
        $self->{return} = 0;
    }
    return \@ARGV;
}


#----------------------------------------------------------------------------

sub CleanUp{
    my ($self) = @_;


     if ($self->errorState){
        my $msg = $self->{msglst}->addMessage ('Summary of critical errors');
        $msg->getSubMsgLst ()->appendMsgLst ($self->getErrMsgLst ());
    }

    #$self->SetProgressHandler();
    delete $self->{instconfig};
    delete $self->{kit};
    my $packageManagerCollection = $self->{'packageManagerCollection'};
    # clean up the 'SAPSystem's which we have cached:
    CleanupSAPSystemCache ();
    delete $self->{'packageManagerCollection'};
    if (defined $packageManagerCollection){
        # All 'MsgList' instances are organized in a tree structure;
        # the packageManagerCollection contains the top level nodes.
        # Here, we set '$self' as the root of the entire tree,
        # and messages are propagated towards the root
        # by calling 'ImportMsgLst' via recursive calls to the
        # 'CleanUp' methods of the nodes,
        # before we can write the logfile.
        $packageManagerCollection->setMsgLstContext ([$self->getMsgLst()]);
        $packageManagerCollection->CleanUp();
    }
    #$self->LogMessages ();
    # writing the log files is also part of Log's dtor code:
    # (but note that in the gui case the log object has been
    #  destroyed some time ago elsewhere in the code
    #  and is already undef here.)
    if (defined $self->{log} && !$self->{log}->writeLogs()) {
        print STDERR ($self->GetErrorString ()) . "\n";
    }
    undef $self->{log};
}

#----------------------------------------------------------------------------


sub DESTROY{
    my ($self) = @_;
}


#-------------------------------------------------------------------------------
# Returns a string containing a lists of the existing systems
# and the used instance numbers.

sub getListSystems {

    my ($self) = @_;

    my $result;
    my $systems      = $self->collectSAPSystems ();
    my $countSystems = (defined $systems) ? scalar keys %$systems : 0;

    if ($countSystems == 0) {
        $result  = "No $gProductName database installation found\n" .
                   "Next available instance number: 00\n";
    }
    else {
        $result = "$gProductName Database Installations:\n\n";

        require SDB::Install::Configuration::NewDB;
        require SDB::Install::Configuration::Upgrade;
        my @occupied_numbers;

        foreach my $sid (sort keys %$systems){

            $result .= $systems->{$sid}->asString(undef, undef, 1)
                     . $self->tryGetPendingInfo($systems->{$sid})
                     . "\n";

            push @occupied_numbers,
                 @{$systems->{$sid}->getOccupiedInstanceNumbers()};
        }

        if (scalar @occupied_numbers > 0) {

        my $instconfig = $self->{instconfig};

        if (!defined $instconfig ||
            !$instconfig->isa ('SDB::Install::Configuration::NewDB')) {

            require SDB::Install::Configuration::NewDB;
            $instconfig = new SDB::Install::Configuration::NewDB();

            if (!$instconfig->InitDefaults ($self->{kit})) {
                return undef;
            }
        }

        my $strNumber = ($countSystems == 1) ? 'number: ' : 'numbers: ';

        $result .= "Already used instance $strNumber"
                                    . join (' ', sort @occupied_numbers) . "\n";
        $result .= "Next available instance number: "
                             . $instconfig->getDefault('InstanceNumber') . "\n";
        }
    }

    return $result;
}

sub collectSAPSystems {
    my $self = shift;
    return SDB::Install::SAPSystem::CollectSAPSystems(@_);
}

#-------------------------------------------------------------------------------
# Returns an empty string or a string containing information
# about pending installation or update.

sub tryGetPendingInfo {

    my ($self, $sapSys) = @_;
    require SDB::Install::Configuration::NewDB;
    my $instConfig = new SDB::Install::Configuration::NewDB();
    $instConfig->{current_sid} = $sapSys->get_sid();
    $instConfig->{sapSys}      = $sapSys;

    my $pendingInfo = getPendingInfoFromPersFile('Installation',
                                          'of',
                                          $instConfig,
                                          \@STEP_NAMES,
                                          "\t");

    my $addHostPending = undef;
    my $pendingHosts   = $sapSys->getPendingAddHosts();
    if (defined $pendingHosts) {
        $addHostPending = "\t" . join("\n\t", @$pendingHosts) . "\n";
    }

    if (defined $pendingInfo){
        $pendingInfo .= $addHostPending if (defined $addHostPending);
        return $pendingInfo;
    }

    $pendingInfo = $self->getPendingUpdateInfo($sapSys);

    $pendingInfo  = '' if (!defined $pendingInfo);
    $pendingInfo .= $addHostPending if (defined $addHostPending);

    my $regRenamePending = $sapSys->getRegisterOrRenamePending();

    if (defined $regRenamePending) {
        $pendingInfo .= "\t" . $regRenamePending . "\n";
    }

    return $pendingInfo;
}


sub getPendingUpdateInfo {
    my ($self, $sapSys) = @_;

    my $updConfig = new SDB::Install::Configuration::Upgrade();
    $updConfig->{current_sid} = $sapSys->get_sid();
    $updConfig->{sapSys}      = $sapSys;

    return getPendingInfoFromPersFile('Update',
                                      'to',
                                      $updConfig,
                                      \@UPGRADE_STEP_NAMES,
                                      "\t");
}


#-------------------------------------------------------------------------------
# handleOptionDumpConfigFileTemplate ()
# writes a config file template based on $instconfig,
# if --dump_configfile_template option is set
#
# Returns:
#           undef       Error
#           0           Success, no config file template written
#           1           Success, config file template written
#

our $cfgFileDumpError = 'Cannot dump config file template';

sub handleOptionDumpConfigFileTemplate{
    my ($self) = @_;
    if (!defined $self->{dump_configfile_template}){
        return 0;
    }
    if (!$self->{dump_configfile_template}) {
# If we get here, it means that the --dump_configfile_template was given
# on the command line, but with empty value
        $self->setErrorMessage("$cfgFileDumpError: Empty value given to --dump_configfile_template. Please provide a filename.");
        return undef;
    }
    my $instconfig = $self->getInstconfig ();
    if (!defined $instconfig){
        $self->setErrorMessage ("$cfgFileDumpError: Instconfig is not defined");
        return undef;
    }
    my $cfgFileName = $self->getAbsPath($self->{dump_configfile_template});
    if (!defined $instconfig->dumpConfigFileTemplate($cfgFileName)){
        $self->{return} = undef;
        $self->setErrorMessage ($cfgFileDumpError, $instconfig->getErrMsgLst());
        return undef;
    }
    $self->getMsgLst()->addProgressMessage ("Config file template '$cfgFileName' written", $instconfig->getMsgLst());
    my $passwordFile = $cfgFileName . '.xml';
    if (-f $passwordFile) {
        $self->getMsgLst()->addProgressMessage ("Password file template '$passwordFile' written", $instconfig->getMsgLst());
    }
    return 1;
}

our $cfgXMLDumpError = 'Cannot dump xml config to stdout';

sub handleOptionDumpConfigXML{
    my ($self) = @_;
    if (!defined $self->{dump_config_xml}){
        return 0;
    }
    my $instconfig = $self->getInstconfig ();
    if (!defined $instconfig){
        $self->setErrorMessage ("$cfgXMLDumpError: Instconfig is not defined");
        return undef;
    }
    $instconfig->setMsgLstContext ([$self->getMsgLst ()]);
    my $xmlRef = $instconfig->configuration2Xml ();
    if (!defined $xmlRef){
        $self->setErrorMessage ($cfgXMLDumpError, $instconfig->getErrMsgLst ());
        return undef;
    }
    print ($$xmlRef);
    return 1;
}




#-------------------------------------------------------------------------------
# getAbsPath ()
# returns the absolute path based on callers cwd
#
# Returns:
#           $absPath
#

sub getAbsPath{
    my ($self, $relpath) = @_;
    return File::Spec->rel2abs ($relpath, $installer->getCallerDir ());
}


1;
