# CollectHostInfo
#
# Parameters:
# ------------
#
# --dataPath=<data_path>                   # checks the existence of this data path (xml tag 'dataPath')
#
# --logPath=<log_path>                     # checks the existence of this log path (xml tag 'logPath')
#
# --collect_only=<xml_tag>[,<xml_tag>]     # selects these xml tags only (e.g. 'SAPSystems,users')
#                                          # <xml_tag>: main tags except 'currentUser', 'datapath', 'logPath'
#
# --skip_collect=all|<xml_tag>[,<xml_tag>] # does not select these xml tags
#                                          # (cannot be applied together with '--collect_only')
#                                          # <xml_tag>: main tags except 'currentUser', 'datapath', 'logPath'
#
# --sapadm=<sapadm_username>               # if xml tag 'users' is skipped the properties
#                                          # of this sapadm are collected
#
# --sidadm=<sidadm_username>               # if xml tag 'users' is skipped the properties
#                                          # of this sidadm are collected
#
# --userid=<uid>                           # if xml tag 'users' is skipped the properties
#                                          # of this sidadm user ID are collected
#
# --groupid=<gid>                          # if xml tag 'groups' is skipped the properties
#                                          # of this sapsys group ID are collected
#
# --usernames=<user1>[,<user2>,...]        # if xml tag 'users' is skipped the properties
#                                          # of all users in the list are collected
#
# --groupnames=<group1>[,<group2>,...]     # if xml tag 'groups' is skipped the properties
#                                          # of all groups in the list are collected
#
# --sidcrypt=<uid>                         # if xml tag 'users' is skipped the properties
#                                          # of this sidcrypt user ID are collected
#
# --sidcrypt_groupid=<gid>                 # if xml tag 'groups' is skipped the properties
#                                          # of this sidcrypt group ID are collected
#

#
#
# Formatted Sample Output:
# ------------------------
#
# <HostDescription>
#
#     <currentUser isAdmin="1" />
#
#     <sapsysid>79</sapsysid>
#
#     # if sapsys exists, collection of other groups (except --groupid) is suppressed
#     <groups>
#         <group id="17">audio</group>
#         <group id="79">sapsys</group>
#     </groups>
#
#     <users>
#         <user id="1" gid="1">bin</user>
#         <user id="2" gid="2">daemon</user>
#         <user id="0" gid="0">root</user>
#         <user id="1002" gid="79">db7adm</user>
#         <user id="1003" gid="79">db1adm</user>
#     </users>
#
#    <SAPSystems>
#         <System sid="DB1">
#             <Landscape id="51ddf0d6-83bd-188a-e100-00007f000002"></Landscape>
#             <Instances>
#                 <Instance type="HDB">01</Instance>
#             </Instances>
#         </System>
#         <System sid="DB7">
#             <Landscape id="51d52dce-8eb9-0d3f-e100-00007f000002"></Landscape>
#             <Instances>
#                 <Instance type="HDB">07</Instance>
#             </Instances>
#         </System>
#     </SAPSystems>
#
#     <networkAddresses>
#         <ip>100.100.100.100/24</ip>
#         <ip>fe80::21a:6bff:fe51:fb16/64</ip>
#         <ip>127.0.0.1/8</ip>
#         <ip>127.0.0.2/8</ip>
#         <ip>::1/128</ip>
#     </networkAddresses>
#
#     <saphostagentSslConfig>135:PFS:HIGH::EC_P256:EC_HIGH</saphostagentSslConfig>
#
#     <saphostagentVersion>7.20.162</saphostagentVersion>
#	  <certificateHostname>myhanahost</certificateHostname>
#	  <dataPath isExists="1" sidadmCanAccess="1" isSharedVolume="1" isSharedSubvolume="1">
#       <MSGL _NO="1" _PROCESS="7381" _THREAD="0" _TIME="2019-03-25 10:31:51.459">
#           <MSG _NO="9857" _TYPE="Error" _ID="0" _COMP="Install" _TEXT="Missing read permission for others at /hanamnt_1/data">
#               <MSG_ARGS _FILE="SDB/Install/MsgLst.pm" _LINE="327" _OBJECT="SDB::Install::MsgLst=HASH(0x7f57c1b588d0)" _TIME="2019-03-25 10:31:51.459" _MESSAGEVERSION="1"/>
#           </MSG>
#       </MSGL>
#     </dataPath>
#	  <logPath isExists="1" sidadmCanAccess="1" isSharedVolume="0" isSharedSubvolume="1">
#       <MSGL _NO="1" _PROCESS="7381" _THREAD="0" _TIME="2019-03-25 10:31:51.459">
#           <MSG _NO="9857" _TYPE="Error" _ID="0" _COMP="Install" _TEXT="Missing read permission for others at /hanamnt_1/data">
#               <MSG_ARGS _FILE="SDB/Install/MsgLst.pm" _LINE="327" _OBJECT="SDB::Install::MsgLst=HASH(0x7f57c1b588d0)" _TIME="2019-03-25 10:31:51.459" _MESSAGEVERSION="1"/>
#           </MSG>
#       </MSGL>
#     </logPath>
#	  <lssPath isShared="1" />
# </HostDescription>


package SDB::Install::App::Console::CollectHostInfo;

use File::stat;
use File::Spec;
use SDB::Install::Globals qw ($gSapsysGroupName);
use SDB::Install::Configuration::NewDB;
use SDB::Install::SAPSystem;
use SDB::Install::Saphostagent qw(getSHAVersion getActiveSaphostexecDir);
use SDB::Install::SysInfo;
use SDB::Common::Utils qw(toBase64);
use LCM::RegenerateCertificates;
use SDB::Install::System qw (isAdmin);
use SDB::Common::BuiltIn;
use Getopt::Long;
use SDB::Install::SAPProfiles qw($cipherSuitesId $recommendedCipherSuites);
use SDB::Install::System qw(isDirectory);
use SDB::Install::System::Mounts;
use SDB::Install::System qw(X_OK);
use strict;

sub _maskContent ($$){
    my ($refData, $refIsBase64Encoded) = @_;
    if ($$refData =~ /[^[:print:]]/){
        $$refIsBase64Encoded = 1;
        $$refData = toBase64($$refData);
    }
    else{
        $$refIsBase64Encoded = 0;
        $$refData =~ s/&/&amp;/sg;
        $$refData =~ s/</&lt;/sg;
        $$refData =~ s/>/&gt;/sg;
        $$refData =~ s/"/&quot;/sg;
    }
}


sub main{
    local @ARGV = @_;

    my $certificateHostnameTag = 'certificateHostname';
    my $dataPathTag            = 'dataPath';           # param --dataPath only
    my $groupsTag              = 'groups';
    my $logPathTag             = 'logPath';            # param --logPath only
    my $networkAddressesTag    = 'networkAddresses';
    my $saphostagentSslConfigTag = 'saphostagentSslConfig';
    my $saphostagentVersionTag = 'saphostagentVersion';
    my $sapsysidTag            = 'sapsysid';
    my $SAPSystemsTag          = 'SAPSystems';
    my $usersTag               = 'users';
    my $systemCheckErrorsTag   = 'systemCheckErrors';
    my $lssPathTag              = 'lssPath';

    my $dataPath    = '';
    my $logPath     = '';
    my $collectOnly = '';
    my $skipCollect = '';
    my $sapadm      = '';
    my $sidadm      = '';
    my $sidcrypt    = '';
    my $userid      = '';
    my $groupid     = '';
    my $usernames   = '';
    my $groupnames  = '';
    my $sidcrypt_groupid  = '';
    my $compilerVersion = '';
    my $lssPath     = '';
    my $sidadmUID   = undef;
    my $chkSharedDataPathFile = undef;
    my $chkSharedLogPathFile = undef;
    my $chkSharedDataSubvolumeFile = undef;
    my $chkSharedLogSubvolumeFile = undef;
    my $chkSharedLssPathFile = undef;

    GetOptions ('dataPath=s'     => \$dataPath,
                'logPath=s'      => \$logPath,
                'collect_only=s' => \$collectOnly,
                'skip_collect=s' => \$skipCollect,
                'sapadm=s'       => \$sapadm,
                'sidadm=s'       => \$sidadm,
                'sidcrypt=s'     => \$sidcrypt,
                'usernames=s'    => \$usernames,
                'groupnames=s'    => \$groupnames,
                'userid=s'       => \$userid,
                'groupid=s'      => \$groupid,
                'sidcrypt_groupid=s'      => \$sidcrypt_groupid,
                'compiler_version=s' => \$compilerVersion,
                'chkshared_datavolume_file=s' => \$chkSharedDataPathFile,
                'chkshared_logvolume_file=s' => \$chkSharedLogPathFile,
                'chkshared_datasubvolume_file=s' => \$chkSharedDataSubvolumeFile,
                'chkshared_logsubvolume_file=s' => \$chkSharedLogSubvolumeFile,
                'chkshared_lsspath_file=s' => \$chkSharedLssPathFile,
                                );

    my %collect = ($certificateHostnameTag => 1,
                   $groupsTag              => 1,
                   $networkAddressesTag    => 1,
                   $saphostagentSslConfigTag => 1,
                   $saphostagentVersionTag => 1,
                   $sapsysidTag            => 1,
                   $SAPSystemsTag          => 1,
                   $usersTag               => 1,
                   $systemCheckErrorsTag   => 1,
                  );

    my $builtin = SDB::Common::BuiltIn->get_instance();

    if ($collectOnly ne '') {
        %collect = ();
        foreach my $item (split(',', $collectOnly)) {
            $collect{$item} = 1;
        }
    }
    elsif ($skipCollect ne '') {
        foreach my $item (split(',', $skipCollect)) {
            if ($item eq 'all') {
                %collect = ();
                last;
            }
            else {
                delete $collect{$item};
            }
        }
    }

    $builtin->print ('<?xml version="1.0" encoding="UTF-8"?>' . "\n");
    $builtin->print ("<HostDescription>\n");

    $builtin->printf ("<currentUser isAdmin=\"%d\" />\n", isAdmin () ? 1 : 0);

    my $sapsysid      = undef;

    if ($collect{$sapsysidTag}) {
        #sapsys group
        $sapsysid = $builtin->getgrnam ($gSapsysGroupName);
        if (defined $sapsysid){
            $builtin->printf ("<$sapsysidTag>%d</$sapsysidTag>\n", $sapsysid);
        }
    }

# ==========================
# Collect groups logic
# ==========================
    my $groupFormat       = "<group id=\"%d\">%s</group>\n";
    my $groupFormatBase64 = "<group id=\"%d\" encoding=\"base64\">%s</group>\n";
    my $isBase64Encoded;
    my $collectedGroups = {};

    if ($collect{$groupsTag}) {
        # Collect all groups
        my ($name, $gid);
        $builtin->setgrent();
        while (($name, undef, $gid) = $builtin->getgrent()) {
            next if ($gid == 0);
            _maskContent (\$name,\$isBase64Encoded);
            $collectedGroups->{$name} = sprintf (($isBase64Encoded) ? $groupFormatBase64 : $groupFormat, $gid, $name);
        }
        $builtin->endgrent();
    }
    # Collect specific groups (needed if getgrent fails to get any users in LDAP case for example)
    if (defined $sapsysid && !$collectedGroups->{$gSapsysGroupName}) {
        $collectedGroups->{$gSapsysGroupName} = sprintf($groupFormat, $sapsysid, $gSapsysGroupName);
    }
    if (defined $groupid && $groupid != 0) {
        my $nameOfGroupid = $builtin->getgrgid($groupid);
        if (defined $nameOfGroupid && !$collectedGroups->{$nameOfGroupid}) {
            $collectedGroups->{$nameOfGroupid} = sprintf ($groupFormat, $groupid, $nameOfGroupid);
        }
    }
    if (defined $sidcrypt_groupid && $sidcrypt_groupid != 0) {
        my $nameOfGroupid = $builtin->getgrgid($sidcrypt_groupid);
        if (defined $nameOfGroupid && !$collectedGroups->{$nameOfGroupid}) {
            $collectedGroups->{$nameOfGroupid} = sprintf ($groupFormat, $sidcrypt_groupid, $nameOfGroupid);
        }
    }
    if (defined $groupnames) {
        foreach my $groupname (split (',', $groupnames)){
            my ($name, undef, $gid) = $builtin->getgrnam ($groupname);
            if (defined $gid && $gid != 0 && !$collectedGroups->{$name}) {
                _maskContent (\$name,\$isBase64Encoded);
                $collectedGroups->{$name} = sprintf ($isBase64Encoded ? $groupFormatBase64 : $groupFormat, $gid, $name);
            }
        }
    }
    if (keys %{$collectedGroups}) {
        $builtin->print ("<$groupsTag>\n");
        $builtin->print ($_) for values(%{$collectedGroups});
        $builtin->print ("</$groupsTag>\n");
    }
# ==========================
# END: Collect groups logic
# ==========================

# ==========================
# Collect users logic
# ==========================
    my $userFormat              = "<user id=\"%d\" gid=\"%d\">%s</user>\n";
    my $userFormatBase64        = "<user id=\"%d\" gid=\"%d\" encoding=\"base64\">%s</user>\n";
    my $collectedUsers   = {};

    # Try and collect all users
    if ($collect{$usersTag}) {
        my ($name, $uid, $gid);
        $builtin->setpwent ();
        while (($name, undef, $uid, $gid) = $builtin->getpwent()) {
            _maskContent (\$name,\$isBase64Encoded);
            $collectedUsers->{$name} = sprintf ($isBase64Encoded ? $userFormatBase64 : $userFormat, $uid, $gid, $name) if ($uid != 0);
        }
        $builtin->endpwent ();
    }
    # Collect specific users only (needed if getpwent fails to get any users in LDAP case for example)
    if ($sapadm ne '' && !$collectedUsers->{$sapadm}) {
        # collect system administrator only
        my ($name, undef, $uid, $gid) = $builtin->getpwnam($sapadm);
        if (defined $uid) {
            $collectedUsers->{$sapadm} = sprintf ($userFormat, $uid, $gid, $name);
        }
    }
    if ($sidadm ne '' && !$collectedUsers->{$sidadm}) {
        # collect system administrator only
        my ($name, undef, $uid, $gid) = $builtin->getpwnam($sidadm);
        if (defined $uid) {
            $sidadmUID  = $uid;
            $collectedUsers->{$name} = sprintf ($userFormat, $uid, $gid, $name);
        }
    }

    if ($sidcrypt ne '' && !$collectedUsers->{$sidcrypt}) {
        # collect lss user only
        my ($name, undef, $uid, $gid) = $builtin->getpwnam($sidcrypt);
        if (defined $uid) {
            $collectedUsers->{$name} = sprintf ($userFormat, $uid, $gid, $name);
        }
    }

    if ($usernames ne '') {
        foreach my $username (split (',', $usernames)){
            # collect each user in the list
            next if $collectedUsers->{$username};

            my ($name, undef, $uid, $gid) = $builtin->getpwnam($username);
            if (defined $uid) {
                $collectedUsers->{$name} = sprintf ($userFormat, $uid, $gid, $name);
            }
        }
    }

    if (($userid ne '') && (!defined $sidadmUID || ($sidadmUID != $userid))){

        my ($name, undef, $uid, $gid) = $builtin->getpwuid($userid);
        if (defined $name && ($name ne $sidadm) && !$collectedUsers->{$name}) {
            $collectedUsers->{$name} = sprintf ($userFormat, $uid, $gid, $name);
        }
    }
    if (keys %{$collectedUsers}) {
        $builtin->print ("<$usersTag>\n");
        $builtin->print ($_) for values(%{$collectedUsers});
        $builtin->print ("</$usersTag>\n");
    }
# ==========================
# END: Collect users logic
# ==========================

    if ($collect{$SAPSystemsTag}) {
        $builtin->print ("<$SAPSystemsTag>\n");
        my $systems = CollectSAPSystems ();
        my $instances;
        my ($reservedInstanceNrs, $attribReservedInstanceNrs);

        foreach my $sid (keys (%$systems)){
            $builtin->printf ("<System sid=\"%s\">\n", $sid);
            my $currSystem = $systems->{$sid};

            # landscape
            if ($currSystem->hasNewDB()) {
                my $instance    = $currSystem->getNewDBInstances()->[0];
                my $landscapeID = (defined $instance) ? $instance->getLandscapeID()
                                                      : undef;
                if (defined $landscapeID) {
                    $builtin->printf ("<Landscape id=\"%s\"></Landscape>\n", $landscapeID);
                }
            }
            # instances
            $builtin->print ("<Instances>\n");
            $instances = $currSystem->get_instances ();
            if (defined $instances){
                foreach my $num (keys (%$instances)){
                    $reservedInstanceNrs = $instances->{$num}->getReservedInstanceNumbers();
                    if ($reservedInstanceNrs > 0){
                        $attribReservedInstanceNrs = " reserved_instance_numbers=\"$reservedInstanceNrs\"";
                    }
                    else{
                        $attribReservedInstanceNrs = '';
                    }
                    $builtin->printf ("<Instance type=\"%s\"%s>%02d</Instance>\n",
                                      $instances->{$num}->get_type (),
                                      $attribReservedInstanceNrs, $num);
                }
            }
            $builtin->print ("</Instances>\n");
            $builtin->print ("</System>\n");
        }
        $builtin->print ("</$SAPSystemsTag>\n");
    }

    if ($collect{$networkAddressesTag}) {
        # network interfaces
        $builtin->print ("<$networkAddressesTag>\n");
        my $sysinfo = new SDB::Install::SysInfo ();
        my $netInterfaces = $sysinfo->interfaceInfo ();
        foreach my $addresses (values %$netInterfaces){
            foreach my $ip (@$addresses){
                $builtin->print ("<ip>$ip->{addr}/$ip->{netbits}</ip>\n");
            }
        }
        $builtin->print ("</$networkAddressesTag>\n");
    }

    if ($collect{$saphostagentSslConfigTag}) {
        my $sslCiphersValue = getSHACiphersValue();
        $builtin->print("<$saphostagentSslConfigTag>$sslCiphersValue</$saphostagentSslConfigTag>\n");
    }

    if ($collect{$saphostagentVersionTag}) {
        my $agentVersion = getSHAVersion();
        if (!defined $agentVersion) {
            $agentVersion = '';
        }
        $builtin->print ("<$saphostagentVersionTag>$agentVersion</$saphostagentVersionTag>\n");
    }

    if ($collect{$certificateHostnameTag}) {
        my $certificateHostname = LCM::RegenerateCertificates::getCertOwner();
        $certificateHostname =~ s/CN=//;
        $builtin->print ("<$certificateHostnameTag><![CDATA[$certificateHostname]]></$certificateHostnameTag>\n");
    }

    if ($collect{$systemCheckErrorsTag} && $compilerVersion) {
        my $newDbConfig   = SDB::Install::Configuration::NewDB->new();
        my $sysinfo       = SDB::Install::SysInfo->new();
        require SDB::Install::Manifest;
        my $dummyManifest = SDB::Install::Manifest->new();
        $dummyManifest->{data}->{'compiler-version'} = $compilerVersion;
        $newDbConfig->checkSystemRequirements($sysinfo, $dummyManifest);
        my $errors = ${$newDbConfig->getErrMsgLst()->getMsgLstString()};
        if ($errors) {
            $builtin->print ("<$systemCheckErrorsTag>$errors</$systemCheckErrorsTag>\n");
        }
    }

	# data and log path existence
	if ($dataPath ne '') {
        my $sharedTmpFileExists = (defined $chkSharedDataPathFile && File::stat::stat($chkSharedDataPathFile)) ? 1 : 0;
        my $sharedTmpSubvolumeFileExists = $sharedTmpFileExists || (defined $chkSharedDataSubvolumeFile && File::stat::stat($chkSharedDataSubvolumeFile)) ? 1 : 0;
        my $msgLst = SDB::Install::MsgLst->new();
        my $sidadmCanAccess = $sharedTmpFileExists || canAccess($dataPath, $sidadmUID , $sapsysid, $msgLst);
        $builtin->printf ("<$dataPathTag isExists=\"%d\"  sidadmCanAccess=\"%d\" isSharedVolume=\"%d\" isSharedSubvolume=\"%d\" >\n", isDirectory($dataPath) ? 1 : 0, $sidadmCanAccess, $sharedTmpFileExists ,$sharedTmpSubvolumeFileExists);
        if(!$msgLst->isEmpty()){
            $builtin->print (${$msgLst->getMsgLstXML()});
        }
        $builtin->print ("</$dataPathTag>\n");
	}
	if ($logPath ne '') {
        my $sharedTmpFileExists = (defined $chkSharedLogPathFile && File::stat::stat($chkSharedLogPathFile)) ? 1 : 0;
        my $sharedTmpSubvolumeFileExists = $sharedTmpFileExists || (defined $chkSharedLogSubvolumeFile && File::stat::stat($chkSharedLogSubvolumeFile)) ? 1 : 0;
        my $msgLst = SDB::Install::MsgLst->new();
        my $sidadmCanAccess = $sharedTmpFileExists || canAccess($logPath, $sidadmUID , $sapsysid, $msgLst);
        $builtin->printf ("<$logPathTag isExists=\"%d\" sidadmCanAccess=\"%d\" isSharedVolume=\"%d\" isSharedSubvolume=\"%d\">\n", isDirectory($logPath) ? 1 : 0, $sidadmCanAccess, $sharedTmpFileExists, $sharedTmpSubvolumeFileExists);
	    if(!$msgLst->isEmpty()){
            $builtin->print (${$msgLst->getMsgLstXML()});
        }
        $builtin->print ("</$logPathTag>\n");
    }

    if ($chkSharedLssPathFile ne '') {
        my $sharedTmpFileExists = File::stat::stat($chkSharedLssPathFile) ? 1 : 0;
        $builtin->printf ("<$lssPathTag isShared=\"%d\"/>\n", $sharedTmpFileExists);
    }

    $builtin->print ("</HostDescription>\n");
    return 0;
}

sub getSHACiphersValue {
    my $hostProfilePath = File::Spec->catfile(getActiveSaphostexecDir(), 'host_profile');
    my $profile = SDB::Install::IniParser::SAPProfile->new($hostProfilePath);

    if (!$profile->read()) {
        return '';
    }
    my $value = $profile->getValue($cipherSuitesId);
    return defined ($value) ? $value : '';
}

sub shouldWarnIfCalledStandalone{
    return 0;
}

sub canAccess {
    my ($path, $uid, $gid , $msgLst ) = @_;
    return SDB::Install::System::access_check($path, 0 , $uid, $gid, $msgLst) ? 1 : 0;
}

1;
