#!/usr/bin/perl
#
# Desc: registers/unregisters a client installation
#


package SDB::Install::App::Console::HdbClientReg;

use base SDB::Install::App::Console;

use Getopt::Long;
use SAPDB::Install::Hostname;
use SDB::Install::Configuration::HdbClientReg;
use SDB::Install::ClientRegisterTools;
use SDB::Install::Log;

use SDB::Install::Globals   qw($gProductNameInstaller $gTmpDir);
use SDB::Install::System    qw(normalizePath);
use SDB::Install::SysVars   qw($isWin $path_separator);

use strict;

our $programName = $SAPDB::Install::Config{ProgramName};
our $programVers = '0001';

#-------------------------------------------------------------------------------

sub new {
    my $self = shift->SUPER::new ();
    return $self;
}

#-------------------------------------------------------------------------------
# Initializes this application
# Without parameters

sub InitApp {
    my ($self) = @_;
    $self->SUPER::InitApp();
    return 1;
}

#-------------------------------------------------------------------------------
# Initializes the program and performs several checks. 
# Without parameters
# Returns undef in case of an error

sub init {
    my ($self) = @_;
    if ($self->isHelp()){
        $self->PrintUsage();
        $self->{return} = 0;
        return undef;
    }
    if (!$self->checkIsAdmin($programName)){
        return undef;
    }
    my $msglst = $self->getMsgLst ();
    my $errlst = $self->getErrMsgLst ();
    my $instconfig = $self->getInstconfig();
    my $caption    = "$gProductNameInstaller - $programName ";
    $caption .= $self->GetInstaller->{version};
    $msglst->addProgressMessage ("\n\n" . $caption . "\n"
                               . ('*' x length($caption)) . "\n\n");
    $self->addStartDateMsg($msglst, "$programName  $programVers");
    my $msg = $msglst->addMessage ("Checking configuration");
    $msg->setEndTag ('Configuration Check');
    my $rc = $instconfig->CheckParams ($self->{batch_mode});
    if (!$rc && $instconfig->ErrorState ()){
        $self->ShowErrorMsg ('Configuration error:',$instconfig->getErrMsgLst ());
        $msg->getSubMsgLst ()->appendMsgLst ($instconfig->getMsgLst());
        return undef;
    }

    if (!$self->{batch_mode} && !defined $self->ConfigureInteractive ($instconfig)){
        return undef;
    }

    $msg->getSubMsgLst ()->appendMsgLst ($instconfig->getMsgLst()); 
    my $logFilename = $programName.'.log';
    my $msgFilename = $programName.'.msg';
    my $logUID      = $self->{options}->{instlog_uid};
    my $logGID      = $self->{options}->{instlog_gid};
    $self->{log}->addLogFileLocation(LOG_FORMAT_PLAIN,   $gTmpDir, 1, $logFilename, LOG_INDICATE_LOCATION_ON_STDOUT,      $logUID, $logGID);
    $self->{log}->addLogFileLocation(LOG_FORMAT_MSGLIST, $gTmpDir, 1, $msgFilename, LOG_DONT_INDICATE_LOCATION_ON_STDOUT, $logUID, $logGID);
    # try to detect the installation path via argv0 of installer:
    my $pattern = join (quotemeta ($path_separator),
        (qw ((.*) install)));
    my ($path) = ($SAPDB::Install::Config{RuntimeDir} =~ /$pattern/);
    if ($path) {
        if ($isWin){
            $path = normalizePath($path);
        }
    }
    else {
        $self->setErrorMessage("Could not determine installation path.");
        return undef;
    }
    my $doUnreg = $instconfig->getValue('Unregister');
    if(not $doUnreg) {
        $rc = registerClient($path, $msglst, $errlst);
        $self->{'whatWeHaveDone'} = "Installation '$path' successfully registered.";
    }
    else {
        $rc = unregisterClient($path, $msglst, $errlst);
        $self->{'whatWeHaveDone'} = "Installation '$path' successfully unregistered.";
    }
    return $rc;
}

#-------------------------------------------------------------------------------
# Initializes the arguments of the command line
#
# Parameters string-array arguments
#
# Returns a reference to a string array containing the arguments
#         or undef in case of an error

sub InitCmdLineArgs {
    my (
        $self,
        $args
    ) = @_;
    $self->{instconfig} = new SDB::Install::Configuration::HdbClientReg
                                        ($self->{options}, $self->{configfile});
    return $self->SUPER::InitCmdLineArgs ($args);
}

#===============================================================================
#   M A I N

sub main{
    my $app = new __PACKAGE__;
    my $rc;
    $app->{stackBacktraceMsglst} = new SDB::Install::MsgLst ();
    eval{
        if (!defined $app->InitCmdLineArgs (\@_)){
           return undef;
        }
        if (defined $app->{return}){
            LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
            return $app->{return};
        }
        $rc = $app->init ();

    };
    if (defined $app->{return}){
        LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
        return $app->{return};
    }
    my $msglst = $app->getMsgLst ();
    if ($@){
        my $signalInfoText;
        if ($@ =~ /^__SIGINT__/){
            $msglst->addMessage ('User canceled '.$programName.' with Ctrl + c.');
            $rc = 2;
        }
        elsif ($@ =~ /^__SIGPIPE__/){
            $app->setErrorMessage ('Broken pipe', $app->{stackBacktraceMsglst});
            $signalInfoText = $app->getSignalInfoText ();
            undef $rc;
        }
        elsif ($@ =~ /^__SIG(\w+)__/){
            $app->setErrorMessage ("Caught signal $1.", $app->{stackBacktraceMsglst});
            $signalInfoText = $app->getSignalInfoText ();
            undef $rc;
        }
        else{
            $app->setErrorMessage ('unhandled exception: '. $@, $app->{stackBacktraceMsglst});
            $rc = undef;
        }
        if ($signalInfoText){
            $app->appendErrorMessage ($signalInfoText);
        }
    }
    if (defined $rc){
        if ($rc == 2){
            $msglst->addProgressMessage($programName.' aborted');
        }
        else{
            $msglst->addProgressMessage($app->{'whatWeHaveDone'});
        }

    }
    else{
        $app->ShowErrorMsg ($programName.' failed.',$app->getErrMsgLst ());
    }
    $app->CleanUp ();
    undef $app;
    return defined $rc ? 0 : 1;
}

sub shouldWarnIfCalledStandalone{
    return 0;
}

1;
