
#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: installation application class

package SDB::Install::App::Console::Installation;

#
# configuration parameters 
#   - installation path or installation key
#   - mode (system/user/portable)
#   - kit directory
#   - os user 
#   - os group
#   - plain text logfile 


# ways of parameter initialization: implicit (upgrade), per command line option, per configfile, per default, interactive 

use SDB::Install::App::Console;
use Getopt::Long;
use SDB::Install::App::Installation;
use SDB::Install::Kit;
use SDB::Install::Tools qw (askConfirmation printTableToArrayOfLines);
use SDB::Install::Globals qw ($gProductNameInstaller $gProductNameStudio);
use SDB::Install::SysVars;
use SDB::Install::DebugUtilities;
use SDB::Install::DatabaseStudioSummary;
use SDB::Install::Utils::DatabaseStudio::DatabaseStudioKnownFeatures;
use strict;

our @ISA = qw (SDB::Install::App::Console SDB::Install::App::Installation);

sub new {
    my $self = shift->SUPER::new ();
    return $self;
}

sub InitApp{
    my ($self) = @_;
    $self->SUPER::InitApp();
    return 1;
}


sub InitCmdLineArgs{
    my ($self,$args) = @_;
    $self->{options} = {};
    my $rc = $self->SUPER::InitCmdLineArgs ($args,1);
    if (!defined $rc){
        return undef;
    }
    $rc = $self->ParseInstallationCmdLineArgs ($rc);
    
    if (!defined $rc){
        return undef;
    }

    if (!defined $self->init ()){
        return undef;
    }
    
    if (defined $self->{return}){
        return undef;
    }

    $rc = $self->ParseConfigCmdLineArgs ($rc);
    
    if ($self->isHelp()) {
        $self->PrintUsage();
        $self->{return} = 0;
        return 1;
    }

    if (!defined $rc){
        return undef;
    }

    if (!defined $self->checkSystemRequirements ()){
        return undef;
    }

    local @ARGV = @$rc;
    
    my %opt_ctrl;

    if (!$self->getOptions (\%opt_ctrl, 0)){
        $self->{return} = -1;
        return undef;
    }
}


#-------------------------------------------------------------------------------
# Returns a reference to an array containing hdbinst options
# without help/info options.

sub GetSwitches {

    return $_[0]->GetInstallationSwitches();
}


#-------------------------------------------------------------------------------
# Returns a reference to an array of arrays
# containg the description of batch and help options.

sub GetBatchHelpUsage {
    my ($self) = @_;

    my $superUsage = $self->SUPER::GetBatchHelpUsage();
    return $self->GetInstallationBatchHelpUsage($superUsage);
}

sub getHelpNotes{
	my ($self) = @_;
	if($self->{kit}->IsDBStudioDistribution () && !$self->{batch_mode}){
    	return "Note: ".ANSWERS_LEGAL_NOTE;
    }
    return '';
}

#-------------------------------------------------------------------------------
# Returns a reference to an array of arrays containg the description
# of hdbinst options without common batch/help options.

sub GetUsage{
    my $list = $_[0]->SUPER::GetUsage ();
    push @$list, @{$_[0]->GetInstallationUsage()};
    return $list;
}


#-------------------------------------------------------------------------------
# Returns a reference to an array containing help/info options of the program.

sub GetHelpSwitches{
    my ($self, $indent)= @_;
    return $self->GetInstallationHelpSwitches($indent);
}


sub SelectPackages{
    my ($self,$update_inst) = @_;
    
    my $inst = $update_inst;
    
    if (defined $update_inst){
        if (!defined $update_inst->GenPackageList()){
            $self->{errlst}->addError ("cannot open installation " . $update_inst->GetName (), $update_inst->getErrMsgLst());
            return undef;
        }
    }
    
    if (!defined $self->{kit}->GenerateTree ($inst)){
        $self->{errlst}->addError ("error in installation kit", $self->{kit}->getErrMsgLst());
        return undef;
    }
    
    my $selected = $self->SelectSoftwareComponentsByCmdLineArguments();
    
    if (!defined $selected){
        return undef;
    }
    
    if ($selected){
        return 1;
    }
    
    if (defined $update_inst){
        $self->{kit}->SelectPackagesByInstallation ($update_inst);
    }
    else{
        #if ($self->{batch_mode}){
            $self->{kit}->SelectAllPackages();
        #}
        #else{
        #   $self->SelectSoftwareComponentsInteractively ();
        #}
        
    }
    return 1;
}

sub configure {
    my ($self) = @_;

    $self->warnIfShouldWarnIfCalledStandalone();
    my $caption = sprintf ('%s - %s Installation %s', $gProductNameInstaller,
        $self->{instconfig}->getShortProductName(), $self->{instconfig}->getProductVersion());

    $self->{msglst}->addProgressMessage ("\n\n".$caption . "\n" . ('*' x length ($caption)) . "\n\n");
    $self->addStartDateMsg($self->{msglst}, $self->getProgramName());
    
    my $instconfig = $self->{instconfig};

    if (!defined $instconfig){
        return 1;
    }

    if (!defined $self->SelectPackages ()){
        return undef;
    }
    $instconfig = $self->{instconfig};

    if (!exists $instconfig->{_kitversion} ||
       ($instconfig->{_kitversion} ne $self->{kit}->GetVersion())) {

        # InitDefaults has not been already called
        # by SDB::Install::App::Installation::initInstConfig

        if (!$instconfig->InitDefaults ($self->{kit})){
            $self->{errlst}->addError ("Initialize configuration failed", $instconfig->getErrMsgLst());
            return undef;
        }
    }


    my $msg = $self->{msglst}->addMessage ("Checking configuration of installation");
    $msg->setEndTag('Configuration Check');
    $instconfig->setMsgLstContext ([$msg->getSubMsgLst ()]);
    my $rc = $instconfig->CheckParams ($self->{batch_mode});

    if ($instconfig->getValueOrBatchValue('CheckOnly')) {
        $self->{checkOnly} = 1;
        $self->{msglst}->addProgressMessage("Installation: check only");
    }

    if($self->{batch_mode}){
        if (!$rc){
            if (defined $rc){
                $self->{errlst}->addError ("running in batch mode", $instconfig->getErrMsgLst());
                #$self->PrintUsage ();
            }
            else{
                $self->{errlst}->addError ('Configuration error (batch mode):', $instconfig->getErrMsgLst());
            }
            $msg->endMessage ();
            return undef;
        }
    }
    else{
        if (!$rc && $instconfig->ErrorState ()){
            if (!defined $rc){
                $self->{errlst}->addError ('Configuration error:',$instconfig->getErrMsgLst());
                $msg->endMessage ();
                return undef;
            }
            $self->ShowErrorMsg ('Configuration error:',$instconfig->getMsgLst());
        }
        if (!$instconfig->isInstallationSelected ()){
	        if (!defined $self->selectInstallationForUpdate ($instconfig)) {
                $msg->endMessage ();
	        	return undef;
	        }
	    }
        my $saveMsgLst = $self->getMsgLst();
        my $icfgMsg = $msg->getSubMsgLst()->addMessage ('Configuring interactively');
        $self->{msglst} = $icfgMsg->getSubMsgLst();
        if (!defined $self->ConfigureInteractive ($instconfig)){
            $msg->endMessage ();
            $self->{msglst} = $saveMsgLst;
            return undef;
        }
        $self->{msglst} = $saveMsgLst;
    }
    $msg->endMessage ();
    $instconfig->setMsgLstContext ([$self->getMsgLst ()]);

    if ($self->{kit}->IsDBStudioDistribution () && !$self->{batch_mode} && !$self->{options}->{noprompt}){
        my $summaryTree = SDB::Install::DatabaseStudioSummary::buildSummaryTree($self->{kit}, $instconfig);
        if ($summaryTree) {
            printSummary( $summaryTree );
            $self->printWarnings();
            print "\n";
            my $userConfirms = askConfirmation();
            print "\n";
            if ( !$userConfirms ) {
            	my $action = $instconfig->{actionProgressive};
                $self->setErrorMessage("$action $gProductNameStudio was canceled by the user");
                return undef;
            }                    	
        }
    }
    elsif (!$self->{kit}->IsDBStudioDistribution()
                    && $instconfig->isa('SDB::Install::Configuration::NewDB')) {
        $instconfig->displayParameterSummary();
        $self->printWarnings ();
        if (!$self->{batch_mode} && !$instconfig->getValue('CheckOnly')
                       && !$self->{options}->{noprompt} && !askConfirmation()) {
            $self->{msglst}->addProgressMessage('Cancelled');
            $self->{return} = 0;
            return 1;
        }
    }
    
    return 1;

}

sub printSummary {
    my ( $summaryTree ) = @_;

    my $headline = 'Summary before execution:';
    print "\n$headline\n" . ('=' x length($headline)) . "\n\n";


    _printNode( $summaryTree, 0 );
}

sub _printNode {
    my ( $nodeRef, $indent ) = @_;

    print " " x ( $indent * 3 ) . $nodeRef->{"text"} . "\n";
    my $childrenRef = $nodeRef->{"children"};
    if ( !defined $childrenRef ) {
        return;
    }
    foreach my $childRef ( @{$childrenRef} ) {
        _printNode ( $childRef, $indent + 1 );
    }
}

sub selectInstallationForUpdate{
	my ($self,$instconfig) = @_;
	
	my $installations = $instconfig->enumExistingInstallations ();
	
	my $productname = $instconfig->getProductName ();
	
	if (!defined $installations){
		$self->{errlst}->addError (undef , $instconfig->getErrMsgLst());
		return 1;
	}
		
	# no installations found - straight on to querying the path
	if (!(keys %$installations)){
		return 1;
	}
	
	my @tab;
	my $id = 1;

	my $modeAvailable = 0;
	my $default;
	my $defaultNo;
	my $date;
	my @stat;
	
	my $configDefault = $instconfig->get_InstallationPathDefault ();
	
    foreach my $p (keys %$installations){
        if ($installations->{$p}->{mode}) {
        	$modeAvailable = 1;
        }
        if (!$installations->{$p}->{canUpdate}) {
        	next;
        }
    	@stat = stat ($p);
    	if (@stat){
    		if (!defined $date || $date < $stat[10])   {
    			$date = $stat[10];
    			$default = $p;
    		}
      	}
        # do not suggest default path for a new installation if it is already used by an existing installation
        if ($p eq $configDefault) {
            $self->{instconfig}->set_InstallationPathDefault (undef);
		}
    }

	if ($modeAvailable) {
		push @tab, ['No', 'Installation Path', 'Version', 'Mode'];
	} else {
		push @tab, ['No', 'Installation Path', 'Version'];
	}

	my $id_str;
    foreach my $path (sort keys %$installations){
    	if ($path eq $default) {
    		$defaultNo = $id;
    		$id_str = sprintf ('[%d]', $id++);
    	}
    	else {
    		$id_str = ' '.$id++;
    	} 
		if ($modeAvailable) {
    	    push @tab, [$id_str, $path, $installations->{$path}->{version}, $installations->{$path}->{mode}];
		} else {
    	    push @tab, [$id_str, $path, $installations->{$path}->{version}];
		}
    }
    if (not defined $defaultNo) {
   		$defaultNo = $id;
   		$id_str = sprintf ('[%d]', $id);
   	}
   	else {
   		$id_str = ' '.$id;
   	} 
        
	my $kitVersion = $self->{kit}->GetVersion();
	if ($modeAvailable) {
		my $kitObjMode = $self->{kit}->is64bit() ? '64bit' : '32bit';
		push @tab, [' ', ' ', ' ', ' '];
	    push @tab, [$id_str, "Install new $productname", "$kitVersion", "$kitObjMode"];
	} else {
   	    push @tab, [' ', ' ', ' '];
	    push @tab, [$id_str, "Install new $productname", "$kitVersion"];
	}
    $self->{msglst}->addProgressMessage ("Select a $productname installation:");
    print "\n";
      
    foreach my $line (@{printTableToArrayOfLines (\@tab, ' | ',1)}){
        $self->{msglst}->addProgressMessage ($line);
    }
    print "\n";
        
    my $result;
    for (;;){
        print "Enter number [$defaultNo]: ";
        $result = $self->readInput();
        if ($result eq '') {
        	$result = $defaultNo;
        }
        if ($result =~ /\D/ or $result !~ /\d/ or int ($result) > $id or int ($result) == 0){
            next;
        }
	    if ($id != $result){
	        # we are a sw update:
	        if (!$instconfig->set_InstallationPath ($tab[$result]->[1])){
	    		my $r_errtxt = $instconfig->getErrMsgLst()->getMsgLstString ();
	    		$$r_errtxt =~ s/^[A-Z]+:\s+//mg;
	    		print $$r_errtxt . "\n";
	    		$instconfig->ResetError();
	    		next;
	    	}
	        $instconfig->setUpdateParams();
	    }
	    last;
    }
    return 1;	
}

#-------------------------------------------------------------------------------
# If the Kit's eventhandler returns a defined ref to an array of strings, the entries are printed
# on stdout, on per line. These are the last lines to appear on stdout before the log file location 
# is indicated and stdout is closed.

sub PrintFinalMessage{
    my (
        $self,
        $msglst
    )= @_;
    if( (defined $self->{'kit'}) && (defined $msglst) ) {
    	my $eventHandler = $self->{'kit'}->getEventHandler();
    	if(defined $eventHandler) {
            my $finalMessageArray = $eventHandler->getFinalMessage();
            if(defined $finalMessageArray) {
                foreach my $line (@{$finalMessageArray}) {
                    $msglst->addProgressMessage($line);
                }
            }
        }
    }
    return 1;
}

#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------

sub main{
    my $app = new __PACKAGE__;
    
    my $rc = $app->InitCmdLineArgs (\@_);
    my $msglst = $app->getMsgLst ();
    my $errlst = $app->getErrMsgLst ();
    if (defined $app->{return}){
        if (!defined $rc && !$errlst->isEmpty()){
            $app->ShowErrorMsg (undef, $errlst);
        }
        else {
            LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
        }
        if ($isWin && !$app->{batch_mode} && !$app->{options}->{noprompt}) {
            require Term::ReadKey;
            print "\npress any key...";
            Term::ReadKey::ReadMode('raw');
            Term::ReadKey::ReadKey();
        }
        return $app->{return};
    }
    if (!defined $rc){
        $app->setErrorMessage (undef, $errlst);
    }
    else{
        $app->{stackBacktraceMsglst} = new SDB::Install::MsgLst ();
        eval{

            $rc = $app->configure ();
            if (defined $app->{return}){
                LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
                return $app->{return};
            }
            if (defined $rc){
                $rc = $app->preinstall($app->{checkOnly});
                if (!$app->{checkOnly}) {
                    if (defined $rc){
                        $rc = $app->install ();
                        if (!defined $rc){
                            $app->setErrorMessage ('error installing', $errlst);
                        }
                    }
                    else{
                        $app->setErrorMessage ('Error checking installation', $errlst);
                    }
                }
            }
            else{
                $app->setErrorMessage (undef, $errlst);
            }
        };

        if ($@){
            my $signalInfoText;
            if ($@ =~ /^__SIGINT__/){
                $msglst->addMessage ('User canceled installation with Ctrl + c');
                $rc = 2;
            }
            elsif ($@ =~ /^__SIGPIPE__/){
                $app->setErrorMessage ('Broken pipe', $app->{stackBacktraceMsglst});
                $signalInfoText = $app->getSignalInfoText ();
                undef $rc;
            }
            elsif ($@ =~ /^__SIG(\w+)__/){
                $app->setErrorMessage ("Caught signal $1", $app->{stackBacktraceMsglst});
                $signalInfoText = $app->getSignalInfoText ();
                undef $rc;
            }
            else{
                $app->setErrorMessage ('unhandled exception: '. $@, $app->{stackBacktraceMsglst});
                $rc = undef;
            }
            if ($signalInfoText){
                $app->appendErrorMessage ($signalInfoText);
            }
        }
    }

    if (defined $rc){
        if ($rc == 2){
            print "\n";
            $msglst->addProgressMessage('Installation aborted');
        }
        elsif ($app->{checkOnly}) {
            $msglst->addProgressMessage('Installation check done');
        }
        elsif (!defined $app->{return}) {
            $msglst->addProgressMessage('Installation done');
        }
        $app->PrintFinalMessage($msglst);
    }
    else{
        $app->ShowErrorMsg ('Installation failed',$app->getErrMsgLst());
    }
    $app->SDB::Install::App::Installation::CleanUp();

    if ($isWin && !$app->{batch_mode} && !$app->{options}->{noprompt}) {
        undef $app;
        SDB::Install::App::Console->waitForKeyPress();
    }

    if (defined $app->{return}){
        LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
        return $app->{return};
    }

    undef $app;

    return defined $rc ? 0 : 1; 
}

sub shouldWarnIfCalledStandalone{
    return 1;
}

1;
