package SDB::Install::App::Console::LSS::AddHostLss;
use parent qw(SDB::Install::App::Console);

use strict;
use warnings;

use Getopt::Long;
use LCM::DevelopmentTrace;
use SDB::Install::Globals qw($gProductNameLSS $gProductName);
use SDB::Install::LSS::LssInstance;
use SDB::Common::Utils qw(getSidcryptName);
use SDB::Install::LSS::Installer;
use SDB::Install::Configuration::LSSConfiguration;
use SDB::Install::SAPSystem qw(CollectSAPSystems);

sub new {
    my $self = shift->SUPER::new ();
    return $self;
}

sub getAction {
    return "Configure";
}

sub getActionScope {
    return $gProductNameLSS;
}

sub getActionDone {
    return "configured";
}

sub getActionProgressive {
    return "Configuring";
}

sub getProgramName {
    return 'lssaddhost';
}

sub createConfiguration {
    my ($self) = @_;
    my $config = SDB::Install::Configuration::LSSConfiguration->new($self->{options});
    $config->defineLSSAddHostParams();
    $config->setMsgLstContext($self->getMsgLstContext());

    $self->setInstconfig($config);
}

sub InitCmdLineArgs{
    my $self = shift();
    $self->createConfiguration();

    return $self->SUPER::InitCmdLineArgs (@_);
}

sub init {
    my ($self) = @_;

    my $installer = SDB::Install::LSS::Installer->new();
    my $sid = $installer->getSid();

    if(!$self->defineLog($self->getProgramName(), $self->getProgramName(), undef, $sid)){
        return undef;
    }
    return undef if(!$sid);

    my $configuration = $self->getInstconfig();
    if (!$configuration->CheckParams(1)) {
        $self->setErrorMessage("Failed to check configuration", $configuration->getErrMsgLst());
        return undef;
    }

    return $self->_configureHost($installer);
}

sub _configureHost {
    my ($self, $installer) = @_;
    my $sid = $installer->getSid();
    my $sidcrypt = getSidcryptName($sid);
    my $lssInstance = $installer->getLssInstance();
    return undef if (!$lssInstance);

    $self->_adaptMessagesContext("Creating needed $sidcrypt OS entities and preparing filesystem layout for $gProductNameLSS...", $lssInstance);

    my $configuration = $self->getInstconfig();
    my $uid = $configuration->getValue('LSSUserID');
    my $gid = $configuration->getValue('LSSGroupID');
    my $homeDir   = $configuration->getValue('LSSUserHomeDir');
    my $userShell = $configuration->getValue('LSSUserShell');
    my $userPassword = $configuration->getValue('LSSPassword');
    if (!$lssInstance->configureHost($uid, $userPassword, $userShell, $homeDir, $gid)) {
        $self->getErrMsgLst()->addError("Failed to configure host for $sidcrypt user");
        return undef;
    }

    $self->_adaptMessagesContext("Configuring $sidcrypt user home directory...", $lssInstance);
    if (!$lssInstance->initialize(undef, 1)) {
        $self->getErrMsgLst()->addError("Failed to configure $sidcrypt user home dir");
        return undef;
    }

    if ($configuration->getValue('RestorePSE')) {
        $self->_adaptMessagesContext("Restoring $gProductNameLSS keys...", $lssInstance);
        my $backupPassword = $configuration->getValue('LSSBackupPassword');
        return undef if !$lssInstance->getLssCfg()->restorePSE($backupPassword);
    }

    return 1;
}

sub _adaptMessagesContext {
    my ($self, $message, $base) = @_;
    my $msg = $self->getMsgLst()->addProgressMessage($message);
    $base->setMsgLstContext([$msg->getSubMsgLst(), $msg->getSubMsgLst()]);
    return $msg;
}

 sub main {
    my $app = new __PACKAGE__;
    $app->mainFunction(\@_);
}

sub shouldWarnIfCalledStandalone {
    return 1;
}

sub CleanUp{
    my ($self) = @_;
    my $installer = SDB::Install::LSS::Installer->new();
    my $sid = $installer->getSid();
    my $systems = CollectSAPSystems();
    my $sapSystem = $systems->{$sid};

    if (defined $sapSystem) {
        $self->addLogToTraceDir($sapSystem, $self->getProgramName());
    }
    $self->SUPER::CleanUp();
}

1;