package SDB::Install::App::Console::LSS::CheckInstallationPath;

use parent qw(SDB::Install::App::Console);

use strict;
use warnings;

use Getopt::Long;

use SDB::Install::Globals qw($gProductNameLSS $gOperationFileExists);
use SDB::Install::RemoteHostctrlHosts;
use SDB::Install::RemoteHosts;
use SDB::Common::BuiltIn;
use SDB::Install::Configuration::LSSConfiguration;

sub new {
    my $self = shift->SUPER::new ();
    return $self;
}

sub getAction {
    return "CheckLSSPath";
}

sub getActionScope {
    return $gProductNameLSS;
}

sub getActionDone {
    return "installation path checked";
}

sub getActionProgressive {
    return "Checking installation path of";
}

sub getProgramName {
    return 'lsspathcheck';
}

sub createConfiguration {
    my ($self) = @_;
    my $config = SDB::Install::Configuration::LSSConfiguration->new($self->{options});
    $config->defineCheckInstallationPathParams();
    $config->setMsgLstContext($self->getMsgLstContext());

    return $config;
}

sub InitCmdLineArgs {
    my $self = shift;
    my $config = $self->createConfiguration();
    $self->setInstconfig($config);
    return undef if(!$self->SUPER::InitCmdLineArgs (@_));

    if (!$config->CheckParams(1)) {
        $self->setErrorMessage("Failed to check configuration", $config->getErrMsgLst());
        return undef;
    }
    return 1;
}

sub checkPathAsRoot {
    my ($self, $tmpFileName , $remoteHostNames) = @_;
    my $remoteHosts = SDB::Install::RemoteHosts->new(@$remoteHostNames);
    my $config = $self->getInstconfig();
    my $remoteUser = $config->getValue('RootUser');
    if(!$remoteUser){
        $self->setErrorMessage("No remote user name given via --root_user");
        return undef;
    }
    $remoteHosts->setUserName($remoteUser,0);

    if ($remoteHosts->connect () != 0){
        $self->setErrorMessage("Failed to connect to remote hosts", $remoteHosts->getErrMsgLst());
        return undef;
    }
        #try key authorization
    if ($remoteHosts->authKey(1) != 0 ){
        #try with given password
        my $password = $config->getPasswordFromXML('root_password', undef, 1);
        if(!$password){
            $self->setErrorMessage("No password given and no public key exists for user '$remoteUser'");
            return undef;
        }
        $remoteHosts->setUserPass($password);
        if($remoteHosts->authPassword() != 0 ){
            $self->setErrorMessage("Failed to authenticate on the remote hosts", $remoteHosts->getErrMsgLst());
            return undef;
        }
    }

    my $hostFailuresInfo = [];
    my $rc = $remoteHosts->existsFile($tmpFileName, undef, $hostFailuresInfo);
    if(defined($rc) && !$rc){
        if (@{$hostFailuresInfo}) {
            $self->appendErrorMessage($self->_constructHostFailureMessage(pop @{$_})) for @{$hostFailuresInfo};
        } else {
            $self->appendErrorMessage(undef, $remoteHosts->getErrMsgLst());
        }
        return undef;
    }

    return 1;
}


sub checkPathAsSidadm {
    my ($self, $tmpFileName , $remoteHostNames) = @_;

    my $cfg = $self->getInstconfig();
    my $sid = $cfg->getValue('SID');
    if(!$sid){
        $self->setErrorMessage("No sid given via --sid");
        return undef ;
    }

    my $password = $cfg->getPasswordFromXML('password', undef, 1); # 1 - not verbose
    if(!$password){
        $self->setErrorMessage("No password given for '".lc($sid)."adm'");
        return undef;
    }

    my $remoteHosts = SDB::Install::RemoteHostctrlHosts->new(@$remoteHostNames);
    $remoteHosts->setCredentials($cfg->getSysAdminUserName($sid), $password);

    my $rc = $remoteHosts->executeHostctrlParallel( $gOperationFileExists,
                                    $cfg, # instconfig
                                    undef, # param IDs
                                    undef, # password IDs
                                    undef, # remote ignore
                                    undef, # remote timeout
                                    undef, #progress mssage
                                    undef, #done message
                                    $self->_constructHostFailureMessage(), #error msg
                                    {   'SID'  => $sid,
                                        'PATH' => $tmpFileName
                                    },
                                    undef, # host option map
                                    $remoteHostNames, # only on hosts
                                    0, # do not fail
                                    1, # suppresses console output
                                    0); # isCollectHostInfo
    if($rc != 0 ){
# Here we append the messages from the msgLst and errMsgLst in a separate manner because we want
# to add only the relevant messages from the errMsgLst of the remoteHosts object, namely the ones,
# which indicate on which host(s) the path, which we check, cannot be accessed.
        $self->getMsgLst()->appendMsgLst($remoteHosts->getMsgLst());
        $self->appendErrorMessage($_->getRawText()) for @{$remoteHosts->getErrMsgLst()->getMessages()};
        return undef;
    }
    return 1;
}

sub checkFileExists {
    my ($self, $tmpFileName , $remoteHostNames) = @_;
    my $success = 1;
    if($self->{options}->{called_by_hostagent}){
        $success = $self->checkPathAsSidadm($tmpFileName , $remoteHostNames);
    } else {
        $success = $self->checkPathAsRoot($tmpFileName , $remoteHostNames);
    }
    return $success;
}

sub _constructHostFailureMessage {
    my ($self, $host) = @_;
    my $config = $self->getInstconfig();
    my $path = $config->getValue('Path');
    if ($host) {
        return sprintf("'%s' is not accessible on host %s", $path, $host);
    } else {
        return sprintf("'%s' is not accessible on host %%s", $path);
    }
}

sub init {
    my ($self) = @_;
    my $cfg = $self->getInstconfig();
    my $hostnamesValue = $cfg->getValue('HostNames');
    my $remoteHostNames = $hostnamesValue ? [ split(',', $hostnamesValue) ] : undef;

    if(!$remoteHostNames) {
        $self->setErrorMessage("No remote hosts specified via --hostnames");
        return undef;
    }

    my $tmpFileName = File::Spec->catfile($cfg->getValue('Path'), time());
    my $tmpFile = IO::File->new(">$tmpFileName");
    if (!$tmpFile){
        $self->setErrorMessage("Cannot create temporary file '$tmpFileName': $!");
        return undef;
    }

    my $success = $self->checkFileExists($tmpFileName , $remoteHostNames);

    if(!SDB::Common::BuiltIn->get_instance()->unlink($tmpFileName)){
        $self->getMsgLst()->addWarning("Cannot delete temporary file '$tmpFileName': $!");
    }

    return $success;
}

sub InitTrace {
# Disable development trace
    return 1;
}

sub main {
    my $app = __PACKAGE__->new();
    $app->mainFunction(\@_);
}

sub shouldWarnIfCalledStandalone {
    return 0;
}

1;