package SDB::Install::Configuration::Generic;

use strict;
use SDB::Install::Configuration::AnyConfig;
use SDB::Install::SysVars;
use SDB::Install::DebugUtilities;
use SDB::Install::Installation::Generic;
use SDB::Install::Persistence::XMLGenerator;
use SDB::Install::Log;
use SDB::Install::Globals qw($gSAPLocation);

our $installationClass = 'SDB::Install::Installation::Generic';
our $kitClass = 'SDB::Install::Kit::Generic';
our $customConfigurationClass;
our $updateFiles = {};

our $installParamsXml = 'InstallParams.xml';
our $installCfgDtd    = 'installcfg.dtd';

our $STEP_EXTRACT      = 7;
our $STEP_EXTRACT_NAME = 'Extracting Software';

our $STEP_POSTINSTALL      = 15;
our $STEP_POSTINSTALL_NAME = 'Calling \'postInstall\' Event Handler';

our @ISA = qw (SDB::Install::Configuration::AnyConfig);

sub new{
    my $self = shift->SUPER::new (@_);
    $self->{params} = {};
    return $self;
}

sub isAdminRequired{
    # enable non-root installation on Linux
    return $isWin;
}

sub enumExistingInstallations{
    # simulate absence of installations
    #return {};

    my ($self) = @_;
    my $errlst = new SDB::Install::MsgLst();
    my $compId = $self->{kit}->getCompId();
    my $installations = SDB::Install::Installation::Generic::EnumGenericInstallations ($compId, $errlst);
    if (!defined $installations){
        $self->AddError (undef, $errlst);
        return undef;
    }
    my ($installation, $version, $objMode, $platform);
    my $result = {};
    foreach my $path (keys (%$installations)){
        $installation = new SDB::Install::Installation::Generic ($path);
        $version = $installation->getVersionByManifest();
        if ($isWin || $isLinux){
            $objMode = $installation->is64bit() ? '64bit' : '32bit';
        }

        $result->{$path} = {
            'version' => (defined $version ? $version : $installations->{$path}),
            'mode' => $objMode,
            'canUpdate' => $self->{kit}->canUpdate ($installation, $self)
        };
    }
    return $result;
}

sub InitDefaults{
    my ($self, $kit) = @_;
    $self->{_kitversion} = $kit->GetVersion;
    $self->{_kitplatform} = $kit->getPlatform ();
    $self->{kit} = $kit;
    my $manifest = $kit->getManifest ();
    if (defined $manifest){
        $self->{compKey} = $manifest->getKeyname ();
        $self->_addPhaseParameter ($manifest);
        $self->{_manifestKeyCaption} = $manifest->getValue ('keycaption');
    }
    return 1;
}


sub _addPhaseParameter{
    my ($self, $manifest) = @_;
    if (!defined $manifest){
        return undef;
    }
    my $phases = $manifest->getSupportedPhases ();

    if (!@$phases){
        return 1;
    }
    $self->{_phases} = $phases;
    $self->{params}->{Phase} = $self->getParamPhase (0,$self->{shortProductName}, $phases);
    $self->clearCachedParamIds ();
    $self->registerResumableSteps ();
    return 1;
}


sub getNextPhase{
    my ($self, $phase) = @_;
    $phase //= $self->getPhase();
    return undef if (!defined $phase || !defined $self->{_phases});

    my $phases = $self->{_phases};
    foreach my $i (0..$#$phases){
        if ($phases->[$i] eq $phase){
            return $phases->[$i + 1];
        }
    }
    return undef;
}


sub registerResumableSteps{
    my ($self, $stepNames) = @_;

    $self->{step_names} = {
        $STEP_EXTRACT     => $STEP_EXTRACT_NAME,
        $STEP_POSTINSTALL => $STEP_POSTINSTALL_NAME
    };

    if (!defined $stepNames){
         return 1;
    }

    $self->{_ownStepsRegistered} = 1;
    my $lastId = $STEP_POSTINSTALL;
    my %names = reverse %{$self->{step_names}};
    my $rc = 1;
    foreach my $stepName ($stepNames){
        if (exists $names{$stepName}){
            $self->appendErrorMessage ("Step '$stepName' already with id '$names{$stepName}' registered");
            $rc = 0;
            next;
        }
        $names{$stepName} = ++$lastId;
        $self->{step_names}->{$names{$stepName}} = $stepName;
    }
    $self->{step_ids} = \%names;
    return $rc;
}


sub enterResumableStepById{
    my ($self, $step) = @_;
    if (!defined $self->pers_filename()){
        return 1;
    }
    if ($self->{step} <= $step){
        $self->setStep ($step);
        return 1;
    }
    return 0;
}

sub enterExtractSoftwareStep{
    return $_[0]->enterResumableStepById ($STEP_EXTRACT);
}

sub enterPostInstallStep{
    return $_[0]->enterResumableStepById ($STEP_POSTINSTALL);
}

sub pers_filename{
    my ($self) = @_;
    if (!defined $self->{_phases} && !$self->{_ownStepsRegistered}){
        return undef;
    }
    if (!defined $self->{_path}){
        return undef;
    }
    return $self->{_path} . $path_separator . lc ($self->{productKey}) . '.hdbinstall';
}

sub writeHdblcmStatusFile{
    my ($self, $useCurrentPhase) = @_;
    my $fileName = $self->getHdblcmStatusFileName ();
    if (!defined $fileName){
        return 1;
    }

    if (!defined $useCurrentPhase){
        $useCurrentPhase = 0;
    }

    my $generator = new SDB::Install::Persistence::XMLGenerator ();
    $generator->setSourceVersionString($self->{_start_version});
    $generator->setTargetVersionString($self->{_kitversion});
    my $dateTime = $self->pers_date_string();
    $dateTime =~ s/[\+-][^\+-]+$//;
    $generator->setCreationDateString ($dateTime);
    my $phase = $self->getPhase ();
    if (defined $phase){
        if (!$useCurrentPhase){
            $phase = $self->getNextPhase ($phase);
        }
        if (!defined $phase){
            # last phase reached
            return 1;
        }
        $generator->setNextPhaseString($phase);
    }
    if (defined $self->{_manifestKeyCaption}){
        $generator->setComponentNameString ($self->{_manifestKeyCaption});
    }
    $generator->setActionString ($self->isUpdate() ? 'update' : 'install');
    $generator->setCurrentStepString ($self->pers_getstepname ($self->{step}));


    my $msg = $self->getMsgLst()->addMessage ("Writing hdblcm status file '$fileName'");
    my $msglst = $msg->getSubMsgLst();
    $msglst->addMessage ("next phase = $phase");
    $msglst->addMessage ("target version = $self->{_kitversion}");
    if (defined $self->{_start_version}){
        $msglst->addMessage ("start version = $self->{_start_version}");
    }

    my $xmlString = $generator->generatePersistenceXMLString();

    if (!defined $xmlString){
        $self->appendErrorMessage ("Cannot create file '$fileName'", $generator->getErrMsgLst());
        return undef;
    }

    if (!open (FD, '>'.$fileName)){
        $self->appendErrorMessage ("Cannot create file '$fileName': $!");
        return undef;
    }
    print FD $xmlString;
    close (FD);
    return 1;
}

sub getHdblcmStatusFileName{
    my ($self, $globalSidDir) = @_;
    my $productKey = $self->{productKey};
    if (!defined $productKey){
        return undef;
    }
    if (!defined $globalSidDir){
        my $sapSys = $self->getSAPSystem ();
        if (!defined $sapSys){
            $productKey = lc ($productKey);
            my $fileName = File::Spec->catfile($gSAPLocation, $self->getValue('SID').'.pending_' . $productKey . '.xml');
            if (! -f $fileName){
                return undef;
            }
            return $fileName;
        }
        $globalSidDir = $sapSys->get_globalSidDir ();
        if (!defined $globalSidDir || !-d $globalSidDir){
            return undef;
        }
    }
    $productKey = lc ($productKey);
    return $globalSidDir . $path_separator . 'pending_' . $productKey . '.xml';
}

sub removeHdblcmStatusFile{
    my ($self, $globalSidDir) = @_;
    my $fileName = $self->getHdblcmStatusFileName ($globalSidDir);
    if (defined $fileName && -f $fileName){
        $self->getMsgLst()->addMessage ("Removing hdblcm status file '$fileName'");
        if (!unlink ($fileName)){
            $self->setErrorMessage ("Cannot remove file '$fileName': $!");
        }
    }
    return 1;
}

sub isStepResumable{
    my ($self, $step) = @_;
    if (!defined $step){
        if (!defined $self->{step}){
            return 0;
        }
        $step = $self->{step};
    }
    if (keys (%{$self->{step_names}}) > 2){
        if (defined $self->{step_names}->{$step}){
            return 1;
        }
        return 0;
    }
    return $step == $STEP_EXTRACT;
}


sub pers_store{
	my $self = shift;
    my $rc = $self->SUPER::pers_store (@_);
    if ($self->isStepResumable()){
        $self->writeHdblcmStatusFile (1);
    }
    else{
        $self->removeHdblcmStatusFile ();
    }
    return $rc;
}

sub pers_remove{
	my $self = shift;
    my $rc = $self->SUPER::pers_remove (@_);
    $self->removeHdblcmStatusFile ();
    return $rc;
}

sub pers_keys{
	my ($self) = @_;
	my @pers_keys = (
                    '_kitversion',
                    '_start_version',
                    'step');

    my $phase = $self->getValue('Phase');
	if (defined $phase) {
		push @pers_keys, 'next_phase';
		$self->{'next_phase'} = $phase;
	}
	return \@pers_keys;
}

sub pers_getstepname{
    my ($self, $step) = @_;
    return $self->{step_names}->{$step};
}


sub isLastPhase{
    my ($self, $phase) = @_;
    $phase //= $self->getPhase();
    return undef if (!defined $phase || !defined $self->{_phases});
    return (!defined $self->getNextPhase($phase));
}


sub validatePersistency{
	my ($self, $pers) = @_;
	if (!defined $pers){
		$pers = $self->pers_load ();
	}
	if (!defined $pers){
		return undef;
	}

	my $ignore_pending_installation = $self->getIgnore ('check_pending_installation');

    my $operation;
    if (defined $pers->{_start_version}){
        $operation = 'upgrade';
    }
    else{
        $operation = 'installation';
    }

	if ($pers->{_kitversion} ne $self->{_kitversion}){
		$self->AddError ("Cannot resume failed $operation: wrong version ($self->{_kitversion})");
		$self->PushError ("$self->{productName} installation kit '$pers->{_kitversion}' required");
		if ($ignore_pending_installation || $self->getIgnore ('check_version')){
			$self->ResetError ();
			$self->AddMessage ("Ignoring error due to command line switch '--ignore'");
		}
		else{
			return undef;
		}
	}

	if ($ignore_pending_installation){
		$self->AddMessage ("Ignoring pending upgrade due to --ignore command line switch and starting a new upgrade");
        $self->pers_remove ();
		return 1;
	}

    if (!defined $self->SUPER::validatePersistency ($pers)){
		return undef;
	}

    my $pendingMsg = "Resuming broken $operation at step '"
                   . $self->pers_getstepname($pers->{step}) . "'.";

    if (defined $self->getBatchValue('Phase')) {
        $self->getMsgLst()->addMessage($pendingMsg);
    } else {
        $self->AddProgressMessage ($pendingMsg);
    }

	return 1;
}


sub getParamFilter{
    return ['installation', 'update'];
}

sub setParams{
	my ($self,$parser) = @_;
	my $xmlParams = $parser->getParameters();
	if (defined $xmlParams){
		my %hash = (%{$self->{params}},%$xmlParams);
		$self->{params} = \%hash;
	}
	$self->clearCachedParamIds ();
	my $product = $parser->getProduct ();
	$self->{productName} = $product->{ProductName};
	$self->{shortProductName} = $product->{ShortProductName};
	$self->{productKey} = $product->{ProductKey};
}

sub setUpdateParams{
	my ($self) = @_;
	my $parser = $self->{configparser};
	my $params = $self->{params};
	foreach my $id (keys (%$params)){
		if ($params->{$id}->{skip_on_update}){
			$params->{$id}->{skip} = 1;
			next;
		}
		my $file = $params->{$id}->{update_file};
		my $paramKey = $params->{$id}->{update_file_key};
		if ($file && $paramKey) {
			my $paramValue = $self->getParamFromUpdateFile($file, $paramKey);
			if ($paramValue) {
				$params->{$id}->{value} = $paramValue;
				$params->{$id}->{set_interactive} = 0;
			}
		}
	}

	return 1;
}

sub getParamFromUpdateFile{
	my ($self, $file, $key) = @_;

	# InstallationPath not replaced in param hash yet
	my $instpath = $self->get_InstallationPath();
	$file =~ s/%\(InstallationPath\)/$instpath/;

	if (defined $self->{updateFiles}->{$file}) {
		return $self->{updateFiles}->{$file}->getValue(undef, $key);
	}

	my $iniFile = new SDB::Install::IniFile($file);
	if (!defined $iniFile) {
		return undef;
	}
	$self->{updateFiles}->{$file} = $iniFile;
	return $self->{updateFiles}->{$file}->getValue(undef, $key);

}

sub getValidJavaVersions{
    if (defined $_[0]->{_valid_java_versions}){
        return $_[0]->{_valid_java_versions};
    }
    return $_[0]->SUPER::getValidJavaVersions ();
}

sub checkSystemRequirements {
	my ($self, $sysinfo) = @_;
	my $parser = $self->{kit}->getConfigParser();

	#check java version
	if ((defined $parser->getP2Call()) && $parser->getP2Call()) {
		my $director = $self->{kit}->GetPackageById ('director');
		if (!defined $director) {
			return undef;
		}
		my @javaVersions = $director->getJavaVersions();
		$self->{_valid_java_versions} = \@javaVersions;
		my $java = $self->findJava ();
		if (!defined $java){
			return undef;
		}
		$self->{java} = $java;
	}

	return $self->SUPER::checkSystemRequirements($sysinfo);
}

sub getJava{
    return $_[0]->{java};
}

sub getInstallationClass{
    return $installationClass;
}

sub getKitClass{
    return $kitClass;
}

sub getCustomConfigurationClass{
	my ($self) = @_;
	if (defined $customConfigurationClass) {
		return $customConfigurationClass;
	}
	my $parser = $self->{configparser};
	if (defined $parser) {
		my $customModules = $parser->getCustomModules();
		if (defined $customModules){
			my $configurationClass = $customModules->{Configuration};
			if (defined $configurationClass) {
				$customConfigurationClass = $configurationClass;
				return $customConfigurationClass;
			}
		}
	}
	return undef;
}

sub isInstallationSelected{
    return $_[0]->isValuePreset ('PATH');
}


sub getProductName{
    return $_[0]->{productName} ? $_[0]->{productName} : 'Unknown product';
}

sub getShortProductName{
     return $_[0]->{shortProductName} ? $_[0]->{shortProductName} : $_[0]->getProductName();
}

#sub new{
#    my $self = shift->SUPER::new (@_);
#    return $self;
#}

sub checkInstallationPath{
    my ($self,$value) = @_;
    $self->{_start_version} = undef;
    delete $self->{step};
    $self->{_path} = $value;
    if ($self->pers_exists()){
        if (!defined $self->validatePersistency ()){
			return undef;
		}
    }

    if (!defined $self->{step} && -d $value){
        my $inst = new SDB::Install::Installation ($value);
        if (!$inst->ErrorState () && $inst->isComponent($self->{compKey})){
            bless ($inst, $self->getInstallationClass ());
            my $errlst = new SDB::Install::MsgLst();
            if (!$self->{kit}->canUpdate($inst, $self, $errlst)){
                $self->PushError (undef, $errlst);
                return 0;
            }
            $self->{_start_version} = $inst->GetVersion;
            $self->AddMessage (undef, $errlst);
        }
    }
    if ($self->isUpdate()){
        $self->setUpdateParams();
    }
    return 1;
}

sub isUpdate {
    my ($self) = @_;
	return defined $self->{_start_version} ? 1 : 0;
}

sub getIgnoreValues{
    return [qw (check_version check_busy_files check_platform check_pending_installation check_diskspace)];
}

sub get_InstallationPath{
	$_[0]->getValue ('InstallationPath');
}

sub get_InstallationPathDefault{
	$_[0]->getDefault ('InstallationPath');
}

sub set_InstallationPath{
	$_[0]->setValue ('InstallationPath', $_[1]);
}

sub set_InstallationPathDefault{
	$_[0]->setDefault('InstallationPath', $_[1]);
}


sub getConfigXmlPath{
    my ($self) = @_;
    if (!defined $self->{kit}){
        return undef;
    }
    return $self->{kit}->getArchiveDir ();
}

sub getReplaceHash{
	my ($self) = @_;
	my $params = $self->{params};
	my %result;
	foreach my $id (keys (%$params)){
		if ($params->{$id}->{skip}){
			$result{'%('.$id.')'} = '';
			next;
		}
		$result{'%('.$id.')'} = $self->getValue ($id);
	}
	return \%result;
}

sub get_UID{
	return undef;
}

sub get_GID{
	return undef;
}

sub defineInstallParams{
    return 1;
}

sub getInstallation{
    return undef;
}

sub isUninstallation{
    return 0;
}

sub parseInstallParamsXml{
    my ($self, $xmlFile, $dtdFile) = @_;
    require SDB::Install::ConfigXMLParser;
    my $xmlDir = $self->getConfigXmlPath ();
    if (!defined $xmlFile){
        $xmlFile = $xmlDir . $path_separator . $installParamsXml;
    }

    if (!defined $dtdFile){
        $dtdFile = $xmlDir . $path_separator . $installCfgDtd;
    }
    if (! -f $xmlFile){
        return 1;
    }
    if (!-f $dtdFile){
        $self->setErrorMessage ("Installation configuration DTD file '$dtdFile' not found");
        return undef;
    }
    $self->{configparser} = new SDB::Install::ConfigXMLParser ();
    $self->{configparser}->setFilterAttributes ($self->getParamFilter());
    eval{
        $self->{configparser}->parsefile ($xmlFile, $dtdFile);
    };
    if ($@){
        $self->setErrorMessage ("Parsing installation configuration file '$xmlFile' failed: $@");
        return undef;
    }

    my $installType = $self->{configparser}->getProduct()->{InstallType};
    my $isUninstallation = $self->isUninstallation();
    if ($installType =~ /^Generic(ServerPlugin)?$/ && $isUninstallation){
        $installType = 'GenericUninstallation';
    }

    if ($installType) {
        my $configClass = "SDB::Install::Configuration::$installType";
        eval ("require $configClass;");
        if ($@){
            $self->setErrorMessage ("InstallType $installType not implemented: " . $@);
            return undef;
        }
        if (!$self->isa($configClass)){
            bless ($self, $configClass);
        }
        $self->defineInstallParams ();
    }
    $self->setParams ($self->{configparser});
    my $installation = $self->getInstallation ();
    if (defined $installation){
        $installation->setConfigParser ($self->{configparser});
    }
    return 1;
}

sub getPhase{
    return $_[0]->getValue ('Phase');
}

sub getProductVersion {
    my ($self) = @_;
    return $self->{kit}->GetVersion();
}

#########################################

sub setLogger {
    my (
        $self,
        $logger
    ) = @_;
    $self->{log} = $logger;
}

sub addLogFile {
    my (
        $self,
        $directory,
        $filename    # if the filename contains the substring 'LOGDATETIME' (in upper case!),
                     # this will be replaced with the current timestamp.
    ) = @_;
    $self->{log}->addLogFileLocation(LOG_FORMAT_PLAIN,
                                      $directory,
                                      1,
                                      $filename,
                                      LOG_DONT_INDICATE_LOCATION_ON_STDOUT,
                                      undef,
                                      undef,
                                      'tryOnly');
}

=example
[
    {
        'format'   => LOG_FORMAT_PLAIN,
        'path'     => '/some/path',
        'histsize' => 1,
        'name'     => 'install.log',
        'indicateLocationOnStdout' => LOG_DONT_INDICATE_LOCATION_ON_STDOUT
    },
    {
        'format'   => LOG_FORMAT_PLAIN,
        'path'     => '/some/trc/path',
        'histsize' => 3,
        'name'     => 'hdbremovehost_LOGDATETIME.log',
        'indicateLocationOnStdout' => LOG_DONT_INDICATE_LOCATION_ON_STDOUT,
        'tryOnly' => LOG_DESTINATION_TRY_ONLY
    }
];
=cut
sub getLogFileLocations {
    my ($self) = @_;
    return $self->{log}->{data};
}

1;
