package SDB::Install::Configuration::HdbCheck;

use strict;

use SDB::Install::Configuration::AnyMultiHostConfig;
use SDB::Install::Globals qw ($gLogDir
                              $gOperationCheckHost
                              $gPlatform
                              $gProductName
                              $gProductNameSystem
                              $gSapsysGroupName
                              $gFlavourPlatform);
use SDB::Install::System  qw (copy_file
                              nslookup
                              getFilesTimestampPID);
use SDB::Install::SysVars qw ($path_separator);
use SDB::Install::Tools   qw (printTableToArrayOfLines);
use SDB::Install::XMLParser;
use File::Basename        qw (basename dirname);

our @ISA = qw (SDB::Install::Configuration::AnyMultiHostConfig);

our $STR_SIDADM           = 'System administrator';
our $STR_UNDEF            = '<undef>';
our $CHECK_FILENAME       = 'hdbcheck';
our $ERR_EXTENSION        = '.err';
our $INFO_EXTENSION       = '.chk';
our $XML_EXTENSION        = '.xml';
our $ERR_XML_EXTENSION    = $ERR_EXTENSION . $XML_EXTENSION;
our $PROPERTY_FILE        = '$Target/$SID/global/hdb/install/support/'
                          . $CHECK_FILENAME . $XML_EXTENSION;
our $TAG_ERRORS           = $CHECK_FILENAME . 'Errors';
our $TAG_GLOBAL_DIR       = $CHECK_FILENAME . '_global_dir';
our $TAG_LOCAL_DIR        = $CHECK_FILENAME . '_local_dir';
our $TAG_HANA_OPTIONS_DIR = $CHECK_FILENAME . '_hana_options_dir';

#-------------------------------------------------------------------------------
# Constructor
sub new {
    my $self  = shift->SUPER::new (@_);

    my $order   = 0;
    my $section = 'HdbCheck';
    $self->{_errorCnt}     = 0;
    $self->{_infoFile}     = [];
    $self->{_checkedFiles} = {};

    my $descScope = "Checking $gProductNameSystem (all hosts) "
                  . 'or checking local instance only';

    $self->{params} = {
        'RemoteExecution'   => $self->getParamRemoteExecution   ($order++, $section),
        'UseHttp'           => $self->getParamUseHttp           ($order++, $section),
        'ScopeCheckTool'    => $self->getParamScope             ($order++, $section, $descScope),
        'Target'            => $self->getParamTarget            ($order++, $section),
        'SID'               => $self->getParamSID               ($order++, $section),
        'PropertyFile'      => $self->getParamPropertyFile      ($order++, $section),
        'TracePropertyFile' => $self->getParamTracePropertyFile ($order++, $section),
        'Password'          => $self->getParamPassword          ($order++, $section),
        'InstallSSHKey'     => $self->getParamInstallSSHKey     ($order++, $section),
        'RootUser'          => $self->getParamRootUser          ($order++, $section),
        'RootPassword'      => $self->getParamRootPassword      ($order++, $section),
        'HostagentUserName' => $self->getParamHostagentUserName ($order++, $section),
        'HostagentPassword' => $self->getParamHostagentPassword ($order++, $section),
        'SlaveHostName'     => $self->getParamHostName          ($order++, $section),
      # 'SystemUser'        => $self->getParamSystemUser        ($order++, $section),
      # 'SQLSysPasswd'      => $self->getParamSQLSysPasswd      ($order++, $section, 'passwd'),
    };

    my $slaveHostParam = $self->{params}->{'SlaveHostName'};
    $slaveHostParam->{init_with_default} = 0;
    $slaveHostParam->{set_interactive}   = 0;
    $slaveHostParam->{hidden}            = 1;

    my $paramScopeCheckTool = $self->{params}->{ScopeCheckTool};
    $self->setDefault('ScopeCheckTool', 'system');
    $paramScopeCheckTool->{init_with_default}       = 1;
    $paramScopeCheckTool->{init_batch_with_default} = 0;
    $paramScopeCheckTool->{skip}                    = 0;

    $self->setSummaryOrder(['PropertyFile',
                            'SlaveHostName',
                            'Target',
                            'SID',
                            'RootUser',
                            'HostagentUserName',
                            'SystemUser']);

	return $self;
}


#-------------------------------------------------------------------------------
# Returns a string containing the file properties e.g. 'drwxr-xr-x 755 1001  79'

sub addFileMsg {
    my ($self,
        $msglst,
        $indent,   # e.g. '    '
        $filepath, # path including filename
        $filename, # may be omitted
        $linkname,
        $type,     # e.g. 'directory'
        $mode,     # e.g. '750'
        $uid,
        $gid,
        $display, # additional output onto console
       ) = @_;

    my $name = (defined $filename) ? $filename : $filepath;
    if (defined $linkname) {
        $name     .= ' -> ' . $linkname;
        $filepath .= ' -> ' . $linkname;
    }

    my $prefix = $mode;
    $prefix   .= ($type eq 'link') ? ' l' : ($type eq 'subdir') ? ' d' :  ' -';

    foreach my $i (0..2) {
        my $c = substr($mode, $i, 1);
        $prefix .= ($c eq 0) ? '---'
                 : ($c eq 1) ? '--x'
                 : ($c eq 2) ? '-w-'
                 : ($c eq 3) ? '-wx'
                 : ($c eq 4) ? 'r--'
                 : ($c eq 5) ? 'r-x'
                 : ($c eq 6) ? 'rw-'
                 : ($c eq 7) ? 'rwx'
                             : '???';
    }

    $prefix .= sprintf("  %4d %3d  ", $uid, $gid);

    if ($display) {
        print $indent . $name . "\n";
    }
    $msglst->addMessage($indent . $prefix . $name);

    my $filetype = ($type eq 'link')   ? 'Link      '
                 : ($type eq 'subdir') ? 'Directory '
                                       : 'File      ';

    $self->{_checkedFiles}->{$filepath} = $filetype . $prefix;
    return 1;
}


#-------------------------------------------------------------------------------
sub checkPassword {
    my ($self, $sidadmPassword) = @_;

    if (!$self->verifySidAdmPassword($self->{_sidadmUser},
                                     $sidadmPassword,
                                     $self->{params}->{Password}->{str})) {
        return 0;
    }
    return 1;
}


#-------------------------------------------------------------------------------
sub checkPropertyFile {

    my ($self, $filename) = @_;

    if (!-f $filename) {
        $self->AddError($self->{params}->{PropertyFile}->{str}
                        . " '$filename' not found");
        return 0;
    }
    return 1;
}


#-------------------------------------------------------------------------------
sub checkSID {

    my ($self, $sid) = @_;

    $self->{params}->{SID}->{value} = $sid;
    my $sapSys   = $self->getSAPSystem();
    my $instance = $self->getOwnInstance();
    $self->{params}->{SID}->{value} = undef;

    if (!defined $sapSys || !$sapSys->hasNewDB || !defined $instance) {
        $self->AddError("$gProductNameSystem '$sid' has no HDB instance");
        return 0;
    }
    return 0 if (!$self->checkLocalHost($sid, $sapSys));

    $self->{_localhost}  = $instance->get_host();
    $self->{_sidadmUser} = $sapSys->getUser()->getSidAdm();
    $self->{_sidadmName} = $self->{_sidadmUser}->getname();
    $self->changePasswordStr($self->{_sidadmName});

    my $cfg = $sapSys->getUserConfig();
    if (defined $cfg) {
        my $cfgUID          = $cfg->{sidadm_id};
        my $cfgGID          = $cfg->{sapsys_groupid};
        $self->{_sidadmUID} = $cfgUID if (defined $cfgUID);
        $self->{_sidadmGID} = $cfgGID if (defined $cfgGID);
    }

    if (!$self->{isSlave} && $instance->exists_remote_host()
                           && ($self->getValue('ScopeCheckTool') eq 'system')) {
        if (!$self->initRemoteHosts()) {
                return undef;
        }
        $self->enableMultipleHostParameters(1); # 1 - $useSidadmForHostctrl
    }

    # Examples of host role specific hash maps:
    #
    # $self->{_hostRolesMap} = { 'lu123456' => {'worker'    => $validHostRoles->{worker},
    #                                           'xs_worker' => $validHostRoles->{xs_worker} },
    #                            'lu222222' => {'standby'   => $validHostRoles->{standby}   },
    #                            'lu333333' => {'rdsync'    => $validHostRoles->{rdsync}    },
    #                          };
    #
    # $self->{_isColumnStoreHost} = {'lu123456' => 1,
    #                                'lu222222' => 1};
    #
    # $self->{_isHANAOptionsHost} = {'lu123456' => 1,
    #                                'lu333333' => 1};

    $self->{_hostRolesMap}      = {};
    $self->{_isColumnStoreHost} = {};
    $self->{_isHANAOptionsHost} = {};
    my $strHostRolesMap = $instance->getHostRolesInfo();

    foreach my $currHost (@{$instance->get_allhosts()}) {

        my $strRoles = $strHostRolesMap->{$currHost};
        my $roleMap  = {};
        if (defined $strRoles) {
            my @roles = split(/\s+/, $strRoles);
            foreach my $currRole (@roles) {

                # $validHostRoles is defined in AnyMultiHostConfig
                my $currRoleInfo = $validHostRoles->{$currRole};

                if (defined $currRoleInfo) {
                    $roleMap->{$currRole} = $currRoleInfo;
                }
                else {
                    $self->errorHostRoleInvalid($currRole);
                    next;
                }

                if ($currRoleInfo->{isColumnStore}) {
                    $self->{_isColumnStoreHost}->{$currHost} = 1;
                }
                if ($currRoleInfo->{isAccelerator}     ||
                    $currRoleInfo->{isExtendedStorage} ||
                    $currRoleInfo->{isStreaming}       ||
                    $currRoleInfo->{isXS2}) {
                    $self->{_isHANAOptionsHost}->{$currHost} = 1;
                }
            }
        }
        $self->{_hostRolesMap}->{$currHost} = $roleMap;
    }

    $self->showSystemProperties();

    return 1;
}


#-------------------------------------------------------------------------------
sub CollectOtherHostInfos {
    my ($self) = @_;

    if ($self->getValue('ScopeCheckTool') eq 'instance') {
        return 0;
    }

    my $skipCollectMap = {'groups' => 1, 'users' => 1};

    return $self->SUPER::CollectOtherHostInfos(undef, # $isNotVerbose
                                               $self->{_sidadmName},
                                               undef, # $skipCollectAll
                                               $skipCollectMap);
}


#-------------------------------------------------------------------------------
# Returns a copy of the specified xml sub-hash without parent entries.

sub copyXML {
    my ($self, $xml) = @_;
    my $xmlCopy = {'type' => $xml->{type}};

    if (defined $xml->{attr}) {
        $xmlCopy->{attr} = {};
        foreach my $attrKey (keys %{$xml->{attr}}) {
            $xmlCopy->{attr}->{$attrKey} = $xml->{attr}->{$attrKey};
        }
    }

    if (defined $xml->{content}) {
        $xmlCopy->{content} = $xml->{content};
    }

    if (defined $xml->{child}) {
        $xmlCopy->{child} = $self->copyXML($xml->{child});
    }

    if (defined $xml->{neighbor}) {
        $xmlCopy->{neighbor} = $self->copyXML($xml->{neighbor});
    }

    return $xmlCopy;
}


#-------------------------------------------------------------------------------
# This subroutine scans the xml file and creates the hash $pathMap.
# Furthermore each variable (e.g. '${trexExeDir}') contained in the xml file
# is replaces by the corresponding value.
#
# The tag '<hostset>' is performed for each hostname. In this case the
# variable '${host}' is replaced by the current hostname.
#
# XML error file sample
# ---------------------
#<hdbcheckErrors>
#
#              $host
#               |
#   <host name="berl00357848a">
#
#        $objTag    $objName
#         |          |
#        <file name="/hana/shared/JA3/global/hdb/install/config/sapprofile_templ.ini">
#
#         $errTag |--------------- $errAttrMap -----------------|
#              |  |                                             |
#            <gid expectedGID="79" filetype="file" invalidGID="0">
#                 <errorText>File '/hana/shared/JA3...' belongs to the group id ...</errorText>
#            </gid>
#
#            <minmode expectedMinimalMode="750" filetype="file" invalidMode="444">
#                 <errorText>File '/hana/shared/JA3...' has the mode 444 on host...</errorText>
#            </minmode>

#            <maxmode expectedMaximalMode="750" filetype="file" invalidMode="444">
#                 <errorText>File '/hana/shared/JA3...' has the mode 444 on host...errorText>
#            </maxmode>
#
#            <uid expectedUID="1001" filetype="file" invalidUID="0">
#                 <errorText>File '/hana/shared/JA3...' belongs to the user id ...</errorText>
#            </uid>
#        </fileError>
#
#    </host>
#
#</hdbcheckErrors>

sub createErrorEntriesFromXML {
    my ($self, $xml) = @_;

    my $currHostXML = ($xml->{type} eq $TAG_ERRORS) ? $xml->{child} : undef;

    while (defined $currHostXML) {

        my $host = (($currHostXML->{type} eq 'host')
                                                && defined $currHostXML->{attr})
                   ? $currHostXML->{attr}->{name}
                   : undef;
        next if (!defined $host);
        my $currObjXML = $currHostXML->{child};

        while (defined $currObjXML) {

            my $objTag  = $currObjXML->{type};
            my $objName = undef;

            if (defined $currObjXML->{attr}) {
                # contains only one attribute: 'name' or 'remoteHost'
                my $attrMap  = $currObjXML->{attr};
                my @attrKeys = keys %$attrMap;
                $objName     = $attrMap->{$attrKeys[0]} if (defined $attrKeys[0]);
            }

            next if (!defined $objTag || !defined $objName);
            my $currErrorXML = $currObjXML->{child};

            while (defined $currErrorXML) {

                my $errTag = $currErrorXML->{type};
                next if (!defined $errTag);

                my $errAttrMap = $currErrorXML->{attr}; # may be an empty hash
                if (defined $currErrorXML->{child}
                    && ($currErrorXML->{child}->{type} eq 'errorText')) {
                    my $errTxt = $currErrorXML->{child}->{content};
                    $errAttrMap->{errorText} = $errTxt;
                }
                next if (!defined $errAttrMap);
                $self->{_errorCnt}++;
                $self->{_errors}->{$host}->{$objTag}->{$objName}->{$errTag} =
                                                                    $errAttrMap;
                $currErrorXML = $currErrorXML->{neighbor};
            }
            $currObjXML = $currObjXML->{neighbor};
        }
        $currHostXML = $currHostXML->{neighbor};
    }
    return 1;
}


#-------------------------------------------------------------------------------
# This recursive subroutine scans the xml property file and creates the
# hash $pathMap.
# Furthermore each variable (e.g. '${trexExeDir}') contained in the xml file
# is replaces by the corresponding value.
#
# The tag '<hostset>' is performed for each hostname. In this case the
# variable '${host}' is replaced by the current hostname.
#
# XML syntax
# ----------
#
# <hdbcheck>
#   <hdbcheck_local_dir>
#       properties
#       directories
#   </hdbcheck_local_dir>
#   <hdbcheck_global_dir>
#       properties
#       directories
#   </hdbcheck_global_dir>
# </hdbcheck>
#
# properties    ::= <property name="name" value="value"/>...
#
# directories   ::=   directories...
#                   | directoryDesc
#                   | <hostset> directoryDesc... </hostset>
#
# directoryDesc ::= <directory path="path[,path...]" [constraint="0"|"1"] >
#                       directoryEntry...
#                   </directory>
#
# The directory is skipped in case of 'constraint="0"'.
#
# directoryEntry ::=  <file   name="name" [fileSpec]               />
#                     <link   name="name" [fileSpec] target='path" />
#                     <subdir name="name" [fileSpec]               />
#
# fileSpec ::= [<max_mode>="fileMode"]           # if omitted, check is skipped
#              [<min_mode>="fileMode"]           # if omitted, check is skipped
#              [<uid="uid">]                     # if ommitted, uid from cfg file is used
#              [<gid="gid">]                     # if ommitted, gid from cfg file is used
#              [<pattern="1">]                   # if specified, the name is check like a Perl pattern
#              [<optional="1">]                  # if specified, object is not mandatory
#
# fileMode    ::= octal: 0 - ownerPermissions - GroupPermissions - worldPermissions
# permissions ::= 7 (rwx) | 6 (rw) | 5 (rx) | 4 (r) | 3 (wx) | 2 (w) | 1 (x)
#
# ==============================================================================
#
# $pathMap = { <path> => <path_desc>,... }
#
# <path_desc> ::= { <filename> => file_properties>,... }
#
# <file_properties> ::=
#         {'gid'      => <group_id>,  # if ommitted, cfg file *) is used
#          'max_mode' => <permission> # if ommitted, min_mode is not checked
#          'min_mode' => <permission> # if ommitted, max_mode is not checked
#          'optional' => 1,           # if ommitted, object is mandatory
#          'pattern'  => 1,           # if ommitted, name is not a Perl pattern
#          'target'   => <path>,      # target path or name in case of a link
#          'type'     => <filetype>,  # object type ( 'dir' | 'file' | 'link')
#          'uid'      => <user_id>,   # if ommitted, cfg file *) is used
#         }
#
# *) cfg file: '<sapmnt>/<SID>/global/hdb/install/support/cfg'
#
#
# Sample:
#
# $pathMap = {'/usr/sap/DB7' => {'dirLevel'   => 0,
#                                'anyPattern' => 0, # ommitted if value == 0
#                                'content'    => {'HDB07'  => {'type'     => 'link',
#                                                              'min_mode' => 0111,
#                                                              'max_mode' => 0777,
#                                                              'target'   => '/hana/shared/DB7/HDB07'
#                                                             },
#                                                 'SYS'    => {'type'     => 'dir',
#                                                              'min_mode' => 0111,
#                                                              'max_mode' => 0777,
#                                                             },
#                                                 'home'   => {'type'     => 'dir',
#                                                              'min_mode' => 0111,
#                                                              'max_mode' => 0777,
#                                                             }
#                               },
#             '/usr/sap/DB7/SYS' => {'dirLevel'   => 1,
#                                    'anyPattern' => 0,
#                                    'content'    => {'exe'    => {'type'     => 'dir',
#                                                                  'min_mode' => 0111,
#                                                                  'max_mode' => 0777,
#                                                                 },
#                                                     'global' => {'type'     => 'link',
#                                                                  'min_mode' => 0111,
#                                                                  'max_mode' => 0777,
#                                                                  'target'   => '/hana/shared/DB7/global'
#                                                                 },
#                                                     'profile'=> {'type'     => 'link',
#                                                                  'min_mode' => 0111,
#                                                                  'max_mode' => 0777,
#                                                                  'target'   => '/hana/shared/DB7/profile'
#                                                                 }
#                                                    }
#                                   }
#             '/usr/sap/DB7/SYS/hdb'=> {'dirLevel'   => 2,
#                                       'anyPattern' => 0,
#                                       'content'    => { ... }
#                                      }
#            };

sub createPathMapFromXML {
    my ($self,
        $xml,
        $xmlVars,
        $ownXMLVars,
        $pathMap,
        $rootPath,
        $currPath,
        $indent,
        $msglst, # if ommitted, output is suppressed
       ) = @_;

    my $currXML = $xml;

    while (defined $currXML) {

        $msglst->addMessage("${indent}<$currXML->{type}>") if (defined $msglst);

        if ($currXML->{type} eq 'hostset') {
            if (defined $currXML->{child}) {
                my $allHosts = $self->getOwnInstance()->get_allhosts();
                my @xmlPerHost;
                foreach my $currHost (@$allHosts) {
                    my $xmlCopy = $self->copyXML($currXML->{child});
                    push @xmlPerHost, $xmlCopy;
                }

                foreach my $i (0 .. ((scalar @xmlPerHost) -1) ) {
                    if (defined $msglst) {
                        $msglst->addMessage("${indent}  Host: '$allHosts->[$i]'");
                    }
                    $ownXMLVars->{'host'} = $allHosts->[$i];
                    $self->createPathMapFromXML($xmlPerHost[$i],
                                                $xmlVars,
                                                $ownXMLVars,
                                                $pathMap,
                                                $rootPath,
                                                $currPath,
                                                $indent . '    ',
                                                $msglst);
                }
                delete $ownXMLVars->{'host'};
            }
            $currXML = $currXML->{neighbor};
            next;
        }

        if (defined $currXML->{attr}) {

            foreach my $attrKey (keys %{$currXML->{attr}}) {

                if (defined $msglst) {
                    $msglst->addMessage
                        ("$indent    $attrKey = '$currXML->{attr}->{$attrKey}'");
                }
                my $val = $currXML->{attr}->{$attrKey};

                while (defined $val && ($val =~ /\$\{\w+}/)) {
                    my ($var) = $val =~ /\$\{(\w+)\}/;
                    my $repl  = $xmlVars->{$var};
                    $repl     = $ownXMLVars->{$var} if (!defined $repl);
                    if (!defined $repl) {
                        $repl = "<$var>";
                        $self->incrErr('XML variable \'${' . $var
                                      . '}\' not found', $indent . '    ');
                    }
                    $val =~ s/\$\{(\w+)\}/$repl/;
                }

                if ($val ne $currXML->{attr}->{$attrKey}) {
                    $currXML->{attr}->{$attrKey} = $val;
                    if (defined $msglst) {
                        $msglst->addMessage("$indent    $attrKey = '$val'");
                    }
                }
            }
        }

        if ($currXML->{type} eq 'property') {

            my $name  = $currXML->{attr}->{name};
            my $value = $currXML->{attr}->{value};

            if (!defined $name) {
                $self->incrErr("Attribute 'name' missing in xml tag"
                               . " '$currXML->{type}'", $indent . '    ');
            }
            elsif (!defined $value) {
                $self->incrErr("Attribute 'value' missing in xml tag"
                               . " '$currXML->{type}'", $indent . '    ');
            }
            else {
                $ownXMLVars->{$name} = $value;
            }
        }
        elsif (($currXML->{type} eq 'file') ||
               ($currXML->{type} eq 'link') ||
               ($currXML->{type} eq 'subdir')) {

            my $name = $currXML->{attr}->{name};

            if (!defined $name) {
                $self->incrErr("Attribute 'name' missing in xml tag"
                               . " '$currXML->{type}'", $indent . '    ');
            }
            else {
                my $entry = {'type' => $currXML->{type}};
                foreach my $attrKey ('gid', 'optional', 'pattern', 'target', 'uid', 'min_mode', 'max_mode') {
                    if (defined $currXML->{attr}->{$attrKey}) {
                        $entry->{$attrKey} = $currXML->{attr}->{$attrKey};
                        if (($attrKey eq 'pattern') && ($entry->{$attrKey} eq '1')) {
                            $pathMap->{$currPath}->{anyPattern} = 1;
                        }
                    }
                }

                $pathMap->{$currPath}->{content}->{$name} = $entry;
            }
        }
        elsif (($currXML->{type} eq 'directory') && defined $currXML->{child}
               &&
               (!defined $currXML->{attr}->{constraint} ||
                ($currXML->{attr}->{constraint} eq '1'))) {

            my @pathArray;
            my $pathItemOrList  = $currXML->{attr}->{path};
            if (!defined $pathItemOrList) {
                $self->incrErr("Attribute 'path' missing in xml tag"
                               . " '$currXML->{type}'", $indent . '    ');
            }
            elsif ($pathItemOrList =~ /,/) {
                @pathArray = split(',', $pathItemOrList);
            }
            else {
                @pathArray = ( $pathItemOrList );
            }

            foreach my $directoryPath (@pathArray) {

                $pathMap->{$directoryPath}->{level} =
                        $self->getPathLevel($directoryPath, $rootPath);

                if (!$self->createPathMapFromXML($currXML->{child},
                                                 $xmlVars,
                                                 $ownXMLVars,
                                                 $pathMap,
                                                 $rootPath,
                                                 $directoryPath,
                                                 $indent.'    ',
                                                 $msglst)) {
                    return undef;
                }
            }
        }
        $currXML = $currXML->{neighbor};
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Adds the action ID to the persfile data in order to check that the slave host
# is reading this persfile.
# Furthermore all IP addresses are added to the persfile data.
#
# AnyConfig::CollectOtherHostInfos has created this hash:
#                       host          IP address array
# $self->{remoteIPs}->{'lu123456'}->['127.0.0.1/8', '100.100.100.150/24']
#                     {'lu222222'}->['127.0.0.1/8', '100.100.100.151/24']

sub createPersData {
    my ($self, $instance) = @_;

    my $persData        = {'actionID' => $self->{options}->{action_id}};
    my $remoteHostNames = $instance->get_hosts();
    my $remoteIPs       = $self->{remoteIPs};

    if (defined $remoteIPs) {
        foreach my $currHost (@$remoteHostNames) {
            my $addrArray = $remoteIPs->{$currHost};
            if (defined $addrArray) {
                $persData->{$currHost} = join(',', sort @$addrArray);
            }
        }
    }

    # add local IP addresses to persData
    my $sysinfo       = new SDB::Install::SysInfo();
    my $netInterfaces = $sysinfo->interfaceInfo();
    my @IPAddrArray;
    my @X = values %$netInterfaces;
    foreach my $arrayNetEntries (values %$netInterfaces){
        foreach my $netEntry (@$arrayNetEntries) {
            push @IPAddrArray, "$netEntry->{addr}/$netEntry->{netbits}";
        }
    }
    $persData->{$self->{_localhost}} = join(',', sort @IPAddrArray);

    return $persData;
}


#-------------------------------------------------------------------------------
sub errorFileGID {
    my ($self,
        $path,
        $filetype,
        $invalidGID,
        $expectedGID,
        $indent) = @_;

    my $localHost = $self->{_localhost};
    my $errorText = ucfirst($filetype)
                  . " '$path' belongs to the group id '$invalidGID'"
                  . " instead of '$expectedGID' on host '$localHost'";
    $self->incrErr($errorText, $indent);
    $self->{_errors}->{$localHost}->{file}->{$path}->{gid} =
                                    {'filetype'    => $filetype,
                                     'invalidGID'  => $invalidGID,
                                     'expectedGID' => $expectedGID,
                                     'errorText'   => $errorText};
}


#-------------------------------------------------------------------------------
sub errorFileIO {
    my ($self,
        $path,
        $filetype,
        $ioType,
        $ioError,
        $indent,
        $sublst) = @_;

    my $localHost = $self->{_localhost};
    my $errorText = "Cannot $ioType $filetype '$path' on host '$localHost'";
    $errorText   .= ": $ioError" if (defined $ioError);
    $self->incrErr($errorText, $indent, $sublst);
    $self->{_errors}->{$localHost}->{file}->{$path}->{$ioType} =
                                    {'filetype'  => $filetype,
                                     'ioError'   => $ioError,
                                     'errorText' => $errorText};
}


#-------------------------------------------------------------------------------
sub errorFileLink {
    my ($self,
        $path,
        $invalidTarget,
        $expectedTarget,
        $indent) = @_;

    my $localHost = $self->{_localhost};
    my $errorText = "Link '$path' refers to '$invalidTarget'"
                  . " instead of '$expectedTarget' on host '$localHost'";
    $self->incrErr($errorText, $indent);
    $self->{_errors}->{$localHost}->{file}->{$path}->{link} =
                                    {'filetype'       => 'link',
                                     'invalidTarget'  => $invalidTarget,
                                     'expectedTarget' => $expectedTarget,
                                     'errorText'      => $errorText};
}

#-------------------------------------------------------------------------------
sub errorFileMinimalMode {
    my ($self, $path, $filetype, $currentMode, $expectedMode, $indent) = @_;
    my $localHost = $self->{_localhost};
    my $messageTemplate = '%s \'%s\' has the mode %s on host \'%s\' (minimal mode %s required)';
    my $errorText = sprintf($messageTemplate, ucfirst($filetype), $path, $currentMode, $localHost, $expectedMode);

    $self->incrErr($errorText, $indent);
    $self->{_errors}->{$localHost}->{file}->{$path}->{minmode} = {
        'filetype'            => $filetype,
        'invalidMode'         => $currentMode,
        'expectedMinimalMode' => $expectedMode,
        'errorText'           => $errorText
    };
}

#-------------------------------------------------------------------------------
sub errorFileMaximalMode {
    my ($self, $path, $filetype, $currentMode, $expectedMode, $indent) = @_;
    my $localHost = $self->{_localhost};
    my $messageTemplate = '%s \'%s\' has the mode %s on host \'%s\' (maximal mode %s allowed)';
    my $errorText = sprintf($messageTemplate, ucfirst($filetype), $path, $currentMode, $localHost, $expectedMode);

    $self->incrErr($errorText, $indent);
    $self->{_errors}->{$localHost}->{file}->{$path}->{maxmode} = {
        'filetype'            => $filetype,
        'invalidMode'         => $currentMode,
        'expectedMaximalMode' => $expectedMode,
        'errorText'           => $errorText
    };
}


#-------------------------------------------------------------------------------
sub errorFileNotFound {
    my ($self,
        $path,
        $filetype,
        $indent,
        $hint) = @_;

    my $localHost = $self->{_localhost};
    my $errorText = ucfirst($filetype)
                  . " '$path' not found on host '$localHost'";
    $errorText .= " ($hint)" if defined $hint;
    $self->incrErr($errorText, $indent);
    $self->{_errors}->{$localHost}->{file}->{$path}->{notFound} =
                                    {'filetype'  => $filetype,
                                     'errorText' => $errorText};
}


#-------------------------------------------------------------------------------
sub errorFileUID {
    my ($self,
        $path,
        $filetype,
        $invalidUID,
        $expectedUID,
        $indent) = @_;

    my $localHost = $self->{_localhost};
    my $errorText = ucfirst($filetype)
                  . " '$path' belongs to the user id '$invalidUID'"
                  . " instead of '$expectedUID' on host '$localHost'";
    $self->incrErr($errorText, $indent);
    $self->{_errors}->{$localHost}->{file}->{$path}->{uid} =
                                    {'filetype'    => $filetype,
                                     'invalidUID'  => $invalidUID,
                                     'expectedUID' => $expectedUID,
                                     'errorText'   => $errorText};
}


#-------------------------------------------------------------------------------
sub errorGroupID {
    my ($self,
        $host,
        $groupName,
        $invalidGID,
        $expectedGID) = @_;

    my $errorText = "Group '$gSapsysGroupName' on host '$host' belongs to"
                  . " group id '$invalidGID' instead of '$expectedGID'";
    $self->incrErr($errorText);
    $self->{_errors}->{$host}->{userGroup}->{$groupName}->{gid} =
                                    {'invalidGID'  => $invalidGID,
                                     'expectedGID' => $expectedGID,
                                     'errorText'   => $errorText};
}


#-------------------------------------------------------------------------------
sub errorGroupNotFound {
    my ($self, $groupName) = @_;

    my $localHost = $self->{_localhost};
    my $errorText = "Group '$groupName' does not exist on host '$localHost'";
    $self->incrErr($errorText);
    $self->{_errors}->{$localHost}->{userGroup}->{$groupName}->{notFound} =
                                    {'errorText' => $errorText};
}


#-------------------------------------------------------------------------------
sub errorGroupNotReceived {
    my ($self, $remoteHost, $groupName) = @_;

    my $localHost = $self->{_localhost};
    my $errorText = "Group '$groupName' not received from host '$remoteHost'";
    $self->incrErr($errorText);
    $self->{_errors}->{$localHost}->{userGroup}->{$groupName}->{notReceived} =
                                    {'remoteHost' => $remoteHost,
                                     'errorText'  => $errorText};
}


#-------------------------------------------------------------------------------
sub errorHostNameMismatch {
    my ($self, $expectedHost) = @_;

    my $localHost = $self->{_localhost};
    my $errorText = "Host '$localHost' is running instead of host '$expectedHost'";
    $self->incrErr($errorText);
    $self->{_errors}->{$localHost}->{hostName}->{$expectedHost}->{mismatch} =
                                    {'errorText' => $errorText};
}


#-------------------------------------------------------------------------------
sub errorHostReply {
    my ($self, $remoteHost) = @_;

    my $localHost = $self->{_localhost};
    my $errorText = "Host '$remoteHost' does not reply";
    $self->incrErr($errorText);
    $self->{_errors}->{$localHost}->{hostReply}->{$remoteHost}->{missing} =
                                    {'errorText'  => $errorText};
}


#-------------------------------------------------------------------------------
sub errorHostRoleInvalid {
    my ($self, $invalidHostRole) = @_;

    my $localHost = $self->{_localhost};
    my $errorText = "Invalid host role '$invalidHostRole' on host '$localHost'";
    $self->incrErr($errorText);
    $self->{_errors}->{$localHost}->{hostRole}->{$invalidHostRole}->{invalid} =
                                    {'errorText' => $errorText};
}


#-------------------------------------------------------------------------------
sub errorIPAddressInvalid {
    my ($self,
        $invalidIPAddr,
        $remoteHost,
        $remoteIPAddrList) = @_;

    my $localHost = $self->{_localhost};
    my $errorText = "Host '$localHost' uses invalid IP address '$invalidIPAddr'"
                  . " to access remote host '$remoteHost' ('$remoteIPAddrList')";
    $self->incrErr($errorText);
    $self->{_errors}->{$localHost}->{IPAddress}->{$remoteHost}->{invalid} =
                                    {'invalidIPAddr'    => $invalidIPAddr,
                                     'remoteIPAddrList' => $remoteIPAddrList,
                                     'errorText'        => $errorText};
}


#-------------------------------------------------------------------------------
sub errorIPAddressLookupFailed {
    my ($self, $remoteHost, $sublst) = @_;

    my $localHost = $self->{_localhost};
    my $errorText = "Cannot resolve host name '$remoteHost': "
                  .'Calling nslookup failed';
    $self->incrErr($errorText, undef, $sublst);
    $self->{_errors}->{$localHost}->{IPAddress}->{$remoteHost}->{lookupFailed} =
                                    {'errorText' => $errorText};
}


#-------------------------------------------------------------------------------
sub errorIPAddressNotReceived {
    my ($self, $remoteHost) = @_;

    my $localHost = $self->{_localhost};
    my $errorText = "IP address not received from host '$remoteHost'";
    $self->incrErr($errorText);
    $self->{_errors}->{$localHost}->{IPAddress}->{$remoteHost}->{notReceived} =
                                    {'errorText' => $errorText};
}


#-------------------------------------------------------------------------------
sub errorPendingProgram {
    my ($self, $program, $outstandingHost, $errorText) = @_;

    my $localHost = $self->{_localhost};
    $self->incrErr($errorText);
    $self->{_errors}->{$localHost}->{program}->{$program}->{pending} =
                                    {'host'      => $outstandingHost,
                                     'errorText' => $errorText};
}


#-------------------------------------------------------------------------------
sub errorPersFileInvalid {
    my ($self,
        $persfile,
        $invalidActionID,
        $expectedActionID) = @_;

    my $localHost = $self->{_localhost};
    my $errorText = "Host '$localHost' reads '$invalidActionID'"
                  . " instead of '$expectedActionID' from file '$persfile'";
    $self->incrErr($errorText);
    $self->{_errors}->{$localHost}->{persFile}->{$persfile}->{invalid} =
                                    {'invalidActionID'  => $invalidActionID,
                                     'expectedActionID' => $expectedActionID,
                                     'errorText'        => $errorText};
}


#-------------------------------------------------------------------------------
sub errorSidadmGID {
    my ($self,
        $host,
        $invalidGID,
        $expectedGID) = @_;

    my $errorText = "$STR_SIDADM '$self->{_sidadmName}' on host '$host' belongs"
                  . " to group id '$invalidGID' instead of '$expectedGID'";
    $self->incrErr($errorText);
    $self->{_errors}->{$host}->{user}->{$self->{_sidadmName}}->{gid} =
                                    {'invalidGID'  => $invalidGID,
                                     'expectedGID' => $expectedGID,
                                     'errorText'   => $errorText};
}


#-------------------------------------------------------------------------------
sub errorSidadmNotFound {
    my ($self) = @_;

    my $localHost = $self->{_localhost};
    my $errorText = "$STR_SIDADM '$self->{_sidadmName}' does not exist"
                  . " on host '$localHost'";
    $self->incrErr($errorText);
    $self->{_errors}->{$localHost}->{user}->{$self->{_sidadmName}}->{notFound} =
                                    {'errorText' => $errorText};
}


#-------------------------------------------------------------------------------
sub errorSidadmNotReceived {
    my ($self, $remoteHost) = @_;

    my $localHost = $self->{_localhost};
    my $errorText = "$STR_SIDADM '$self->{_sidadmName}' not received"
                  . " from host '$remoteHost'";
    $self->incrErr($errorText);
    $self->{_errors}->{$localHost}->{user}->{$self->{_sidadmName}}->{notReceived} =
                                    {'remoteHost' => $remoteHost,
                                     'errorText'  => $errorText};
}

#-------------------------------------------------------------------------------
sub errorSidadmUID {
    my ($self,
        $host,
        $invalidUID,
        $expectedUID) = @_;

    my $errorText = "$STR_SIDADM '$self->{_sidadmName}' on host '$host' belongs"
                  . " to user id '$invalidUID' instead of '$expectedUID'";
    $self->incrErr($errorText);
    $self->{_errors}->{$host}->{user}->{$self->{_sidadmName}}->{uid} =
                                    {'invalidUID'  => $invalidUID,
                                     'expectedUID' => $expectedUID,
                                     'errorText'   => $errorText};
}


#-------------------------------------------------------------------------------
sub errorXMLParser {
    my ($self, $xmlFile, $parserError) = @_;

    my $localHost = $self->{_localhost};
    my $errorText = "Parsing of xml file '$xmlFile' failed: $parserError";
    $self->incrErr($errorText);
    $self->{_errors}->{$localHost}->{xml}->{$xmlFile}->{parserError} =
                                    {'parserError' => $parserError,
                                     'errorText'   => $errorText};
}


#-------------------------------------------------------------------------------
sub errorXMLTag {
    my ($self, $xmlFile, $xmlTag) = @_;

    my $localHost = $self->{_localhost};
    my $errorText = "XML Tag '$xmlTag' not found in xml file '$xmlFile'";
    $self->incrErr($errorText);
    $self->{_errors}->{$localHost}->{xml}->{$xmlFile}->{tagNotFound} =
                                    {'tag'       => $xmlTag,
                                     'errorText' => $errorText};
}


#-------------------------------------------------------------------------------
sub getErrCntInfo {
    my ($self, $startErrorCnt) = @_;

    my $errCnt = (defined $startErrorCnt) ? $self->{_errorCnt} - $startErrorCnt
                                          : $self->{_errorCnt};
    return  ($errCnt == 1)
            ? '*** 1 error detected'
            : ($errCnt > 1) ? "*** $errCnt errors detected"
                            : 'No errors detected';
}


#-------------------------------------------------------------------------------
sub getParamPropertyFile {
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order,
        'opt'               => 'property_file',
        'hostctrl_opt'      => 'PROPERTY_FILE',
        'opt_arg'           => '<path>',
        'type'              => 'path',
        'section'           => $section,
        'value'             => undef,
        'default'           => $PROPERTY_FILE,
        'str'               => 'Property File',
        'desc'              => 'XML file containing the description of the directories',
        'init_with_default' => 1,
        'set_interactive'   => 0,
    };
}


#-------------------------------------------------------------------------------
# The xml tags 'groups' and 'users' are skipped by default (security).

sub getParamTracePropertyFile {
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order,
        'opt'               => 'trace_property_file',
        'type'              => 'boolean',
        'section'           => $section,
        'value'             => undef,
        'default'           => 0,
        'str'               => 'Writes additional log output when scanning the property file',
        'init_with_default' => 1,
        'set_interactive'   => 0,
    };
}


#-------------------------------------------------------------------------------
# Returns the number of entries within the path.
sub getPathLevel {
    my ($self, $path, $rootPath) = @_;

    my $level          = 0;
    my $currPath       = $path;
    my $rootPathLength = length($rootPath);

    while (defined $currPath && (length($currPath) > $rootPathLength)
           && ($currPath ne $rootPath)) {
        $level++;
        $currPath = dirname($currPath);
    }
    return $level;
}


#-------------------------------------------------------------------------------
# Returns a file path that is used to copy an output file to
# '<sapmnt>/<SID>/HDB<nr>/<hostname>/trace/<filename>_<actionID><extension>'

sub getSlaveCopyFilepath {
    my ($self,
        $host,
        $filename,  # e.g. 'hdbcheck_lu123456'
        $extension, # e.g. '.chk'
       ) = @_;
    my $traceDir = $self->getOwnInstance()->get_hostNameTraceDir(lc($host));
    return $traceDir . $path_separator . $filename
           . '_' . $self->{options}->{action_id} . $extension;
}


#-------------------------------------------------------------------------------
sub getSlaveInfoFilePath {
    my ($self, $host) = @_;
    return $self->getSlaveCopyFilepath($host,
                                       $CHECK_FILENAME . '_' . lc($host),
                                       $INFO_EXTENSION);
}


#-------------------------------------------------------------------------------
sub getStrFiletype {
    my ($self, $desc) = @_;
    return ($desc->{type} eq 'subdir') ? 'directory' : $desc->{type};
}


#-------------------------------------------------------------------------------
sub getTimeoutValues {
    return undef;
}


#-------------------------------------------------------------------------------
# Returns the xml parser and parses the property file.

sub getXMLParser {
    my ($self, $filePath) = @_;

    if (!-f $filePath) {
        $self->errorFileNotFound($filePath, 'file', '');
        return undef;
    }

    require SDB::Install::XMLParser;
    my $xmlParser = new SDB::Install::XMLParser();

    eval{
        $xmlParser->parsefile($filePath);
    };

    if ($@){
        $self->errorXMLParser('$filePath', $@);
        return undef;
    }

    return $xmlParser;
}


#-------------------------------------------------------------------------------
# Returns a hash that contains the values of these xml variables:
#    '${exeDirAbs}'
#    '${globalDir}'
#    '${hdbInstance}'
#    '${isSidadmDefaultHome}'
#    '${localDir}'
#    '${platform}'
#    '${sid}'
#
# For all other xml variables the xml file has to provide 'property' tags, e.g.
#    <property name="trexExeDir" value="${globalDir}/exe/${platform}/hdb"/>

sub getXMLVariables {
    my ($self) = @_;

    my $instance = $self->getOwnInstance();
    my $flavour  = $instance->getManifest()->getHANAFlavour();
    my $sid      = $instance->get_sid();
    my $localDir = join($path_separator, '', 'usr', 'sap', $sid);

    my $isDefaultSidadm   = 0;
    if ($self->{_sidadmUser}->exists()
        && (dirname($self->{_sidadmUser}->home()) eq $localDir)) {
        $isDefaultSidadm = 1;
    }

    my $isAccelerator     = 0;
    my $isExtendedStorage = 0;
    my $isStreaming       = 0;
    my $isXS2             = 0;
    my $localRoles        = $self->{_hostRolesMap}->{$self->{_localhost}};
    my $isIsolatedMultiDb = $instance->isIsolatedMultiDb();

    foreach my $currRoleInfo (values %$localRoles) {

        next if ($currRoleInfo->{isColumnStore});

        if ($currRoleInfo->{isAccelerator}) {
            $isAccelerator = 1;
        }
        elsif ($currRoleInfo->{isExtendedStorage}) {
            $isExtendedStorage = 1;
        }
        elsif ($currRoleInfo->{isStreaming}) {
            $isStreaming = 1;
        }
        elsif ($currRoleInfo->{isXS2}) {
            $isXS2 = 1;
        }
    }

    return {'exeDirAbs'           => basename($instance->get_globalTrexExeDirAbs()),
            'globalDir'           => $instance->get_globalSidDir(),
            'hdbInstance'         => 'HDB' . $instance->get_nr(),
            'isPlatform'          => $flavour eq $gFlavourPlatform,
            'isAccelerator'       => $isAccelerator,
            'isExtendedStorage'   => $isExtendedStorage,
            'isSidadmDefaultHome' => $isDefaultSidadm,
            'isStreaming'         => $isStreaming,
            'isXS2'               => $isXS2,
            'localDir'            => $localDir,
            'platform'            => $gPlatform,
            'sid'                 => $sid,
            'isHighIsolationMultiDb'   => $isIsolatedMultiDb,
            'isLowIsolationMultiDb' => !$isIsolatedMultiDb,
           };
}


#-------------------------------------------------------------------------------
sub incrErr {
    my ($self, $msg, $indent, $sublst) = @_;

    $indent = '' if (!defined $indent);
    my $err = "*** $msg";
    $self->AddError($indent . $err, $sublst);
    if ($self->{isSlave}) {
        print "$err ...\n";
    }
    else {
        print $indent . $err . "\n";
    }
    $self->{_errorCnt}++;
}


#-------------------------------------------------------------------------------
sub incrErrShowMsglst {
    my ($self) = @_;
    $self->{_errorCnt}++;
    print ${$self->getErrMsgLst()->getMsgLstString('***')};
}


#-------------------------------------------------------------------------------
# Inspects the directories using the xml description.

sub inspectDirectories {
    my ($self,
        $xmlParser,
        $xmlVars,
        $kind,     # e.g. 'local directories'
        $mainTag,  # e.g. 'hdbcheck_local_dir'
        $rootPath, # e.g. '/usr/sap/DB7'
       ) = @_;

    print "\n";
    my $mainMsg      = $self->AddProgressMessage("Checking $kind...");
    my $mainMsglst   = $mainMsg->getSubMsgLst();
    my $saveMainCtxt = $self->setMsgLstContext([$mainMsglst]);
    my $subMsg       = undef;
    my $subMsglst    = undef;
    my $saveSubCtxt  = undef;
    my $startErrCnt  = $self->{_errorCnt};
    my $filePathMap  = {};
    my $rc           = 1;

    if ($self->getValue('TracePropertyFile')) {
        $subMsg     = $mainMsglst->addMessage("Scanning xml tag '$mainTag'...");
        $subMsglst  = $subMsg->getSubMsgLst();
        $saveSubCtxt= $self->setMsgLstContext([$subMsglst]);
    }

    my $xml = $xmlParser->getElementByTagName($mainTag);

    if (!defined $xml) {
        $self->errorXMLTag($self->getValue('PropertyFile'), $mainTag);
    }
    else {
        my $ownXMLVars = {};
        if (!$self->createPathMapFromXML($xml->{child},
                                         $xmlVars,
                                         $ownXMLVars,
                                         $filePathMap,
                                         $rootPath,
                                         undef,
                                         '    ',
                                         $subMsglst)) {
           $rc = undef;
        }
    }

    if (defined $subMsg) {
        $subMsg->endMessage(1);
        $self->setMsgLstContext($saveSubCtxt);
    }

    if ($rc && defined $xml) {
        foreach my $currPath (sort keys %$filePathMap) {

            if (!opendir (DH, $currPath)) {
                $self->errorFileIO($currPath, 'directory', 'open', $!, '');
                next;
            }

            my @entriesFromDisk = readdir (DH);
            closedir (DH);

            my $level  = $filePathMap->{$currPath}->{level};
            my $indent = '';
            if (defined $level && ($level > 0)) {
                $indent .= '  ' x $level;
            }

            if (!defined $level || ($level <= 1)) {
                print "    $currPath\n";
            }
            $mainMsglst->addMessage($indent . $currPath);

            $rc = $self->inspectSubDirectory($currPath,
                                             \@entriesFromDisk,
                                             $filePathMap->{$currPath},
                                             $indent . '      ',
                                             $mainMsglst);
            last if (!$rc);
        }
    }
    $mainMsg->endMessage(1, ucfirst($kind) . ' directory checked: '
                         . $self->getErrCntInfo($startErrCnt));
    $self->setMsgLstContext($saveMainCtxt);
    return $rc;
}


#-------------------------------------------------------------------------------
# Inspects the remote host names.
#
# AnyConfig::CollectOtherHostInfos has created this hash:
#                       host          IP address array
# $self->{remoteIPs}->{'lu123456'}->['127.0.0.1/8', '100.100.100.150/24']
#                     {'lu222222'}->['127.0.0.1/8', '100.100.100.151/24']

sub inspectIPAddresses {
    my ($self) = @_;

    if (!$self->{isSlave} && !defined $self->getRemoteHosts()) {
        return 1;
    }

    my $rc  = 1;
    my $msg = $self->AddProgressMessage ('Checking IP addresses...');

    my $msglst           = $msg->getSubMsgLst();
    my $saveCtxt         = $self->setMsgLstContext([$msglst]);
    my $startErrCnt      = $self->{_errorCnt};
    my $instance         = $self->getOwnInstance();
    my $remoteHostnames  = $instance->get_hosts();
    my $inspectedHostMap = undef;
    my $wantedSlaveHost  = $self->getValue('SlaveHostName');

    if ($self->{isSlave} && defined $wantedSlaveHost
        && ($wantedSlaveHost ne $self->{_localhost})) {
        $self->errorHostNameMismatch($wantedSlaveHost);
    }

    require SDB::Install::System;

    foreach my $hostname (@$remoteHostnames) {
        my $errlst   = new SDB::Install::MsgLst();
        my $nslookup = nslookup($hostname, $errlst);
        my $IPAddr   = (defined $nslookup) ? $nslookup->{address} : undef;

        if(!defined $IPAddr){
            $self->errorIPAddressLookupFailed($hostname, $errlst);
            $rc = undef;
            last,
        }

        if (!defined $self->{remoteIPs} ||
            !defined $self->{remoteIPs}->{$hostname}) {

            $self->errorIPAddressNotReceived($hostname);
        }
        else {
            my @remoteIPs;
            my $pattern = '^' . $IPAddr . '\/';
            my $match   = 0;
            foreach my $currIP (@{$self->{remoteIPs}->{$hostname}}) {
               if (($currIP =~ /$pattern/) || ($currIP eq $IPAddr)) {
                   $inspectedHostMap->{$hostname} = $IPAddr;
                   $match = 1;
                   last;
               }
               elsif (($currIP =~ /^\w/) && ($currIP !~ /^127/)) {
                       push @remoteIPs, $currIP;
               }
            }
            if (!$match) {
                $self->errorIPAddressInvalid($IPAddr,
                                             $hostname,
                                             join("', '", @remoteIPs));
            }
            else {
                push @{$self->{_infoFile}}, $self->{_localhost}
                     . " - IP address '$IPAddr' of host '$hostname'\n";
            }
        }
    }

    if ($rc) {
        my @tab;
        push @tab, ['Remote Host', 'IP Address'];
        foreach my $currHost (sort @$remoteHostnames) {
            my $ip = $inspectedHostMap->{$currHost};
            $ip    = '<invalid>' if (!defined $ip);
            push @tab, [$currHost, $ip];
        }
        $self->printTab($msglst, \@tab);
    }

    $msg->endMessage(1, 'IP addresses checked: '
                        . $self->getErrCntInfo($startErrCnt));
    $self->setMsgLstContext($saveCtxt);

    return $rc;
}


#-------------------------------------------------------------------------------
# Checks existence of persistent files for hdbaddhost, hdbreg and hdbrename

sub inspectPendingPrograms {
    my ($self) = @_;

    my $msg = $self->AddProgressMessage ('Checking programs (check pending)...');

    my $startErrCnt     = $self->{_errorCnt};
    my $msglst          = $msg->getSubMsgLst();
    my $saveCtxt        = $self->setMsgLstContext([$msglst]);
    my $sapSys          = $self->getSAPSystem();
    my $program         = 'hdaddhost';
    my $pendingAddHosts = $sapSys->getPendingAddHosts(1);

    push @{$self->{_infoFile}},
         $self->{_localhost} . " - Program '$program' (check pending)\n";

    if (defined $pendingAddHosts) {
        foreach my $currHost (keys %$pendingAddHosts) {
            $self->errorPendingProgram($program,
                                       $currHost,
                                       $pendingAddHosts->{$currHost});
        }
    }

    require SDB::Install::Configuration::HdbReg;
    require SDB::Install::Configuration::Rename;

    $self->inspectPendingregisterRename($sapSys, 'hdbreg', 'Register',
                                     new SDB::Install::Configuration::HdbReg());

    $self->inspectPendingregisterRename($sapSys, 'hdbrename', 'Rename',
                                     new SDB::Install::Configuration::Rename());

    $msg->endMessage(1, 'Programs checked: '
                        . $self->getErrCntInfo($startErrCnt));
    $self->setMsgLstContext($saveCtxt);
}


#-------------------------------------------------------------------------------
sub inspectPendingregisterRename {
    my ($self,
        $sapSys,
        $program,           # e.g. 'hdbreg'
        $action,            # e.g. 'Register'
        $programInstConfig, # new SDB::Install::Configuration::HdbReg()
       ) = @_;

    push @{$self->{_infoFile}},
         $self->{_localhost} . " - Program '$program' (check pending)\n";

    $programInstConfig->{sapSys} = $sapSys;
    my $outstandingHosts         = $programInstConfig->getOutstandingHosts(1);

    if (defined $outstandingHosts) {
        foreach my $currHost (@$outstandingHosts) {
            my $errorText = "$action is pending on host '$currHost'";
            $self->errorPendingProgram($program, $currHost, $errorText);
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
sub inspectRemoteHosts{
    my ($self,
        $app, # SDB::Install::App::Console::HdbCheck
       ) = @_;

    my $classRemoteHosts = $self->getRemoteHosts();

    if (!defined $classRemoteHosts) {
        # is undef in case of single-host system and in case of 'scope=instance'
        return 1;
    }

    print "\n";

    my $instance = $self->getOwnInstance();
    my $dataPath = $instance->getDataPath(1);
    my $logPath  = $instance->getLogPath(1);
    my $persData = $self->createPersData($instance);

    foreach my $currHost (sort @{$instance->get_hosts()}) {

        $self->writePersFile($instance->get_hostNameDir($currHost),
                             $CHECK_FILENAME,
                             $persData);

        if ($self->{_isColumnStoreHost}->{$currHost}) {
            my $filename = $CHECK_FILENAME . '_' . $currHost;
            $self->writePersFile($dataPath, $filename, $persData);
            $self->writePersFile($logPath,  $filename, $persData);
        }
    }

    foreach my $currHost (sort @{$instance->get_hosts()}) {

        my $cmdLineParams    = $self->getOpt('SlaveHostName') . '=' . $currHost;
        my $hostctrlParamMap = { $currHost => {'HOSTNAME' => $currHost} };

        my $rc = $self->excuteRemoteSerialOrParallel
                                    ($app->getActionProgressive(),
                                     $app->getActionDone(),
                                     $CHECK_FILENAME,
                                     'HdbCheckHost',
                                     $gOperationCheckHost,
                                     ['HostagentPassword', 'Password'],
                                     ['PropertyFile'],
                                     $cmdLineParams,
                                     undef, # $remoteHosts
                                     undef, # $remoteInstconfig
                                     $hostctrlParamMap,
                                     undef, # $templateParamRepl
                                     [$currHost],
                                    );
        if (!$rc) {
            $self->incrErr("Checking remote host '$currHost' failed");
        }
        $self->readSlaveErrorFile($currHost);
        my $infoFile = $self->getSlaveInfoFilePath($currHost);
        if (!-f $infoFile) {
            $self->errorHostReply($currHost);
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
sub inspectSubDirectory {
    my ($self,
        $wantedDir,  # path of the directory to be checked
        $entriesFromDisk,
        $dirDesc,
        $indent,
        $msglst,
       ) = @_;

    my %checkedFileMap;
    my ($dev, $ino, $mode, $nlink, $uid, $gid, $modeFromDisk);
    my $anyPattern = ($dirDesc->{anyPattern}) ? 1 : 0;

    foreach my $filenameFromDisk (sort @$entriesFromDisk) {

        my $currPath = $wantedDir . $path_separator . $filenameFromDisk;
        my $currDesc = $dirDesc->{content}->{$filenameFromDisk};

        if (defined $currDesc) {
            $checkedFileMap{$filenameFromDisk} = 1;
        }
        elsif ($anyPattern) {
            foreach my $pattern (keys %{$dirDesc->{content}}) {
                if ($dirDesc->{content}->{$pattern}->{pattern}
                    && ($filenameFromDisk =~ /$pattern/)) {
                    $currDesc = $dirDesc->{content}->{$pattern};
                    $checkedFileMap{$pattern} = 1;
                    last;
                }
            }
        }

        if (defined $currDesc) {

            if ($currDesc->{type} eq 'link') {
                if (!-l $currPath) {
                    $self->errorFileNotFound($currPath, 'link', $indent);
                    next;
                }
                ($dev, $ino, $mode, $nlink, $uid, $gid) = lstat(_);
                $mode &= 07777;
                $modeFromDisk = sprintf("%o", $mode);
                my $linkFromDisk = readlink $currPath;
                if (!defined $linkFromDisk) {
                    $self->errorFileIO($currPath, 'link', 'read', $!, $indent);
                }
                elsif ($linkFromDisk ne $currDesc->{target}) {
                    $self->errorFileLink($currPath,
                                         $linkFromDisk,
                                         $currDesc->{target},
                                         $indent);
                }
                else {
                    $self->addFileMsg($msglst,
                                      $indent,
                                      $currPath,
                                      $filenameFromDisk,
                                      $linkFromDisk,
                                      $currDesc->{type},
                                      $modeFromDisk,
                                      $uid,
                                      $gid);
                }
            }
            else {
                if (($currDesc->{type} eq 'subdir')) {
                    if (!-d $currPath) {
                        $self->errorFileNotFound($currPath,
                                                 'directory',
                                                 $indent);
                        next;
                    }
                }
                elsif (($currDesc->{type} eq 'file') && !-f $currPath) {
                    $self->errorFileNotFound($currPath, 'file', $indent);
                    next;
                }
                ($dev, $ino, $mode, $nlink, $uid, $gid) = stat(_);
                $mode &= 07777;
                $modeFromDisk = sprintf("%o", $mode);

                $self->addFileMsg($msglst,
                                  $indent,
                                  $currPath,
                                  $filenameFromDisk,
                                  undef,            # $linkname
                                  $currDesc->{type},
                                  $modeFromDisk,
                                  $uid,
                                  $gid);
            }

            if (defined($currDesc->{min_mode})) {
                my $minimalMode = oct($currDesc->{min_mode});
                my $targetMode = $mode & $minimalMode;
                my $fileType = $self->getStrFiletype($currDesc);

                if ($targetMode != $minimalMode) {
                    $self->errorFileMinimalMode($currPath, $fileType, $modeFromDisk, $currDesc->{min_mode}, $indent);
                }
            }

            if (defined($currDesc->{max_mode})) {
                my $maximalMode = oct($currDesc->{max_mode});
                my $targetMode = $maximalMode | $mode;
                my $fileType = $self->getStrFiletype($currDesc);

                if ($targetMode != $maximalMode) {
                    $self->errorFileMaximalMode($currPath, $fileType, $modeFromDisk, $currDesc->{max_mode}, $indent);
                }
            }

            my $wantedUID = (defined $currDesc->{uid}) ? $currDesc->{uid}
                                                       : $self->{_sidadmUID};

            if (defined $wantedUID && ($wantedUID ne $uid)) {
                $self->errorFileUID($currPath,
                                    $self->getStrFiletype($currDesc),
                                    $uid,
                                    $wantedUID,
                                    $indent);
            }

            my $wantedGID = (defined $currDesc->{gid}) ? $currDesc->{gid}
                                                       : $self->{_sidadmGID};

            if (defined $wantedGID && ($wantedGID ne $gid)) {
                $self->errorFileGID($currPath,
                                    $self->getStrFiletype($currDesc),
                                    $gid,
                                    $wantedGID,
                                    $indent);
            }
        }
    }

    foreach my $currName (sort keys %{$dirDesc->{content}}) {

        my $currDesc = $dirDesc->{content}->{$currName};

        if (!$checkedFileMap{$currName} && !$currDesc->{optional}) {
            if ($currDesc->{pattern}) {
                if (!($currName =~ s/\(.*\)\?\\d\*/*/)) {     # 'HdbAbc(_v\d)?\d*.conf' -> 'HdbAbc*.conf'
                    if (!($currName =~ s/\\d\+/*/)) {         # 'HdbXyz_v\d+.conf'      -> 'HdbXyz_v*.conf'
                        $currName =~ s/\^\\w\[\\w\\\.\]\+/*/; # '^\w[\w\.]+_abc'        -> '*_abc'
                    }
                }
            }
            $self->errorFileNotFound($wantedDir . $path_separator . $currName,
                                     $self->getStrFiletype($currDesc),
                                     $indent);
        }
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Collected host info:
#                           user             host       userID  groupID
# $self->{remoteSidadms}->{'db1adm'}->[0]->['lu123456',  1010,  79]
#                                     [1]->['lu222222',  1010,  79]
#                         {'db6adm'}->[0]->['lu222222',  1003,  79]
#                         {'db7adm'}->[0]->['lu123456',  1002,  79]
#                                     [1]->['lu222222',  1002,  79]
#
# $self->{sapsysids}->{79}->['lu123456', 'lu222222']

sub inspectSystemAdministrator {
    my ($self) = @_;

    print "\n";
    my $msg = $self->AddProgressMessage ('Checking ' . lc ($STR_SIDADM)
                                         . " '$self->{_sidadmName}'...");

    my $msglst      = $msg->getSubMsgLst();
    my $saveCtxt    = $self->setMsgLstContext([$msglst]);
    my $startErrCnt = $self->{_errorCnt};
    my $sapSys      = $self->getSAPSystem();
    my $sapsysGID   = getgrnam ($gSapsysGroupName);
    my $instance    = $self->getOwnInstance();
    my $ownHost     = $self->{_localhost};

    if (!defined $sapsysGID) {
        $self->errorGroupNotFound($gSapsysGroupName);
    }

    my $cfg = $sapSys->getUserConfig();

    if (!defined $cfg) {
        $self->errorFileIO($sapSys->getUserConfigFile(),
                           'file',
                           'read',
                           undef,  # $errorStr
                           undef,  # $indent
                           $sapSys);
    }

    my $cfgUID = $self->{_sidadmUID};
    my $cfgGID = $self->{_sidadmGID};

    if (defined $cfgGID && defined $sapsysGID && ($cfgGID ne $sapsysGID)) {
        $self->errorGroupID($ownHost, $gSapsysGroupName, $sapsysGID, $cfgGID);
    }

    my $infoPrefixUser   = $self->{_localhost} . " - $STR_SIDADM '$self->{_sidadmName}'";
    my $infoPrefixSapsys = $self->{_localhost} . " - Group '$gSapsysGroupName'";
    my @groupInfo;

    if (defined $sapsysGID) {
        push @groupInfo, $infoPrefixSapsys
            . " ($sapsysGID) of host '$self->{_localhost}'\n";
    }

    my @tab;
    push @tab, ['Source',
                'User',
                'User ID',
                'Group ID',
                "ID of Group '$gSapsysGroupName'"];

    if (defined $cfgUID && defined $cfgGID) {
        push @tab, ["File 'cfg'",
                    $self->{_sidadmName},
                    $cfgUID,
                    $cfgGID,
                    '--'
                   ];
    }

    my $ownUserExists = $self->{_sidadmUser}->exists();
    my $ownUID        = $self->{_sidadmUser}->id();
    my $ownGID        = $self->{_sidadmUser}->gid();

    if (!$ownUserExists) {
        $self->errorSidadmNotFound();
    }
    else {
        if (defined $ownUID && defined $cfgUID && ($ownUID ne $cfgUID)) {
            $self->errorSidadmUID($ownHost, $ownUID, $cfgUID);
        }

        if (defined $ownGID && defined $cfgGID && ($ownGID ne $cfgGID)) {
            $self->errorSidadmGID($ownHost, $ownGID, $cfgGID);
        }
        push @{$self->{_infoFile}}, $infoPrefixUser
            . " ($ownUID $ownGID) of host '$self->{_localhost}'\n";
    }

    push @tab, ["Host '$ownHost'",
                ($ownUserExists) ? $self->{_sidadmName} : $STR_UNDEF,
                ($ownUserExists && defined $ownUID) ? $ownUID    : $STR_UNDEF,
                ($ownUserExists && defined $ownGID) ? $ownGID    : $STR_UNDEF,
                (defined $sapsysGID)                ? $sapsysGID : $STR_UNDEF,
               ];

    if (defined $self->getRemoteHosts()) {

        my %hostSidadmMap;
        my %hostSapsysMap;

        # $self->{remoteSidadms}->{'db1adm'}->[0]->['lu123456',  1010,  79]
        #                                     [1]->['lu222222',  1010,  79]

        my $sidadmArray = (defined $self->{remoteSidadms})
                          ? $self->{remoteSidadms}->{$self->{_sidadmName}}
                          : undef;

        if (defined $sidadmArray) {

            foreach my $sidadmEntry (@$sidadmArray) {
                my $currHost = $sidadmEntry->[0];
                $hostSidadmMap{$currHost} = {'userid'  => $sidadmEntry->[1],
                                             'groupid' => $sidadmEntry->[2]};
            }
        }

        if (defined $self->{sapsysids}) {

            # $self->{sapsysids}->{79}->['lu123456', 'lu222222']

            foreach my $currGID (keys %{$self->{sapsysids}}) {

                foreach my $currHost (@{$self->{sapsysids}->{$currGID}}) {
                    $hostSapsysMap{$currHost} = $currGID;
                }
            }
        }

        my $remoteHotsNames = $self->getRemoteHosts()->getHostNames();

        foreach my $currHost (sort @$remoteHotsNames) {

            my $currSidadm    = $hostSidadmMap{$currHost};
            my $currSapsysGID = $hostSapsysMap{$currHost};
            my $currUID       = undef;
            my $currGID       = undef;

            if (!defined $currSidadm) {
                $self->errorSidadmNotReceived($currHost);
            }
            else {
                $currUID = $currSidadm->{userid};
                $currGID = $currSidadm->{groupid};

                if (defined $currUID && defined $cfgUID && ($currUID ne $cfgUID)) {
                    $self->errorSidadmUID($currHost, $currUID, $cfgUID);
                }

                if (defined $currGID && defined $cfgGID && ($currGID ne $cfgGID)) {
                    $self->errorSidadmGID($currHost, $currGID, $cfgGID);
                }
            }

            if (!defined $currSapsysGID) {
                $self->errorGroupNotReceived($currHost, $gSapsysGroupName);
            }
            elsif (defined $cfgGID && $currSapsysGID ne $cfgGID) {
                $self->errorGroupID($currHost,
                                    $gSapsysGroupName,
                                    $currSapsysGID,
                                    $cfgGID);
            }

            my $uidStr   = (defined $currUID)      ? $currUID      : $STR_UNDEF;
            my $gidStr   = (defined $currGID)      ? $currGID      : $STR_UNDEF;
            my $sapsysStr= (defined $currSapsysGID)? $currSapsysGID: $STR_UNDEF;

            push @tab, ["Host '$currHost'",
                        (defined $currSidadm) ? $self->{_sidadmName}: $STR_UNDEF,
                        $uidStr,
                        $gidStr,
                        $sapsysStr,
                       ];

            if (defined $currSidadm) {
                push @{$self->{_infoFile}}, $infoPrefixUser
                    . " ($uidStr $gidStr) of host '$currHost'\n";
            }
            if (defined $currSapsysGID) {
                push @groupInfo, $infoPrefixSapsys
                    . " ($currSapsysGID) of host '$currHost'\n";
            }
        }
    }

    foreach my $currLine (@groupInfo) {
        push @{$self->{_infoFile}}, $currLine;
    }

    $self->printTab($msglst, \@tab);

    $msg->endMessage(1, "$STR_SIDADM '$self->{_sidadmName}' checked: "
                        . $self->getErrCntInfo($startErrCnt));
    $self->setMsgLstContext($saveCtxt);

    return 1;
}


#-------------------------------------------------------------------------------
sub inspectVolumes {
    my ($self) = @_;

    print "\n";
    my $msg         = $self->AddProgressMessage('Checking volumes...');
    my $msglst      = $msg->getSubMsgLst();
    my $saveCtxt    = $self->setMsgLstContext([$msglst]);
    my $startErrCnt = $self->{_errorCnt};
    my $instance    = $self->getOwnInstance();
    my $filetype    = 'directory';
    my $indent      = '    ';
    my $localRoles  = $self->{_hostRolesMap}->{$self->{_localhost}};

    my ($dev, $ino, $mode, $nlink, $uid, $gid, $modeFromDisk, @volumePaths);

    foreach my $currRoleInfo (values %$localRoles) {

        if ($currRoleInfo->{isColumnStore}) {
            push @volumePaths, $instance->getDataPath(1);
            push @volumePaths, $instance->getLogPath(1);
            my $dataBackup =   $instance->getDataBackupPath(1);
            my $logBackup  =   $instance->getLogBackupPath (1);
            push @volumePaths, $dataBackup if (defined $dataBackup && -e $dataBackup);
            push @volumePaths, $logBackup  if (defined $logBackup && -e $logBackup);
        }
        elsif ($currRoleInfo->{isAccelerator}) {
            push @volumePaths, $instance->getAcceleratorDataPath(1);
            push @volumePaths, $instance->getAcceleratorLogPath(1);
        }
        elsif  ($currRoleInfo->{isExtendedStorage}) {
            push @volumePaths, $instance->getEsDataPath(1);
            push @volumePaths, $instance->getEsLogPath(1);
        }
    }

    foreach my $currPath (@volumePaths) {
        if (!-d $currPath) {
            $self->errorFileNotFound($currPath, $filetype, $indent);
            next;
        }

        ($dev, $ino, $mode, $nlink, $uid, $gid) = stat(_);

        my $modeFromDisk = sprintf("%o", $mode &07777);

        $self->addFileMsg($msg->getSubMsgLst(),
                          $indent,
                          $currPath,
                          undef,     # $filename
                          undef,     # $linkname
                          $filetype,
                          $modeFromDisk,
                          $uid,
                          $gid,
                          1); # $display

        if (($mode & 0700) != 0700){
            $self->errorFileMode($currPath,
                                 $filetype,
                                 $modeFromDisk,
                                 '7**',
                                 $indent);
        }

        if ($uid ne $self->{_sidadmUID}) {
            $self->errorFileUID($currPath,
                                $filetype,
                                $uid,
                                $self->{_sidadmUID},
                                $indent);
        }

        if ($gid ne $self->{_sidadmGID}) {
            $self->errorFileGID($currPath,
                                $filetype,
                                $gid,
                                $self->{_sidadmGID},
                                $indent);
        }
    }

    $msg->endMessage(1, 'Volumes checked: '.$self->getErrCntInfo($startErrCnt));
    $self->setMsgLstContext($saveCtxt);

    return 1;
}


#-------------------------------------------------------------------------------
sub performSystemCheck {
    my ($self,
        $app, # SDB::Install::App::Console::HdbCheck
       ) = @_;

    my $xmlParser = $self->getXMLParser($self->getValue('PropertyFile'));
    my $xmlVars   = $self->getXMLVariables();

    if ($self->{isSlave}) {
        if (!$self->readAndDeletePersFiles()) {
            return undef
        }
    }
    else {
        if (!$self->inspectPendingPrograms()) {
            return undef;
        }
        if (!$self->inspectSystemAdministrator()) {
            return undef;
        }
    }

    if (!$self->inspectIPAddresses()) {
        return undef;
    }

    if (defined $xmlParser
        && !$self->inspectDirectories($xmlParser,
                                      $xmlVars,
                                      'local directories',
                                      $TAG_LOCAL_DIR,
                                      $xmlVars->{'localDir'})) {
        return undef;
    }

    if (!$self->{isSlave} && defined $xmlParser
        && !$self->inspectDirectories($xmlParser,
                                      $xmlVars,
                                      'global directories',
                                      $TAG_GLOBAL_DIR,
                                      $xmlVars->{'globalDir'})) {
        return undef;
    }

    if (defined $xmlParser
        && $self->{_isHANAOptionsHost}->{$self->{_localhost}}
        && !$self->inspectDirectories($xmlParser,
                                      $xmlVars,
                                      "directories of $gProductName options",
                                      $TAG_HANA_OPTIONS_DIR,
                                      $xmlVars->{'globalDir'})) {
        return undef;
    }

    if (!$self->inspectVolumes()) {
        return undef;
    }

    if (!$self->{isSlave} && !$self->inspectRemoteHosts($app)) {
        return undef;
    }

    print "\n";
    my $programName = $app->getProgramName($self->{_localhost});

    $self->writeInfoFile($programName);

    if (defined $self->{_errors}) {
        print "\n";
        $self->writeXMLErrorFile($programName);
        if (!$self->{isSlave}) {
            $self->writeErrorFile($programName);
        }
    }

    print "\n";
    $self->AddProgressMessage($self->getErrCntInfo());

    return 1;
}


#-------------------------------------------------------------------------------
sub printTab {
    my ($self, $msglst, $tab) = @_;

    $msglst->addProgressMessage('');
    foreach my $line (@{printTableToArrayOfLines ($tab, ' | ', 1)}){
        $msglst->addProgressMessage($line);
        #print $line . "\n";
        #$self->AddSubMessage($msg, $line);
    }
    $msglst->addProgressMessage('');
}


#-------------------------------------------------------------------------------
sub readAndDeletePersFiles {
    my ($self) = @_;

    my $instance     = $self->getOwnInstance();
    my $hostnames    = $instance->get_allhosts();
    my $hostMap      = {};
    my $mainPersFile = $instance->get_hostNameDir()
                       . $path_separator . $CHECK_FILENAME;
    my @persFiles;

    foreach my $currHost (@$hostnames) {
        $hostMap->{$currHost} = 1;
    }

    push @persFiles, $mainPersFile;

    my $isBasepathShared = $instance->isBasePathShared();
    my $usesStorageApi = $instance->usesStorageApi();
    if (!defined $isBasepathShared || !defined $usesStorageApi) {
        $self->incrErr("Failed to determine basepath info and storage API settings from global.ini");
        return 1;
    }
    if ($self->{_isColumnStoreHost}->{$self->{_localhost}} && ($isBasepathShared && !$usesStorageApi)) {
        my $filename  = $CHECK_FILENAME . '_' . $self->{_localhost};
        push @persFiles, $instance->getDataPath(1). $path_separator . $filename;
        push @persFiles, $instance->getLogPath(1) . $path_separator . $filename;
    }

    foreach my $persFile (@persFiles) {

        if ($self->pers_exists($persFile)) {

            my $persData = $self->pers_load($persFile);

            if (!defined $persData || !defined $persData->{actionID}) {
                $self->errorFileIO($persFile, 'file', 'read');
            }
            elsif ($persData->{actionID} ne $self->{options}->{action_id}) {
                $self->errorPersFileInvalid($persFile,
                                            $persData->{actionID},
                                            $self->{options}->{action_id});
            }
            elsif ($persFile eq $mainPersFile) {
                foreach my $key (keys %$persData) {
                    if ($hostMap->{$key}) {
                        # persData key is a hostname
                        my @IPAddrArray = split(',', $persData->{$key});
                        $self->{remoteIPs}->{$key} = \@IPAddrArray;
                    }
                }
            }

            if (!$self->pers_remove($persFile)) {
                $self->errorFileIO($persFile, 'file', 'remove');
            }
        }
        else {
             $self->errorFileNotFound($persFile, 'file', '',
                                      'check mounted directory');
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
sub readSlaveErrorFile {
    my ($self, $host) = @_;

    my $xmlParser = undef;
    my $errFile   = $self->getSlaveCopyFilepath
                                    ($host,
                                     $CHECK_FILENAME . '_' . $host,
                                     $ERR_XML_EXTENSION);

    if (-f $errFile) {
        $xmlParser = $self->getXMLParser($errFile);
        next if (!defined $xmlParser);
        my $xml = $xmlParser->getElementByTagName($TAG_ERRORS);
        next if (!defined $xml);
        $self->createErrorEntriesFromXML($xml);
    }

    return 1;
}


#-------------------------------------------------------------------------------
sub setPassword {
    my ($self, $sidadmPassword) = @_;

    my $rc = $self->checkPassword($sidadmPassword);

    if ($rc) {
        $self->{params}->{Password}->{value} = $sidadmPassword;

        if ($self->isUseSAPHostagent() && defined $self->getRemoteHosts()) {
            if (!$self->CollectOtherHostInfos()) {
                $self->incrErrShowMsglst();
            }
        }
    }
    return $rc;
}


#-------------------------------------------------------------------------------
sub writeAnyFile {
    my ($self,
        $fileInfo,  # e.g. 'List of errors'
        $filename,  # e.g. 'hdbcheck'
        $extension, # e.g. '.err'
        $buffers,   # array of array of lines: each line should contain "\n"
       ) = @_;

    if (!defined $buffers || !@$buffers) {
        return 1;
    }

    my $contentFound = 0;
    foreach my $lineArray (@$buffers) {
        if (defined $lineArray && @$lineArray) {
            $contentFound = 1;
            last;
        }
    }

    return 1 if (!$contentFound);

    my $filepath = $gLogDir . $path_separator . $filename . $extension;

    if (!open (FD, ">$filepath")) {
       $self->AddMessage("Cannot create file '$filepath': $!");
       return undef;
    }

    foreach my $lineArray (@$buffers) {
        if (! print FD @$lineArray) {
            $self->AddMessage ("Cannot write file '$filepath': $!");
            close (FD);
            return undef;
        }
    }

    close (FD);

    if ($> == 0) { # $> (effective UID) == 0 (root user id)

        chmod(0640, $filepath); # rw-r-----
        if (defined $self->{_sidadmGID}) {
            chown(-1, $self->{_sidadmGID}, $filepath);
        }
        # ignore chown errors
    }

    $self->AddProgressMessage("$fileInfo written to '$filepath' on host '"
                              . $self->{_localhost} . "'.");

    if ($self->{isSlave}) {

        require SDB::Install::System;

        my $traceDir = $self->getOwnInstance()->get_hostNameTraceDir();
        my $oldFiles = getFilesTimestampPID($traceDir, $filename, $extension);

        foreach my $currOldFile (@$oldFiles) {
            unlink $traceDir . $path_separator . $currOldFile;
        }

        my $newFile = $self->getSlaveCopyFilepath($self->{_localhost},
                                                  $filename,
                                                  $extension);

        my $copyCfg     = {};
        $copyCfg->{uid} = $self->{_sidadmUID};
        $copyCfg->{gid} = $self->{_sidadmGID};

        if (!copy_file($filepath, $newFile, $copyCfg)) {
            # ignore copy error
            $self->AddMessage("Cannot copy file '$filepath' to '$newFile'",
                              $copyCfg);
        }
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Uses the hash '$self->{_errors}' to write a file containing one error per line.
#
# These variables are used when scanning the error hashes:
#
#                        $currHost  $currType  $currObj        $currErr
#                            |         |         |               |
# e.g. $self->{_errors}->{lu123456}->{file}->{'/usr/sap/...'}->{notFound}->{errorText}->'File...'
#      |----- $hostErrors --------|                         |
#      |----- $objErrors -----------------------------------|

sub writeErrorFile {
    my ($self, $programName) = @_;

    if (!defined $self->{_errors}) {
        return 1;
    }
    my @lines;

    foreach my $currHost (sort keys %{$self->{_errors}}) {
        my $hostErrors = $self->{_errors}->{$currHost};

        foreach my $currType (sort keys %{$hostErrors}) {

            foreach my $currObj (sort keys %{$hostErrors->{$currType}}) {
                my $objErrors = $hostErrors->{$currType}->{$currObj};

                foreach my $currErr (sort keys %{$objErrors}) {
                    my $errTxt = $objErrors->{$currErr}->{errorText};
                    if (defined $errTxt) {
                        push @lines, $currHost . ' - ' . $errTxt . "\n";
                    }
                }
            }
        }
    }

    return $self->writeAnyFile('List of errors',
                               $programName,
                               $ERR_EXTENSION,
                               [\@lines]);
}


#-------------------------------------------------------------------------------
sub writeInfoFile {
    my ($self, $programName) = @_;

    foreach my $currPath (sort keys %{$self->{_checkedFiles}}) {
        push @{$self->{_infoFile}}, $self->{_localhost} . ' - '
                                    . $self->{_checkedFiles}->{$currPath}
                                    . $currPath . "\n";
    }

    my $fileBuffers = [];

    if ($self->{isSlave} || !defined $self->getRemoteHosts()) {
        push @$fileBuffers, $self->{_infoFile};
    }
    else {
        my $allHosts = $self->getOwnInstance()->get_allhosts();

        foreach my $currHost (sort @$allHosts) {

            if ($currHost eq $self->{_localhost}) {
                push @$fileBuffers, $self->{_infoFile};
            }
            else {
                my $infoFilePath = $self->getSlaveInfoFilePath($currHost);
                if (-f $infoFilePath) {
                    if (!open (FD, $infoFilePath)){
                        $self->AddMessage("Cannot open '$infoFilePath': $!");
                        next;
                    }
                    my @buffer = <FD>;
                    close (FD);
                    push @$fileBuffers, \@buffer;
                    unlink $infoFilePath;
                }
            }
        }
    }

    return $self->writeAnyFile('List of checked elements',
                               $programName,
                               $INFO_EXTENSION,
                               $fileBuffers);
}


#-------------------------------------------------------------------------------
sub writePersFile {
    my ($self, $dirPath, $filename, $persData) = @_;

    if (defined $dirPath) {
        my $persFile = $dirPath . $path_separator . $filename;

        unlink $persFile if (-f $persFile); # ignore error

        if (!$self->pers_store($persFile, $persData)) {
            $self->errorFileIO($persFile, 'file', 'write');
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
sub writeXMLErrorFile {

    my ($self, $programName) = @_;

    my @xmlStream = ('<?xml version="1.0" encoding="UTF-8"?>' . "\n",
                     "<$TAG_ERRORS>\n");
    my $hostIndent   = '    ';
    my $objIndent    = $hostIndent . '    ';
    my $errIndent    = $objIndent  . '    ';
    my $errTxtIndent = $errIndent  . '    ';

    foreach my $host (sort keys %{$self->{_errors}}) {
        my $objTagMap  = $self->{_errors}->{$host};
        push @xmlStream, "$hostIndent<host name=\"$host\">\n";

        foreach my $objTag (sort keys %$objTagMap) {
            my $xmlObjOutPrefix = "$objIndent<$objTag ";
            $xmlObjOutPrefix   .= ($objTag eq 'IPAddress') ? 'remoteHost'
                                                           : 'name';
            foreach my $objName (sort keys %{$objTagMap->{$objTag}}) {
                my $errTagMap  = $objTagMap->{$objTag}->{$objName};
                push @xmlStream,
                     $xmlObjOutPrefix . '="' . $objName . '">' . "\n";

                foreach my $errTag (sort keys %$errTagMap) {
                    my $xmlErrOutput = "$errIndent<$errTag";
                    my $errTxt       = '';
                    foreach my $errAttr (sort keys %{$errTagMap->{$errTag}}) {
                        my $attrValue = $errTagMap->{$errTag}->{$errAttr};
                        if ($errAttr eq 'errorText') {
                            $errTxt = $attrValue;
                        }
                        else {
                            $xmlErrOutput .= " $errAttr=\"$attrValue\"";
                        }
                    }
                    push @xmlStream, $xmlErrOutput . ">\n";
                    push @xmlStream, "$errTxtIndent<errorText>$errTxt</errorText>\n";
                    push @xmlStream, "$errIndent</$errTag>\n";
                }
                push @xmlStream, "$objIndent</$objTag>\n";
            }
        }
        push @xmlStream, "$hostIndent</host>\n";
    }
    push @xmlStream, "</$TAG_ERRORS>\n";

    return $self->writeAnyFile('XML error file',
                               $programName,
                               $ERR_XML_EXTENSION,
                               [\@xmlStream]);
}

1;
