package SDB::Install::Configuration::UpdRep;

use strict;
use SDB::Install::Configuration::AnyConfig;
use SDB::Install::SAPSystem;
use SDB::Install::SysVars qw ($path_separator);
use SDB::Install::Sql::SqlConnectionProvider;

our @ISA = qw (SDB::Install::Configuration::AnyConfig);
our $section = 'import_content';

sub new {
	my $self = shift->SUPER::new (@_);
	my $order = 0;
	$self->{params} = {
		'SID' => $self->getParamSID($order++, $section),
		'UserStoreKey' => {
			'order' => $order++,
			'opt' => 'user_store_key',
			'short_opt' => 'U',
			'type' => 'string',
			'section' => $section,
			'value' => undef,
			'default' => undef,
			'str' => 'User store key',
            'opt_arg' => '<name>',
			'init_with_default' => 0,
			'set_interactive' => 0,
			'mandatory' => 0
		},
		'SystemUser' => $self->getParamSystemUser($order++, $section),
		'SQLSysPasswd' => $self->getParamSQLSysPasswd ($order++, $section, 'passwd', 0),
		'ContentDir' => {
			'order' => $order++,
			'opt' => 'content_directory',
			'type' => 'path',
			'section' => $section,
			'value' => undef,
			'default' => undef,
			'str' => 'Directory containing delivery units',
            'opt_arg' => '<path>',
			'set_interactive' => 0,
			'init_with_default' => 0
		},
		'DeliveryUnit' => {
			'order' => $order++,
			'opt' => 'delivery_unit',
			'type' => 'path',
			'section' => $section,
			'value' => undef,
			'default' => undef,
			'str' => 'Delivery unit file',
            'opt_arg' => '<file_name>',
			'set_interactive' => 0,
			'init_with_default' => 0
		}
	};
	return $self;
}

sub isUpdate{
	return 1;
}

sub checkContentDir{
	my (
		$self,
		$value
	) = @_;
	if (!defined $value) {
		return 1;
	}
	if (defined $self->{params}->{DeliveryUnit}->{value}) {
		$self->PushError ("Cannot process both content_directory and delivery_unit, choose one");
		return 0;
	}
	my $absPath = $self->getAbsPath ($value);
	if (! -d $absPath) {
		$self->PushError ("Content directory $value not found");
		return 0;
	}
	return 1;
}

sub setContentDir{
	my (
		$self,
		$value
	) = @_;

	if ($self->checkContentDir($value)) {
		my $absPath = $self->getAbsPath ($value);
		$self->{params}->{ContentDir}->{value} = $absPath;
		return 1;
	} else {
		return 0;
	}
}

sub checkDeliveryUnit{
	my (
		$self,
		$value
	) = @_;

	if (!defined $value) {
		return 1;
	}
	if (defined $self->{params}->{ContentDir}->{value}) {
		$self->PushError ("Cannot process both delivery_unit and content_directory, choose one");
		return 0;
	}
	my $absPath = $self->getAbsPath ($value);
	if (! -f $absPath) {
		$self->PushError ("Delivery unit $value not found");
		return 0;
	}
	return 1;
}

sub setDeliveryUnit{
	my (
		$self,
		$value
	) = @_;

	if ($self->checkDeliveryUnit($value)) {
		my $absPath = $self->getAbsPath ($value);
		$self->{params}->{DeliveryUnit}->{value} = $absPath;
		return 1;
	} else {
		return 0;
	}
}


sub checkUserStoreKey{
    my (
        $self,
        $value
    ) = @_;
    if ($value eq ''){
        $self->PushError ("User store key is empty");
        return 0;
    }

    $self->setSkip ('SystemUser');
    $self->setSkip ('SQLSysPasswd');

    my $instance = $self->getOwnInstance();
    my $connectionProvider = new SDB::Install::Sql::SqlConnectionProvider();
    my $sql = $connectionProvider->getSqlConnection(
                undef,    # db name
                $instance,
                undef,    # user
                undef,    # password
                $value);


    if (!defined $sql){
        if($connectionProvider->isECONNREFUSED ()) {
            $self->setErrorMessage ("No connection available", $connectionProvider->getSqlObj()->getErrMsgLst());
            $self->{params}->{UserStoreKey}->{no_retry} = 1;
            return 0;
        }
        $self->appendErrorMessage ("Invalid user store key", $connectionProvider->getErrMsgLst());
        return 0;
    }
    $self->AddMessage("User store key verified.");
    #
    # check whether the system is locked down
    #
    my $license_check_rc = $self->licenseCheck($instance, $sql);
    if (!$license_check_rc){
        $self->{params}->{UserStoreKey}->{no_retry} = 1;
    }
    return $license_check_rc;
}

sub licenseCheck{
	my (
		$self,
		$instance,
		$sql
	) = @_;

        my $msglst = new SDB::Install::MsgLst();
        my $license_check_rc = $instance->checkLicense ($sql,$msglst);
        if ($license_check_rc){
            $self->AddMessage ("License check successfully performed.");
        }
        else{
            $self->AddError (undef, $msglst);
            if ($self->getIgnore ('check_license')){
                $self->AddMessage ("Ignoring error due to command line switch '--ignore=check_license'");
                $license_check_rc = 1;
             }
        }
        return $license_check_rc;

}

sub checkSID {
   my ($self, $value) = @_;
   # check SID
   my $rc = $self->checkExistingSID ($value, $self->{params}->{SID}->{str});

   my $systems = CollectSAPSystems();
   my $system = $systems->{$value};
   if (!defined $system) {
       $self->AddError("Checking parameter SID failed.");
       return 0;
   }
   $self->{sapSys} = $system;
   my $instance = $self->getOwnInstance();
   my $errLst = new SDB::Install::MsgLst ();

	# check if this is a secondary system (replication scenario)
	my $secondary = $instance->isSecondarySystem($self->getMsgLst(), $errLst);

	if (!defined $secondary) {
		if ($self->getIgnore('check_secondary_system')) {
			$self->AddMessage ("Ignoring failed secondary system check due to switch ignore=check_secondary_system");
		}
		else{
			$self->appendErrorMessage("Error performing secondary system check", $errLst);
			$self->{params}->{SID}->{no_retry} = 1;
			$self->{params}->{SQLSysPasswd}->{skip} = 1;
			$self->{params}->{UserStoreKey}->{skip} = 1;
			return 0;
		}
	}
	if ($secondary) {
		if ($self->getIgnore('check_secondary_system')) {
			$self->AddMessage ("Ignoring result of secondary system check due to switch ignore=check_secondary_system");
		}
		else{
			$self->PushError("This is a secondary system");
			$self->{params}->{SID}->{no_retry} = 1;
			$self->{params}->{SQLSysPasswd}->{skip} = 1;
			$self->{params}->{UserStoreKey}->{skip} = 1;
			return 0;
		}
	}

   return $rc;
}

sub getIgnoreValues{
    return [qw (check_version check_secondary_system)];
}


#-------------------------------------------------------------------------------
# Returns undef (hdbupdrep does not start or stop a SAP HANA instance)

sub getTimeoutValues{
    return undef;
}

1;
