package SDB::Install::Configuration::Upgrade;

use strict;

our $section = 'Server';

our $statusFileExtension = 'hdbupgrade';



use SDB::Install::Configuration;
use SDB::Install::Configuration::AnyConfig;
use SDB::Install::Configuration::HdbReg;
use SDB::Install::Configuration::NewDB;
use SDB::Install::System qw (isAdmin);
use SDB::Install::SysVars;
use SDB::Install::Globals;
use SAPDB::Install::Hostname;
use SDB::Install::Configuration::ServerUpgradeChecker;

use SDB::Install::SAPSystem qw (CollectSAPSystems ExistsInstanceNumber
	$STEP_UPGRADE_EXTRACT
	@UPGRADE_STEP_NAMES @STEP_NAMES getPendingInfoFromPersFile);

our @ISA = qw (SDB::Install::Configuration::AnyConfig SDB::Install::Configuration::ServerUpgradeChecker);

#
# reserve well known installation names
#

sub InitDefaults{
	my ($self, $kit) = @_;
    $self->{kit} = $kit;
    if (!$isWin){
        if (!defined getgrnam ( $gSapsysGroupName)){
            $self->PushError ("Group '$gSapsysGroupName' not found");
            return undef;
        }
    }

	if (defined $kit) {
		$self->{_kitversion} = $kit->GetVersion ();
        $self->setValue ('versionId', $kit->getVersionIdString ());
        my $mf = $kit->getManifest();
        if (!defined $mf){
            $self->AddError (undef, $kit);
            return undef;
        }
        if (!$isWin && $> != 0 && $mf->isCustomerReleaseBranch ()){
            $self->PushError ("Wrong user: please start upgrade as root");
            return undef;
        }
    }

	if (!$isWin && $> != 0){
		my $name = getpwuid ($>);
		my ($sid) = ($name =~ /^(\w{3})adm$/);
		if (!$sid){
			$self->PushError ("Wrong user: please start upgrade as root or <sid>adm user");
			return undef;
		}
		$sid = uc ($sid);
		if (!$self->setValue ('SID', $sid)){
			return undef;
		}
	}
	return 1;
}

sub new{
	my $self = shift->SUPER::new (@_);
	my $order = 0;
	
	$self->{params} = {
		'CheckOnly'          => $self->getParamCheckOnly          ($order++, $section, 'upgrade'),
        'RemoteExecution'    => $self->getParamRemoteExecution    ($order++, $section),
        'UseHttp'            => $self->getParamUseHttp            ($order++, $section, 1, 0),
        'SAPControlUseHttp'  => $self->getParamSAPControlUseHttp  ($order++, $section),
        'InstallHostagent'   => $self->getParamInstallHostagent   ($order++, $section),
		'SID'                => $self->getParamSID                ($order++, $section, 1),
        'Scope'              => $self->getParamScope              ($order++, $section,
                                "Scope 'system' performs additional updates of operating system integration on remote hosts"),
        'ScopeInteractive'   => $self->getParamScopeInteractive   ($order++, $section,
                                'Update operating system integration on remote hosts'),
        'SSOCertificate'     => $self->getParamSSOCertificate     ($order++, $section),
		'Password'           => $self->getParamPassword           ($order++, $section),
        'StripSymbols'       => $self->getParamStripSymbols       ($order++, $section),
		'NoStart'            => $self->getParamNoStart            ($order++, $section,
		                        'Do not start the instance after upgrade', 'Does not start the instance after upgrade', 0),
		'ISCMode'            => $self->getParamISCMode            ($order++, $section),
		'MachineUtilization' => $self->getParamMachineUtilization ($order++, $section),
        'SecureStore'          => $self->getParamSecureStore($order++, $section),
        'ChangeInitialSSFSKey' => $self->getParamChangeInitialSSFSKey  ($order++, $section),
        'installType' => {
            'order' => $order++,
            'opt' => 'installType',
            'short_opt' => 'T',
            'type' => 'string',
            'section' => $section,
            'value' => undef,
            'default' => undef,
            'str' => 'Installation Type',
            'init_with_default' => 1,
            'set_interactive' => 0,
            'hidden' => 1,
            'persist_for_upgrade' => 1,
            'DEPRECATED' => 1
        },
        'EnableXSEngine' => {
            'order' => $order++,
            'opt' => 'xs_engine',
            'alias_opts' => ['xsengine','execution_service'],
            'type' => 'boolean',
            'section' => $section,
            'value' => undef,
            'default' => 1,
            'hidden' => 1,
            'str' => 'Enable XS Engine',
            'desc'=> 'Enables the XS engine',
            'init_with_default' => 0,
            'set_interactive' => 0
        },
        'XSEngineHttpPort' => {
            'order' => $order++,
            'opt' => 'xs_engine_http_port',
            'alias_opts' => ['execution_service_http_port'],
            'type' => 'number',
            'opt_arg' => '<port>',
            'section' => $section,
            'value' => undef,
            'default' => undef,
            'str' => 'XS Engine HTTP Port',
            'desc'=> 'Specifies the HTTP port of the XS engine',
            'init_with_default' => 0,
            'set_interactive' => 0
        },
        'XSEngineHttpsPort' => {
            'order' => $order++,
            'opt' => 'xs_engine_https_port',
            'alias_opts' => ['execution_service_https_port'],
            'type' => 'number',
            'opt_arg' => '<port>',
            'section' => $section,
            'value' => undef,
            'default' => undef,
            'str' => 'XS Engine HTTPS Port',
            'desc'=> 'Specifies the HTTPS port of the XS engine',
            'init_with_default' => 0,
            'set_interactive' => 0
        },
        'ImportContent'=> $self->getParamImportContent($order++,$section),
        'SystemUser'   => $self->getParamSystemUser  ($order++, $section),
        'SQLSysPasswd' => $self->getParamSQLSysPasswd($order++, $section, 'passwd', 1),
        # internal parameter to build versioned exe directory
        'versionId' => {
            'order' => $order++,
            'type' => 'string',
            'value' => undef,
            'default' => undef,
            'init_with_default' => 0,
            'set_interactive' => 0,
            'hidden' => 1,
            'persStep' => $STEP_UPGRADE_EXTRACT
        },
        ($isWin
            ? ('HostagentUserName' => $self->getParamHostagentUserName($order++, $section),
              )
            : ('InstallSSHKey'     => $self->getParamInstallSSHKey    ($order++, $section),
               'RootUser'          => $self->getParamRootUser         ($order++, $section),
               'RootPassword'      => $self->getParamRootPassword     ($order++, $section),
              )
        ),
        'SkipHostagentPw'   => $self->getParamSkipHostagentPw  ($order++, $section, undef, 0),
        'HostagentPassword' => $self->getParamHostagentPassword($order++, $section),
        'PrepareUpgrade' => {
           'order' => $order++,
           'opt' => $gPhasePrepareOptionAlias,
           'type' => 'boolean',
           'section' => $section,
           'value' => undef,
           'default' => 0,
           'str' => 'Stop update before software version switch, resumable',
           'desc' => 'Stop update before software version switch, resumable',
           'init_with_default' => 1,
           'hidden' => 0,
           'set_interactive' => 0
        },
        'Phase' => $self->getParamPhase ($order++, $section, [$gPhasePrepareOption, $gPhaseOfflineOption, $gPhaseConfigureOption, $gPhaseOnlineOption]),
        ($isWin
            ? ()
            : ('SkipModifySudoers' => $self->getParamSkipModifySudoers ($order++, $section))
        ),
    };

    $self->setSummaryOrder(['SID',
                            'SystemUser',
                            'ImportContent',
                            'InstallHostagent',
                            'Scope',
                            'RemoteExecution',
                            'RootUser',
                            'HostagentUserName',
                            'CheckOnly',
                            'NoStart',
                            'PrepareUpgrade',
                            'Phase',
                            'SkipModifySudoers',
                           ]);
	return $self;
}

sub checkEnableXSEngine{
    my ($self, $enable) = @_;
    $self->getMsgLst ()->addWarning ("WARNING: Ignoring deprecated command line switch '--xsengine'!");
    $self->getMsgLst ()->addWarning ("WARNING: It will be removed in future releases.");
    return 1;
}

sub getPersistenceFileModifiers {
	return (undef, undef, 0644); # uid, gid, mode
}

sub checkNoStart{
	my ($self, $value) = @_;
	if ($value =~ /$bool_true_pattern/i){
		$self->setValue('ImportContent', 'false');
	}
	return 1;
}

sub checkImportContent{
	my ($self, $value) = @_;

    my $phase = $self->getBatchValue('Phase');
    my $isOfflinePhase  = (defined $phase && ($phase eq $gPhaseOfflineOption));

    my $skip = $isOfflinePhase || ($value !~ /$bool_true_pattern/i);
    $self->setSkip ('SystemUser', $skip);
    $self->setSkip ('SQLSysPasswd', $skip);
    if (!$skip){
        return 1;
    }

    #
    # Upgrades from < HANA 2.0 need system user credentials
    # to perform Hana2PersistenceMigrationHelper.sql.
    # To be downward compatible, these parameters are NOT MANDATORY in this case and
    # can only be applied via command line option or confinuration file.
    #
    #
    # CAN BE REMOVED WHEN INPLACE MIGRTION IS AVAILABLE
    #
    #

    my $srcVersion = new SDB::Install::Version (split ('\.', $self->{_start_version}));
    my $version20 = new SDB::Install::Version (2,0,0,0);
    if($version20->isNewerThan ($srcVersion)){
        my $param;
        foreach my $paramKey ('SystemUser', 'SQLSysPasswd'){
            $param = $self->{params}->{$paramKey};
            $self->setSkip ($paramKey, 0);
            $param->{mandatory} = 0;
            $param->{set_interactive} = 0;
        }
    }
	return 1;
}

sub getProductName{
	return $gProductNameEngine;
}

sub getProductVersion {
    my ($self) = @_;
    if (!defined $self->{kit}){
        return $self->SUPER::getProductVersion ();
    }
    return $self->{kit}->GetVersion();
}

sub getShortProductName{
	return $gShortProductNameEngine;
}

sub getProperSIDs{
	my ($self, $withHosts) = @_;
	
	my $systems = CollectSAPSystems ();
	my $id = 0;
	my @tab;
	my $system;
	
	my @proper;
	my @nonproper;
	my $ignore_pending_upgrade = $self->getIgnore ('check_pending_upgrade');
	my $ignore_pending_installation = $self->getIgnore ('check_pending_installation');
	my $ignore_version = $self->getIgnore ('check_version');
	my $kit = $self->{kit};
	my $kitmanifest = $kit->getManifest ();
	foreach my $sid (sort keys (%$systems)){
		$system = $systems->{$sid};
		if (!$ignore_pending_installation){
	        my $instconfig = new SDB::Install::Configuration::NewDB ($self);
			$instconfig->{current_sid} = $sid;
			$instconfig->{sapSys}      = $system;
		    my $pendingInfo = getPendingInfoFromPersFile('Installation',
                                                         'of',
                                                         $instconfig,
                                                         \@STEP_NAMES,
                                                         "\t");
            if($pendingInfo) {
			    push @nonproper, [$sid, $pendingInfo];
			    next;
            }
		}
		if (!$system->hasNewDB()){
			if ($system->hasTrex ()){
				push @nonproper, [$sid, "Cannot upgrade Trex / old NewDB instance"];
			}
			else{
				push @nonproper, [$sid, "No $gProductNameEngine instance"];
			}
			next;
		}

		my $pendingInfo = $system->getRegisterOrRenamePending();
		if (defined $pendingInfo) {
			push @nonproper, [$sid, $pendingInfo];
			next;
		}

		my $version = $system->GetVersion();
		if (!defined $version){
			push @nonproper, [$sid, "Version is unknown"];
			next;
		}
		my $hostInfo = '';
		if ($withHosts) {
			my $hosts = $system->getNewDBInstances()->[0]->getAllHostsWithRoles(1);
			if (@$hosts) {
				$hostInfo = (scalar @$hosts == 1)
				            ? "\nhost: " . $hosts->[0]
				            : "\nhosts: " . join(', ', @$hosts);
			}
		}

	    my $updConfig  = new SDB::Install::Configuration::Upgrade();
		$updConfig->{current_sid} = $sid;
		$updConfig->{sapSys}      = $system;
		my $updPendingInfo = getPendingInfoFromPersFile('Update',
                                                     'to',
                                                     $updConfig,
                                                     \@UPGRADE_STEP_NAMES,
                                                     "\n");
		my $pluginInfo = '';
		my $pluginInstallations = $system->getNewDBInstances()->[0]->getPluginInstallations();
		my @pluginKeys;
		if (defined $pluginInstallations && @$pluginInstallations){
			foreach my $inst (@$pluginInstallations){
				push @pluginKeys, $inst->getProductKey();
			}
			if (@pluginKeys) {
				$pluginInfo = (scalar @pluginKeys == 1)
				              ? "\nplugin: " . $pluginKeys[0]
				              : "\nplugins: " . join(', ', @pluginKeys);
			}
		}
		my $sysManifest = $system->getManifest ();
		if (!defined $sysManifest){
			$self->getErrMsgLst()->appendMsgLst ($system->getErrMsgLst ());
		}
		if ($ignore_version || defined $sysManifest && $self->canUpgradeComponent ($system->getManifest, $kitmanifest)){
			$updPendingInfo = '' if (!defined $updPendingInfo);
			push @proper, [$sid, 'version: ' . $version . $hostInfo . $pluginInfo . $updPendingInfo];
		}
		else{
			my $errorString = $self->getErrorString ();

			my $pers = $self->pers_load();
			if (defined $pers){
				if (!$ignore_pending_upgrade && $pers->{_kitversion} ne $self->{_kitversion}){
					push @nonproper, [$sid, "Cannot resume pending upgrade: wrong kit version (requires $pers->{_kitversion})"];
				}
				else{
					push @proper, [$sid, $version . $hostInfo . $pluginInfo];
                }
            }
            else{
                push @nonproper, [$sid, $errorString];
            }
		}
	}
	return \@proper, \@nonproper;
}


sub checkSID{
	my ($self, $value) = @_;
	$self->resetError ();
	my $properties = $self->{params}->{SID};
		
	if ($value !~ /^[A-Z][A-Z0-9][A-Z0-9]$/){
		$self->AddError ("Invalid $properties->{str}");
		return 0;
	}

	my $systems = $self->getCollectSAPSystems();

	if (!exists $systems->{$value}){
		$self->AddError ("$properties->{str} \"$value\" " .
							"not found");
		return 0;
	}
	
	my $system = $systems->{$value};
	
	if (!$system->hasNewDB()){
			if ($system->hasTrex()){
				my $path = join ($path_separator,$system->{_usrSapSid},qw (SYS global trex install bin hdbuninst));
				if ($isWin){
					$path .= '.exe';
				}
				$self->AddError ("Cannot upgrade old NewDB instance (incompatible file system changes).");
				$self->PushError ("Please uninstall it using '$path' and reinstall it afterwards!");
			}
			else{
				$self->AddError ("$properties->{str} \"$value\" " .
					"has no $gProductNameEngine instance");
			}
			return 0;
	}
        if (!$self->checkMetaDataSeparation ($system)){
            return 0;
        }

	$self->{sapSys} = $system;
    $self->tryEnableScopeParams();
    $self->enableHostagentPasswordIfRequired();

	#
	# checking if there is a new installation, which isn't finished yet
	#
	$self->{current_sid} = $value;
	
	my $origin_class = ref ($self);
	bless ($self, 'SDB::Install::Configuration::NewDB');
	
	my $newinstallation_pending = $self->pers_exists ();
	bless ($self,$origin_class);
	if ($newinstallation_pending){
		$self->AddError ("The initial installation of $gProductNameEngine $value is pending. (" . $self->pers_date_string () . ")");
		if ($self->getIgnore ('check_pending_installation')){
			$self->AddMessage ("Ignoring error due to command line switch '--ignore=check_pending_installation'");
		}
		else{
			$self->PushError ("Please resume installation.");
			return 0;
		}
	}

	my $rc = $self->pers_exists ();
	if (!defined $rc){
		$self->AddError ("Runtime error");
		return undef;
	}
	if ($rc){
		if (!defined $self->validatePersistency){
			return undef;
		}
	}
	else{
		# checking for deprecated status file in /usr/sap
		my $pers = $self->validateOldLocalPersistencyFile ();
		if (defined $pers){
			if (!defined $self->validatePersistency ($pers)){
				return undef;
			}
		}
	}

	my $version = $system->GetVersion();
    my $targetManifest = $self->{kit}->getManifest ();
    my $srcManifest = $system->getManifest ();

	if (defined $version){
		if (!defined $srcManifest){
			$self->setErrorMessage ("Cannot upgrade $properties->{str} '$value'", $system->getErrMsgLst());
			return 0;
		}
		if (!$self->canUpgradeComponent ($srcManifest,$targetManifest)){
			if (defined $self->{options}->{force_downgrade} || $self->getIgnore ('check_version')){
                if ($self->isCalledBySHA()) {
                    $self->getErrMsgLst()->addError("Downgrade is not allowed during sidadm execution");
                    return 0;
                }
				$self->ResetError();
				$self->AddMessage ("Ignoring error due to command line switch '--ignore'");
			}
			else{
				return 0;
			}
		}
	}
	
	
	if (!defined $self->{_start_version}){
        $self->{_start_version} = $version;
	}

    #
    # checking for plugins and their HDB dependency
    #

	my $ignore_dependencies = $self->getIgnore ('check_plugin_dependencies');
	my $phase = $self->getBatchValue('Phase');

	if (($phase ne $gPhasePrepareOption) && !$system->checkPlugins ($self->{kit}, undef, $ignore_dependencies)){
		if (!$ignore_dependencies) {
			$self->AddError (undef, $system);
			return 0;
		} else {
			$self->AddMessage ('Ignoring failed dependency check for plugin due to option --ignore=check_plugin_dependencies', $system);
		}
	}

	my $user = new SDB::Install::NewDBUser ($value);

    if (!isAdmin ()){
        if (!$user->checkExistingUser ()){
            $self->AddMessage (undef, $user);
            $self->AddError ("Operating system user configuration has to be changed. Please start the upgrade as root user.");
            return 0;
        }
        if (!$isWin){
            my $shmdir = '/var/lib/hdb/' . $value;
            if (!-d $shmdir){
                $self->AddError ('The file system structure has to be changed. Please start the upgrade as root user.');
                return 0;
            }
            if (!-w _){
                $self->AddError ('The file system structure has to be changed. Please start the upgrade as root user.');
                return 0;
            }
        }
    }
	
	my $userName = $user->getSidAdmName ();
		
	if ($user->exists()){
	    $self->changePasswordStr($userName);
		$self->{user} = $user->getSidAdm();
	}
	else{
		$self->AddError ("Invalid SID : operating system user '$userName' not found");
		return 0;
	}

    if (defined $srcManifest){
        # if start rev < 90, force http communication for SAPControl
        if (new SDB::Install::Version ('1','00','90','00')->isNewerThan ($srcManifest->getVersionObject ())){
            $self->setValue ('SAPControlUseHttp', 1);
        }
    }

    if (!$isWin){
        my $isDistributed = $self->isDistributedSystem();
        if (defined $isDistributed && !$isDistributed){
            $self->setSkip('Password');
        }
    }

	return 1;
}

sub checkPassword{
    my ($self, $value) = @_;
    
    if (!defined $self->{user}){
        #$self->PushError ("Cannot check password: user is unknown");
        return 0;
    }
        
    my $rc = $self->{user}->verifyPassword ($value);
    
    if (!defined $rc){
        $self->AddWarning ("Cannot check password", $self->{user});
        $self->AddMessage ("=> pass check");
        return 1;
    }
    
    if (!$rc){
        $self->AddError ('Unknown user password combination');
        return 0;
    }

    return 1;   
}


#-------------------------------------------------------------------------------
# Sets the password.
# Calls CollectOtherHostInfos in case of scope 'system'
# and remote execution via SAPHostControl.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub setPassword {

    my ($self, $value) = @_;

    my $rc = $self->checkPassword($value);
    if (!$rc) {
        return $rc;
    }

    $self->{params}->{Password}->{value} = $value;

    if ($self->isUseSAPHostagent()) {
        if (!defined $self->CollectOtherHostInfos()) {
            $self->{params}->{Password}->{no_retry} = 1;
            return undef;
        }
    }

    return 1;
}

#-------------------------------------------------------------------------------
# Does not collect information of remote hosts because SAP HANA system has to be
# correct installed on remote hosts. Remote DBUpgradeHost (i.e. hdbreg) is
# called to update the operation system integration only.

sub CollectOtherHostInfos {
    my ($self) = @_;

    # provisional enabling of hostagent password
    my $remoteHosts = $self->getRemoteHostsObjects();
    if (defined $remoteHosts && @$remoteHosts
                                       && $self->getValue('InstallHostagent')) {
        $self->setSkip('HostagentUserName', 0);
        $self->setSkip('HostagentPassword', 0);
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Returns a file name.
# File contains the last step of an aborted update.

sub pers_filename{
    return $_[0]->getPersFilenameInstUpd ($statusFileExtension);
}

sub getOldLocalPersFilename{
    return $_[0]->getOldLocalPersFilenameInstUpd ($statusFileExtension);
}

sub pers_keys{
	my ($self) = @_;
	my @pers_keys = (@{$self->SUPER::pers_keys()},
                    '_keep_default_log_mode',
                    '_keep_default_table_type',
                    '_turn_off_ssl_client_pki',
		            '_start_version',
		            $gPhaseOptionPerskey);

	if ( defined $self->getValue( 'Phase' ) ) {
		push @pers_keys, 'next_phase';
		$self->{'next_phase'} = $self->getValue( 'Phase' );
	}

	return \@pers_keys;
}

sub pers_getstepname{
	return $UPGRADE_STEP_NAMES[defined $_[1] ? $_[1] : $_[0]->{step}];
}

sub pers_remove{
    my $self = shift;
    my $oldStatusFile = $self->getOldLocalPersFilenameInstUpd ($statusFileExtension);
    if (defined $oldStatusFile){
        $self->SUPER::pers_remove ($oldStatusFile);
    }
    return $self->SUPER::pers_remove (@_);
}

sub setKeepDefaultLogMode{
    my ($self,$mode) = @_;
    $self->{_keep_default_log_mode} = $mode;
}


sub getKeepDefaultLogMode{
    return $_[0]->{_keep_default_log_mode};
}

sub setKeepDefaultTableType{
    my ($self,$type) = @_;
    $self->{_keep_default_table_type} = $type;
}


sub getKeepDefaultTableType{
    return $_[0]->{_keep_default_table_type};
}

sub setTurnOffSslClientPki{
    $_[0]->{_turn_off_ssl_client_pki} = 1;
}

sub getTurnOffSslClientPki{
    return $_[0]->{_turn_off_ssl_client_pki};
}

sub getStartVersion{
	$_[0]->{'_start_version'};
}

sub getPhase {
    my ($self) = @_;
    return $self->getValue('Phase');
}

sub isPrepareUpdate {
    my ($self) = @_;
    return $self->getValue('PrepareUpgrade') || $self->isPhase($gPhasePrepareOption);
}

sub validatePersistency{
	my ($self, $pers) = @_;
	if (!defined $pers){
		$pers = $self->pers_load ();
	}
	if (!defined $pers){
		return undef;
	}
	
	my $ignore_pending_upgrade = $self->getIgnore ('check_pending_upgrade');
    my $preparedPhase = $pers->{$gPhaseOptionPerskey};
    my $preparedOrBroken = (defined $preparedPhase) ? "prepared" : "broken";
	
	if ($pers->{_kitversion} ne $self->{_kitversion}){
		$self->AddError ("Cannot resume $preparedOrBroken upgrade of '$self->{current_sid}': wrong version ($self->{_kitversion})");
		$self->PushError ("$gProductNameEngine installation kit '$pers->{_kitversion}' required");
		
		if ($ignore_pending_upgrade || $self->getIgnore ('check_version')){
			$self->ResetError ();
			$self->AddMessage ("Ignoring error due to command line switch '--ignore'");
		}
		else{
			return undef;
		}
	}

	if ($ignore_pending_upgrade){
		$self->AddMessage ("Ignoring pending upgrade due to --ignore command line switch and starting a new upgrade");
		return 1;
	}

	if (!defined $self->SUPER::validatePersistency ($pers)){
		return undef;
	}


    my $pendingMsg = "Resuming broken upgrade at step '"
                   . $self->pers_getstepname($pers->{step}) . "'.";

    if(defined $preparedPhase) {
        my $prettyPrepareTimestamp = $self->pers_date_string();
        if($preparedPhase eq $gPhasePrepareOption) {
            $pendingMsg = "Resuming upgrade with software version switch(prepared at $prettyPrepareTimestamp).";
        }
        elsif($preparedPhase eq $gPhaseOfflineOption) {
            $pendingMsg = "Resuming upgrade with system configuration (prepared, offline since $prettyPrepareTimestamp)."
        }
        elsif($preparedPhase eq $gPhaseConfigureOption) {
            $pendingMsg = "Resuming upgrade with restart (prepared at $prettyPrepareTimestamp).";
        }
    }
    $self->setResume();
    if (defined $self->getBatchValue('Phase')) {
        $self->getMsgLst()->addMessage($pendingMsg);
    } else {
        $self->AddProgressMessage ($pendingMsg);
    }

	return 1;
}

sub getIgnoreValues{
    my ($self) = @_;
    my @ignoreVals = qw (
        check_busy_files
        check_diskspace
        check_hosts
        check_license
        check_min_mem
        check_pending_installation
        check_pending_upgrade
        check_secondary_system
        check_plugin_dependencies
        check_version
    );
    return \@ignoreVals;
}

sub getHiddenIgnoreValues{
	return [qw (check_hardware check_platform)];
}

sub isUpdate {
	return 1;
}

sub isInstallationSelected{
    return $_[0]->isValuePreset ('SID');
}


#-------------------------------------------------------------------------------
# In case of distributed system and if parameter 'root_user' is specified,
# this method calls DBUpgradeHost for each remote host to change modified
# configuration objects.
#
# Returns undef in case of an error

sub tryUpgradeRemoteHosts {

    my ($self) = @_;

    if (!defined $self->getRemoteHosts()) {
        return 1;
    }
    my $isSAPControlUseHttpRequired = !$isWin && !$self->isUseHttps ();
    return $self->excuteRemoteParallel('Updating',
                                       'updated',
                                       'hdbreg',           # executable
                                       'DBUpgradeHost',    # main program
                                       $gOperationDBUpgradeHost, # hostctrl operation
                                       ['HostagentPassword', 'Password'],
                                       ['InstallHostagent',
                                        'SkipModifySudoers',
                                        $isSAPControlUseHttpRequired ?
                                        'SAPControlUseHttp' :
                                        ()]);
}


#-------------------------------------------------------------------------------
# Allow <sid>adm upgrade

sub isAdminRequired{
    return 0;
}


#-------------------------------------------------------------------------------
# Enables the multiple host parameters.
# sidadm user is used instead of hostagent user in case of SAPHostControl.
#
# A comment at the beginning of AnyConfig shows depending parameters.

sub enableMultipleHostParameters {
    return $_[0]->SUPER::enableMultipleHostParameters(1); # 1 - $useSidadmForHostctrl
}


#-------------------------------------------------------------------------------
#
# import system check from new installation
#

sub checkSystemRequirements;
*checkSystemRequirements = \&SDB::Install::Configuration::NewDB::checkSystemRequirements;

sub checkMinMem;
*checkMinMem = \&SDB::Install::Configuration::NewDB::checkMinMem;


sub shouldWarnIfCalledStandalone{
    my ($self) = @_;
    return $self->_calledStandaloneCriterion();
}

sub getExeName{
    return "hdbupd";
}

sub getResidentHdblcmPath{
    my ($self) = @_;
    return $self->_constructResidentHdblcmPath();
}

1;
