#
# $Header$
# $DateTime$
# $Change$
#
# Desc: generic utilities for debugging and development testing.


package SDB::Install::DebugUtilities;

use Exporter;


#----------------------------------------------------------------------------

our @ISA = qw (Exporter);
our @EXPORT = qw (dumpThings);

#----------------------------------------------------------------------------

=functionComment

Prints the contents of the data structure given by 'p_what'.
This may be a scalar or reference a hash or an array.
If 'p_depth' is set to a nonzero value, the structure is traversed 
and printed recursively down to that depth.
(note that dumpThings will fall into an infinite loop
 when it encounters a cycle in the structure)
'p_indentStep' controls indentation depth.
The output format is a syntactically correct perl literal.

=cut
sub dumpThings {
    my (
        $p_what,
        $p_depth,
        $p_indentStep,
        $p_outputFileHandle
    ) = @_;
    my $fh = *STDOUT;
    if(defined $p_outputFileHandle) {
        $fh = $p_outputFileHandle;
    }
    dumpThings_aux($p_what, $p_depth, 0, $p_indentStep, $fh, $p_depth);
}

#----------------------------------------------------------------------------

sub dumpThings_aux {
    my (
        $p_what,
        $p_currentDepth,
        $p_currentIndent,
        $p_indentStep,
        $p_fh,
        $p_maxDepth
    ) = @_;
    my $whatRef = ref($p_what);
    if($p_currentDepth && ($whatRef !~ /^SCALAR/) && ($p_what =~ /(^.*=|^)ARRAY/)) {
        print $p_fh "\n".(" " x ($p_currentIndent+2*$p_indentStep))."[ # <$whatRef> \n";
        foreach my $value (@$p_what) {
                print $p_fh " " x ($p_currentIndent+3*$p_indentStep);
                dumpThings_aux($value, $p_currentDepth-1, $p_currentIndent+$p_indentStep, $p_indentStep, $p_fh, $p_maxDepth);
                print $p_fh ", \n";
        }
        print $p_fh " " x ($p_currentIndent+2*$p_indentStep)."]";
    }
    elsif($p_currentDepth && ($whatRef !~ /^SCALAR/) && ($p_what =~ /(^.*=|^)HASH/)) { 
        print $p_fh  "\n".(" " x ($p_currentIndent+2*$p_indentStep))."{ # <$whatRef> \n";
        #sort this:
        while (($key,$value) = each %$p_what) {
                print $p_fh " " x ($p_currentIndent+3*$p_indentStep)."'".$key."'=>";
                dumpThings_aux($value, $p_currentDepth-1, $p_currentIndent+2*$p_indentStep, $p_indentStep, $p_fh, $p_maxDepth);
                print $p_fh ", \n";
        }
        print $p_fh " " x ($p_currentIndent+2*$p_indentStep)."}";
    }
    else {
        if($p_currentDepth == $p_maxDepth) {
            print $p_fh "\n";
        }
        if(!defined $p_what) {
            print $p_fh 'undef';
        }
        else {
            print $p_fh "'".$p_what."'";
        }
    }
}


#----------------------------------------------------------------------------

=functionComment

Returns a reference of a structure resembling the hash from
'filesInUseByPID' in "Package.pm", i.e.
(
        100000  =>  {   'commandLine' => '/usr/bin/program01',
                        'lockedFiles' => ["/aaa/zzz.htm", "bbb.pbx"]
                    },
        1000001 =>  {   'commandLine' => '/usr/bin/program02',
                        'lockedFiles' => ["/zzz.tgz", "ccc.jar"]
                    },
        50000   =>  {   'commandLine' => '/usr/bin/program03',
                        'lockedFiles' => ["/vvv/aaa/zzz.exe", "bbb.pdb"]
                    },
        20000   =>  {   'commandLine' => '/usr/bin/program04',
                        'lockedFiles' => ["/sss/ddd/fff/aaa/zzz.txt", "bbb.html"]
                    },
        30000   =>  {   'commandLine' => '/usr/bin/program05',
                        'lockedFiles' => ["/aaa.dll", "bbb.jpg"]
                    },
        40000   =>  {   'commandLine' => '/usr/bin/program06',
                        'lockedFiles' => ["/aaa/zzz.exe", "bbb.dll"]
                    }
)
The content of the structure is made up randomly, the RNG is seeded 
with parameter "$p_seed".

=cut
sub fakeFilesInUseByPID {
    my (
        $p_seed
    ) = @_;
    my $retval = {};
    my $minNoOfPIDs = 1;
    my $maxNoOfPIDs = 100;
    my $fileSelections = ["/aaa/zzz.exe", "bbb.dll", "/aaa.dll", "bbb.jpg", "/sss/ddd/fff/aaa/zzz.txt", "bbb.html",
                      "/vvv/aaa/zzz.exe", "bbb.pdb", "zzz.tgz", "ccc.jar", "/aaa/zzz.htm", "bbb.pbx"];
    my $pidSelections = [1234, 5678, 1111, 2222, 6666, 9999,
                      6566, 3676, 6787, 2343, 4556, 3454];
    my $minNoOfFiles = 0;
    my $maxNoOfFiles = 30;
    # nothing to configure below this line
    my $noOfFileSelections = scalar @$fileSelections;
    my $noOfPidSelections = scalar @$pidSelections;
    my $sep = ($^O =~ /mswin/i) ? '\\' : '/';
    my $select;
    my $i;
    my $pid;
    my $j;
    my $file;
    if($p_seed) {
	srand($p_seed);
    }
    my $noOfPIDs = $minNoOfPIDs + int(rand(($maxNoOfPIDs - $minNoOfPIDs +1)*1000)/1000);
    for($i = 0; $i < $noOfPIDs; $i++) {
        $select = int(rand($noOfPIDs*1000)/1000);
        $pid = $pidSelections->[$select];
        my $noOfFiles = $minNoOfFiles + int(rand(($maxNoOfFiles - $minNoOfFiles +1)*1000)/1000);
        $retval->{$pid}->{'commandLine'} = 'cmdl'.$pid;
        $retval->{$pid}->{'lockedFiles'} = [];
        for($j = 0; $j < $noOfFiles; $j++) {
            $select = int(rand($noOfFileSelections*1000)/1000);
            $file = $fileSelections->[$select];
            push( @{$retval->{$pid}->{'lockedFiles'}}, ($file));
        }
    }
    return $retval;
}

#----------------------------------------------------------------------------

1;

