package SDB::Install::DeliveryUnit;

use SDB::Install::SysVars;
use SDB::Install::System qw (exec_program isAdmin);
use SDB::Install::Tools qw (trim rtrim ltrim);
use SDB::Install::Archive;
use SDB::Install::Globals qw($gLogDir);

sub trim ($);
sub ltrim ($);
sub rtrim ($);

our @ISA = qw (SDB::Install::Archive);

sub init{
	my ($self) = @_;
	my $metadata;

	while (my $tar_header = $self->next()){
		if (!exists $tar_header->{name}){
			# eof
			last;
		}
		if ($tar_header->{name} eq 'manifest.txt'){
			$metadata = $self->get_file_content ();
			if (!defined $metadata){
				$self->AddError ("No meta data found");
				return undef;
			}
			last;
		}
	}
	if (!defined $metadata){
		$self->AddError ("No meta data found");
		return undef;
	}
	my ($key, $value);
	my %meta;
	foreach my $line (split ("\n", $metadata)){
		trim (\$line);
		if ($line eq ''){
			# empty line
			next;
		}
		if ($line =~ /^#/){
			# comment line
			next;
		}
		($key, $value) = ($line =~ /([^=]*)=(.*)/);
		if ($key){
			rtrim (\$key);
			ltrim (\$value);
			if ($key){
				$meta{$key} = $value;
			}
		}
	}
	$self->{metadata} = \%meta;
	return 1;
}

sub getArchiveName{
	my ($self) = @_;
	return $self->{archive}
}

sub getName{
	my ($self) = @_;
	return $self->{metadata}->{delivery_unit};
}

sub getVersion{
	my ($self) = @_;
	return $self->{metadata}->{delivery_unit_version};
}

sub getSP{
	my ($self) = @_;
	return $self->{metadata}->{version_sp};
}

sub getPatch{
	my ($self) = @_;
	return $self->{metadata}->{version_patch};
}


our @sslProperties = qw (sslCryptoProvider sslKeyStore sslTrustStore sslValidateCertificate);


sub importContent{
	my ($self, $instance, $sql_user, $sql_passwd, $user_store_key, $port, $secureConnectProperties) = @_;

    my ($uid, $gid) = (undef, undef);
    if(isAdmin()){
        $uid = $instance->get_user()->uid();
        $gid = $instance->get_user()->gid();
    }

	my $cfg = {
		'stdin_buf' => [$sql_passwd],
		'UID'       => $uid,
		'GID'       => $gid,
	};

    my $exeDir = $instance->get_instanceExeDir ();
	my $regi = join($path_separator, $exeDir, 'regi');
	my $rc;

	my ($sslConnectPropertiesHash, $sslEnv);
	if (defined $secureConnectProperties){
		($sslConnectPropertiesHash, $sslEnv) = @$secureConnectProperties;
	}

	local %ENV = %ENV if defined $sslEnv || $isLinux;

	if (defined $sslEnv){
		foreach my $varName (keys %$sslEnv){
			$ENV{$varName} = $sslEnv->{$varName};
		}
	}

	$instance->_getHDBEnv (\%ENV);

    # try to remove regi log from previous execution:
    chdir($gLogDir);
    my $regiLogfile = $gLogDir.$path_separator."commands.log";
    if(-f $regiLogfile && not -w $regiLogfile) {
        $self->AddMessage("Trying to unlink write-protected '$regiLogfile' from previous regi execution.");
        if (!unlink($regiLogfile)) {
            $self->AddWarning("Could not unlink '$regiLogfile': $!");
        }
    }

	my @sslArgs;
	if (defined $sslConnectPropertiesHash){
		push @sslArgs, '--sslEncrypt=1',
		my $sslPropertyValue;
		foreach my $sslProperty (@sslProperties){
			$sslPropertyValue = $sslConnectPropertiesHash->{$sslProperty};
			if (defined $sslPropertyValue){
				push @sslArgs, "--$sslProperty=$sslPropertyValue";
			}
		}
	}
    
    if ($user_store_key) {
		$rc = exec_program ($regi,
			['--verbose',
			'--workspace=empty',
			'--key=' . $user_store_key,
			@sslArgs,
			'import',
			$self->{archive}],
			$cfg);
	} else {
		my @databaseArgs;
		if($instance->getSQLDatabaseName() eq 'SYSTEMDB') {
			push @databaseArgs, "--database=SYSTEMDB";
		}
		$rc = exec_program ($regi,
			['--verbose',
			'--workspace=empty',
			'--host=' . $instance->getSqlHost () . ':3' . $instance->get_nr() . $port,
			'--user=' . $sql_user,
			'--readFromStdIn',
			@databaseArgs,
			@sslArgs,
			'import',
			$self->{archive}],
			$cfg);
	}
	$self->AddMessage (undef,$cfg);
	if (!defined $rc){
		$self->AddError ("Error importing delivery units", $cfg);
		return undef;
	}
	if ($rc != 0){
		$self->AddError ("Error importing delivery units: regi returned exit code '$rc'");
		return undef;
	}
	return 1;
}

1;
