#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: base class of NewDB installation


package SDB::Install::Installation;
use SDB::Install::DebugUtilities;

use strict;

use SDB::Install::Log;
use Exporter;
use SDB::Install::Tools qw (arrayToHash readini);
use SDB::Install::System;
use SDB::Install::PackageManager;
use SDB::Install::SysVars;
use SAPDB::Install::Hostname;
use SDB::Install::Globals qw ($gProductName $gChecksumDirname);
use SDB::Install::Manifest;
use SAPDB::Install::MD5Sum;
require File::stat;
use SDB::Common::BuiltIn;

our $persistentInitialInstallerOptionsFile = 'installation.ini';
our $persistenceSectionLabel = 'installation_ini';

our $systemAppDataPath = getAllUsersAppDataPath();
our $installRegistryFile = 'INSTREG';


our @ISA = qw (SDB::Install::PackageManager Exporter);

our @EXPORT = qw (SDB_O_CREAT SDB_O_RDWR SDB_O_RDONLY);

sub SDB_O_RDONLY () {0}
sub SDB_O_RDWR () {2}
sub SDB_O_CREAT () {0100}


#----------------------------------------------------------------------------
# Creates an instance of Installation.
#
# If $instPath is undefined, the installation path can be set later
# via 'setInstallationPath'.
#
# Parameter: $instPath   string
#            $flags bit  map  (flags: SDB_O_RDONLY / SDB_O_RDWR / SDB_O_CREAT)

sub new{
	my $self = shift->SUPER::new ();

	# properties	
	my ($instPath, $flags) = @_;

    if (!defined $flags){
        $flags = 0;
    }

    $self->{create_new} = 0;
    $self->{datapath}   = undef;
    $self->{exclusive}  = 0;
    $self->{instance}   = undef;
    $self->{key}        = undef;
    $self->{log}        = undef;
    $self->{name}       = undef; # => sid
    $self->{path}       = undef;
    $self->{registry}   = undef;

    if (defined $instPath) {
       $self->initInstallationWithInstPath($instPath, $flags);
    }

	return $self;
}

sub EnumInstallations{
    my ($self, $nocache) = @_;
    if ($self->{_installations} && !$nocache){
        return $self->{_installations};
    }
    my $regFile = $self->getRegFile();

    if (!defined $regFile){
        return {};
    }

    if (!-e $regFile){
        return {};
    }
    if (!open (RG, $regFile)){
        $self->AddError ("Cannot open registry: $!");
        return undef;
    }
    my @buffer = <RG>;
    close (RG);
    my $msg = $self->AddMessage ('Looking for ' . $self->getProductName() . ' installations...');
    $self->SetFormatInfo ($msg, 'h1', 'Looking for ' . $self->getProductName() . ' installations');
    my $msglst = $msg->getSubMsgLst ();
    my ($path, $version);
    $self->{_installations} = {};
    foreach my $i (0 .. $#buffer){
        if ($buffer[$i] =~ /^\s*#/){
            next;
        }
        ($path, $version) = ($buffer[$i] =~ /(.*)=(.*)/);

        if (!defined $path){
            next;
        }
        $path =~ s/^\s*//;
        $path =~ s/\s*$//;
        $version =~ s/\s*$//;
        $version =~ s/^\s*//;
        $self->{_installations}->{$path} = $version;
        $msglst->addMessage ("$version   $path");
    }
    return $self->{_installations};
}

sub isUserMode{
    return 0;
}

#
# regDir is the directory where regFile is located
# regFile is a ini file which contains a list of all installed installations
# and thier versions
#

sub getSysModeRegDir{
    return join ($path_separator, $systemAppDataPath, '.hdb', lc (hostname ()));
}


sub getUserModeRegDir{
    my $homeDir = getHomeDir ();
    return join ($path_separator, $homeDir,
    '.hdb', lc (hostname ()));
}

sub getRegDir{
    my ($self) = @_;
    if ($self->isUserMode ()){
        return $self->getUserModeRegDir ();
    }
    return $self->getSysModeRegDir ();
}

sub getRegFileName{
    return undef;
}

sub getRegFile{
    my $regFileName = $_[0]->getRegFileName();
    if (!defined $regFileName){
        return undef;
    }
    return $_[0]->getRegDir() . $path_separator . $regFileName;
}


#----------------------------------------------------------------------------
# Sets the installation path, handles the flags and creates specific
# installation classes in case of packages.
#
# Parameter: $instPath   string
#            $flags bit  map  (flags: SDB_O_RDONLY / SDB_O_RDWR / SDB_O_CREAT)

sub initInstallationWithInstPath {

    my ($self, $instPath, $flags) = @_;

    if (defined $instPath) {
        $self->{path} = ($isWin) ? getRealPathName ($instPath) : $instPath;
    }

    if (defined $flags) {
        $self->{create_new} = $flags & SDB_O_CREAT ? 1 : 0;
        $self->{exclusive}  = $flags & SDB_O_RDWR  ? 1 : 0;
    }

    if (ref ($self) eq __PACKAGE__ && !$self->{create_new}){
        my $mf = $self->getManifest ();
        if (defined $mf && $mf->exists()){
            if ($mf->isClient()){
                require SDB::Install::Installation::Client;
                bless ($self, 'SDB::Install::Installation::Client');
            }
            elsif ($mf->isStudio()){
                require SDB::Install::Installation::DatabaseStudio;
                bless ($self, 'SDB::Install::Installation::DatabaseStudio');
            }
            elsif ($mf->isOfficeClient()){
                require SDB::Install::Installation::OfficeClient;
                bless ($self, 'SDB::Install::Installation::OfficeClient');
            }
            elsif ($mf->isAFL() || $mf->isLCAPPS() || $mf->isServerPlugin()){
                require SDB::Install::Installation::GenericServerPlugin;
                bless ($self, 'SDB::Install::Installation::GenericServerPlugin');
            }
            else{
                require SDB::Install::Installation::Generic;
                bless ($self, 'SDB::Install::Installation::Generic');
            }
            $self->init ();
        }
    }
}

#----------------------------------------------------------------------------

sub GetVersion{
	my ($self) = @_;
	if (defined $self->{kit_version}){
		return $self->{kit_version};
	}
	return $self->getVersionByManifest();
}

#----------------------------------------------------------------------------

sub getVersionByManifest{
	my ($self, $nocache) = @_;
	
	my $manifest = $self->getManifest ($nocache);
	
	if (defined $manifest){
		my $version = $manifest->getVersion ();
		if (!defined $version){
			$self->AddError ('Cannot get version', $manifest);
			return undef;
		}
		return $version;
	}
	
	return undef;
}

#----------------------------------------------------------------------------

sub getManifestDir{
    if ($isApple){
        foreach my $path ($_[0]->{path}, $_[0]->{path} . '/Contents/Eclipse'){
            if (-f "$path/manifest"){
                return $path;
            }
        }
    }
    return $_[0]->{path};
}

#----------------------------------------------------------------------------

sub setConfiguration{
    my (
        $self,
        $value
    ) = @_;
    my $oldvalue = $self->{'configuration'};
    $self->{'configuration'} = $value;
    return $oldvalue;
}

#----------------------------------------------------------------------------

sub getConfiguration{
    my ($self) = @_;
    return $self->{'configuration'};
}

#----------------------------------------------------------------------------



sub asString{
	my ($self) = @_;
	return $self->getProductName() .($self->{path} ? " in '$self->{path}'" : '');
}

sub OpenExclusive{
	my ($self) = @_;
	if (defined $self->{registry}){
		if ($self->{exclusive}){
			return 1;
		}
		else{
			$self->CloseRegistry ();
		}
	}
	$self->{exclusive} = 1;
	if (defined $self->{log}){
		$self->{log}->Suppress (0);
	}	

	return $self->OpenRegistry ();
}

#----------------------------------------------------------------------------

sub RegistryFileExists{
	my ($self) = @_;
	if (defined $self->{registry}){
		return 1;
	}
	if (defined  $self->{registrypath}){
		if (-f "$self->{registrypath}/$installRegistryFile"){
			return 1;
		}	
	}
	return 0;
}

#----------------------------------------------------------------------------

sub getNoLock{
	return undef;
}

#----------------------------------------------------------------------------

sub OpenRegistry{
	my ($self) = @_;
	if (defined $self->{registry}){
		return 1;
	}

	if (!defined  $self->{registrypath}){
		$self->AddError ("Registry path is not defined");
		return undef;
	}

	require SDB::Install::Registry;

	if (! -d $self->{registrypath}){
		if ($self->{create_new}){
			$self->AddError ("Cannot create install registry file: directory '$self->{registrypath}' doesn\'t exist");
		}
		else{
			$self->AddError ("Install registry file not found: directory '$self->{registrypath}' doesn\'t exist");
		}
		return undef;
	}

	my $msg = $self->AddMessage ('Opening install registry');
	$self->{registry} = new SDB::Install::Registry ($self->{registrypath},
         $self->{exclusive},
         $self->getNoLock (),
         $self->getUID (),
         $self->getGID ());

	if ($self->{registry}->ErrorState ()){
		$self->AddError ('Cannot open install registry', $self->{registry});
		undef $self->{registry};
		return undef;
	}

	$self->AddSubMsgLst ($msg, $self->{registry});

	return 1;
}

#----------------------------------------------------------------------------

sub init{
    my ($self) = @_;
    


    return 1 if $self->{initialized};

    $self->ResetError();

    $self->{initialized} = 1;
    return 1;
}

#----------------------------------------------------------------------------

sub getPersistentInitialInstallerOptionsFilePath{
    my ($self) = @_;
    if (!defined $self->{'registrypath'}){
        return undef;
    }
    return $self->{'registrypath'} . $path_separator .
        $persistentInitialInstallerOptionsFile;
}

#----------------------------------------------------------------------------

sub loadPersistentInstallationParams{
    my ($self) = @_;
    
    my $registryPath = $self->{'registrypath'};
    my $persistenceFullPath = $self->getPersistentInitialInstallerOptionsFilePath ();
    my $instconfig = $self->{'configuration'};
    if(-f $persistenceFullPath) {
        my $persistentInitialInstallerOptions = new SDB::Install::IniFile(
            $persistenceFullPath
        );
        if(!defined $persistentInitialInstallerOptions) {
        	return undef;
        }
        my $section = $persistentInitialInstallerOptions->getSection($persistenceSectionLabel);
        if(defined $section) {
        	my $fixedSectionKey; 
	        foreach my $sectionKey (keys %{$section}){
	        	my $value = $persistentInitialInstallerOptions->getValue($persistenceSectionLabel, $sectionKey);
	        	$fixedSectionKey = $sectionKey == "Hostname" ? "HostName" : $sectionKey; 
	        	if(exists $instconfig->{'params'}->{$fixedSectionKey} && not $instconfig->{'params'}->{$fixedSectionKey}->{'DEPRECATED'}) {
	        		if(not defined $instconfig->{'params'}->{$fixedSectionKey}->{'value'}) {
	        		    if(!$instconfig->setValue($fixedSectionKey, $value)) {
	        		        my $msg = $self->AddError("Value \"$value\" for key \"$fixedSectionKey\" is not permitted.");
	        		        $self->AddSubMsgLst ($msg, $instconfig);
	        		        return undef;
	        		    }
                        my $paramDesc = $instconfig->{'params'}->{$fixedSectionKey}->{'str'};
                        my $paramValueDesc = $instconfig->getUIStringFromValue($fixedSectionKey, $value);
	        	        my $msgTxt = "$paramDesc: $paramValueDesc";
	        	        $self->AddProgressMessage($msgTxt);
	        		}
	        		else {
	        			# preset param from instconfig (set by cmdl, config file, etc)
	        			# should override param from persistence
	        			# so do nothing here.
	        			# note that this implies also that a default will override
	        			# the setting from persistence (yes, that's not nice), so
	        			# do not set defaults in a SDB::Install::Configuration
	        			# object for persistent parameters.
	        		}
	        	    
	        	}
	        }
        }
    }
    return 1;
}

#----------------------------------------------------------------------------

sub writePersistentInstallationParams{
    my ($self) = @_;
    my $registryPath = $self->{'registrypath'};
    my $instregPath = $registryPath.$path_separator.$installRegistryFile;
    my $persistenceFullPath = $self->getPersistentInitialInstallerOptionsFilePath ();
    my $instconfig = $self->{'configuration'};
    my $persistentParams = {};
    foreach my $paramKey (keys %{$instconfig->{'params'}}){
        my $param = $instconfig->{'params'}->{$paramKey};
        if(!$param->{'persist_for_upgrade'}) {
            next;
        }
        else {
        	if(defined $param->{'value'}) {
                $persistentParams->{$paramKey} = $param;
        	}
        }
    }
    if(-f $persistenceFullPath && !unlink $persistenceFullPath) {
        $self->AddError("Could not remove   \"$persistenceFullPath\".");
        return undef;
    }
    my $persistentParamCount = keys %{$persistentParams};
    if($persistentParamCount > 0) {
        my $uid = undef;
        my $gid = undef;
        if (!$isWin){
            $uid = (stat ($instregPath))[4];
            $gid = (stat (_))[5];
        }
        my $persistentInitialInstallerOptions = new SDB::Install::IniFile(
	            $persistenceFullPath,
	            $uid,
	            $gid
	    );
        if(!defined $persistentInitialInstallerOptions) {
            return undef;
        }
	    foreach my $persistentParamKey (keys %{$persistentParams}){
            my $persistentParam = $persistentParams->{$persistentParamKey};
            my $persistentParamValue = $persistentParam->{'value'};
            if($persistentParam->{'DEPRECATED'} || not defined $persistentParamValue) {
                $persistentParamCount--;
            }
            else {
            	$persistentParamKey = $persistentParamKey eq 'HostName' ? 'Hostname' : $persistentParamKey;
              	$persistentInitialInstallerOptions->setValue($persistenceSectionLabel, $persistentParamKey, $persistentParamValue);
            }
	    }
	    if($persistentParamCount > 0) {
            my $rc = $persistentInitialInstallerOptions->write();
            if(!$rc) {
                $self->AddError("Could not write \"$persistenceFullPath\"", $persistentInitialInstallerOptions);
                return undef;
            }
	    }
    }
    if($persistentParamCount == 0) {
        if(-f $persistenceFullPath && !unlink $persistenceFullPath) {
            $self->AddError("Could not remove \"$persistenceFullPath\".");
            return undef;
        }
    }
    return 1;
}

#------------------------------------------------------------------------------
# Changes the owner/group ID of registry files if an old ID matches

sub changePackageOwnerInRegistry{
	my ($self,$oldUid, $uid, $oldGid, $gid) = @_;
	
	if ($isWin){
		return 1;
	}
	
	if (!$self->{exclusive}){
		if (!defined $self->OpenExclusive ()){
			return undef;
		}
	}
	if (!defined $self->GenPackageList()){
		return undef;
	}
	my $rc = 1;
	foreach my $package (values %{$self->{packages}}){
		if (!defined $package->changeFileOwnerInRegistry ($oldUid, $uid, $oldGid, $gid)){
			$self->PushError ("Cannot change owner of package " . $package->GetName (), $package);
			$rc = undef;
		}
	}
	return $rc;
}


#
sub GetProgramPath ($){
	getRealPathName (shift->{path});
}

#----------------------------------------------------------------------------

sub GetInstallationPath($);
*GetInstallationPath = \&GetProgramPath;

#----------------------------------------------------------------------------

sub getProgramPath($);
*getProgramPath = \&GetProgramPath;

#----------------------------------------------------------------------------

sub getTargetLstDir {
    my ($self) = @_;
    return join ($path_separator, $self->getProgramPath(), $gChecksumDirname);
}

#----------------------------------------------------------------------------

sub GetPackageIds{
	my ($self) = @_;
	return $self->{registry}->GetPackageIds ();  
}

#----------------------------------------------------------------------------

sub GenPackageList{
	my ($self, $msglst) = @_;
	
	if (defined $self->{packages}){
		return $self->{packages};
	}
	
	if (!defined $msglst){
		$msglst = $self;
	}
	
	my %packages;
	unless (defined $self->{registry}){
		unless (defined $self->OpenRegistry ()){
			$msglst->AddError ("Cannot initialize install registry",$self);
			return undef;
		}
	}
	
	foreach my $pack_id ($self->{registry}->GetPackageIds ()){
		$packages{$pack_id} = $self->{registry}->GetPackage ($pack_id);
		unless (defined $packages{$pack_id}){
			$msglst->AddError ('Cannot get package $pack_id',$self->{registry});
			return undef;
		}
		$packages{$pack_id}->{installation} = $self;
	}
	$self->{packages} = \%packages;
	return $self->{packages};
}

#----------------------------------------------------------------------------

sub AddPackage{
	my ($self,$package) = @_;

	unless (exists $self->{packages}->{$package->{id}}){
		$self->{packages}->{$package->{id}} = $package;
	}
	
	if (defined $package->{tree_node}){
		return 1;
	}
	
	$package->CheckDependenciesByInstallation ($self);
		
	if (!defined $package->{dependencies} || !%{$package->{dependencies}}){
		#package has no depenedencies
		unless (defined $self->{tree}){
			$self->{tree} = {};
		}
		$package->{tree_node} = {};
		$self->{tree}->{$package->{id}} = $package->{tree_node};
		return 1;	
	}
	
	my $rc = 1;
	
	my @err_list;
	
	foreach my $dependency (keys (%{$package->{dependencies}})){
			my $resolved = 0;
			my $parent_node;
			if ($package->{dependencies}->{$dependency}->{resolved_by_installation}){
				unless (defined $self->{tree}){
					$self->{tree} = {};
				}
				if (defined $self->{packages}->{$dependency}->{tree_node}){
					$parent_node = $self->{packages}->{$dependency}->{tree_node};	
				}
				elsif (defined $self->AddPackage($self->{packages}->{$dependency})){
					$parent_node = $self->{packages}->{$dependency}->{tree_node};
				}
				$resolved = 1;
			}
			
			if ($resolved){
				if (exists $package->{tree_node}){
					$parent_node->{$package->{id}} = $package->{tree_node};
				}
				else{
					$parent_node->{$package->{id}} = {};
					$package->{tree_node} = $parent_node->{$package->{id}};
				}	
			}
			else{
				push @err_list, $self->AddError ('unresolved dependency: '.$package->{dependencies}->{$dependency}->{str});
				$rc = undef;
			}
	}


	if (!defined $package->{tree_node}){
		$package->{tree_node} = {};
		$self->{tree}->{$package->{id}} = $package->{tree_node};
	}


	if (@err_list){
		$self->{last_error} = \@err_list;
	}

	return $rc;
}



#----------------------------------------------------------------------------

sub checkDependenciesAfterUpgrade{
	my ($self, $kit, $node, $visited) = @_;
	
	if (!defined $node){
		if (!$self->GenerateTree ()){
			$self->AddWarning ("Dependencies of $gProductName installation \"" . $self->{_sid} . "\" are already broken: disable checkDependenciesAfterUpgrade");
			return 1;
		}
		$node = $self->{tree};
	}	
	
	my $rc = 1;
	
	if (!defined $visited){
		$visited = {};
	}
	my ($kpackage, $package, $kdep_package, @names);

	foreach my $id (keys %$node){

		if (exists $visited->{$id}){
			next;
		}

		$kpackage = $kit->GetPackageById ($id);

		if (!defined $kpackage || $kpackage->isSkipped () || !$kpackage->isSelected ()){
			$package = $self->GetPackageById ($id);
				if (!$package->CheckDependenciesByInstallation ($kit)){
					@names = ();
					foreach my $dependency (keys (%{$package->{dependencies}})){
						if (!$package->{dependencies}->{$dependency}->{resolved_by_installation}){
							$kdep_package = $kit->GetPackageById ($dependency);
							if (defined $kdep_package && $kdep_package->isSelected ()){
								push @names, $kdep_package->GetName ();
							}
						}
					}

					if (@names){
						$rc = undef;
						$self->PushError ('Upgrade of '. 
							(scalar @names == 1 ? 
							('package "' . $names[0]) : 
							( 'packages "' . join ('", "', @names))) . 
							'" violates dependency of installed package "' . $package->GetName() . '"');
					}
			}
		}
		elsif (defined $node->{$id} && %{$node->{$id}}){
			if (!defined $self->checkDependenciesAfterUpgrade ($kit,$node->{$id},$visited)){
					$rc = undef;
			}
		}
	}
	return $rc;
}

#----------------------------------------------------------------------------

sub removeClientInstRegDir{
    my ($self, $progPath) = @_;
    my $statProgPath = File::stat::stat($progPath);
    if (defined $statProgPath && -d $statProgPath){
        my $originalMode = enableWritePermissions($progPath, $statProgPath);
        rmdir($progPath.'/install');
        if (defined $originalMode){
            chmod($originalMode, $progPath);
        }
    }
    return 1;
}

#----------------------------------------------------------------------------

sub removeLstDir {
    my ($self) = @_;
    my $targetLstDir = $self->getTargetLstDir();
    my $statObj = File::stat::stat($targetLstDir);
    return 1 if !$statObj || ! -d $statObj;

    my $msg = $self->AddMessage("Removing directory '$targetLstDir'");
    return deltree($targetLstDir, $msg->getSubMsgLst(), $msg->getSubMsgLst());
}

#----------------------------------------------------------------------------

sub Uninstall{
	my ($self, $keep_user, $force) = @_;
	
	my $rc = 1;
	my $msglst = $self->getMsgLst ();

	unless ($self->{exclusive}){
		$self->setErrorMessage('Operation not permitted: no exclusive access to installation');
		return undef;
	}
	
	my $rmdirs_config = {};

	my $packages = $self->GetPackages ();
			
	if (!defined $packages) {
		$self->AddError ('Cannot remove installation',$self);
		return undef;
	}
	my $extLists = {};
	my $extList;
	foreach my $package (reverse @$packages){
		my $msg = $msglst->addProgressMessage ('Uninstalling package ' . $package->{data}->{name}.'...');
		$package->setMsgLstContext ([$msg->getSubMsgLst()],1);
		if (!defined $package->Uninstall ()){
			$self->AddError ('Cannot remove package ' . $package->{data}->{name},$package);
			$rc = undef;
			if (defined $msglst->{progress_handler}){
				$msglst->{progress_handler}->StepFinished(1);
			}
		}
		else{
            $extList = $package->getExternalFileListName ();
            if (defined $extList){
                $extLists->{$extList} = 1;
            }
			if (defined $msglst->{progress_handler}){
				$msglst->{progress_handler}->StepFinished();
			}
		}
	}

    foreach $extList (keys %$extLists){
        $self->syncExtFileList ($extList);
    }

	removeEmptyDirs ($self->getProgramPath (),$rmdirs_config);

	$msglst->AddMessage ("Removing empty program directories after uninstallation", $rmdirs_config);

    if (!$self->removeLstDir()) {
        return undef;
    }

	if (!$self->GetNumberOfPackages ()){
		if (!defined $self->Unregister ()){
			$msglst->PushError ('Unregistration failed');
			undef $rc;
		}
	
		if (!defined $self->PostUninstall ()){
			$msglst->PushError ('PostUninstall failed');
			undef $rc;
		}

		my $installationIniFilePath =
			$self->getPersistentInitialInstallerOptionsFilePath ();
		my $statInstallationIniFilePath =
			defined $installationIniFilePath ?
				File::stat::stat($installationIniFilePath) : undef;
		if (defined $statInstallationIniFilePath && -f $statInstallationIniFilePath){
			my $builtin = SDB::Common::BuiltIn->get_instance();
			if ($builtin->unlink ($installationIniFilePath)){
				$msglst->AddMessage ("File '$installationIniFilePath' deleted");
			}
			else{
				$msglst->PushError ("Cannot delete file '$installationIniFilePath': $!");
			}
		}

		$self->CloseRegistry ();
		$self->RemoveRegistry ();
        if(!$self->isa('SDB::Install::SAPSystem')) {
        	# remove '~/hdbclient/install/' and '~/hdbclient/'
        	# after we got rid of the INSTREG:
            if ($isWin){
                chdir('C:\\');
            }
            else {
                chdir('/');
            }
            my $progPath = $self->getProgramPath();
            $self->removeClientInstRegDir($progPath);
            rmdir($progPath);
        }
	}
	return $rc;
}

#----------------------------------------------------------------------------

sub AddInstallable{
	my ($self,$package) = @_;

	my $installed = $self->{registry}->AddPackage ($package);
	if (!defined $installed){
		$self->AddError ("Cannot create new installed package",$self->{registry});
		return undef;	
	}
	$installed->{installation} = $self;	
	if (!defined $self->AddPackage ($installed)){
		return undef;
	}
	return $installed;
}

#----------------------------------------------------------------------------

sub GetPackageList{
	my ($self) = @_;
	if (defined $self->{packages}){
		return $self->{packages};
	}
	if (!defined $self->{registry}){
		return {};
	}
	return $self->GenPackageList();
}

#----------------------------------------------------------------------------

sub CloseRegistry{
	my ($self) = @_;
	if (defined $self->{registry}){
		$self->{registry}->setMsgLstContext ([$self->getMsgLst ()]);
	}
	undef $self->{registry};
}

#----------------------------------------------------------------------------

sub RemoveRegistry{
	my ($self) = @_;
	if (defined $self->{registry}){
		$self->PushError ("Registry is still opened");
		return 0;
	}
	my $path = $self->{registrypath};
	
	if (!defined $path){
		$self->PushError ('Cannot get registry path');
		return 0;
	}
	
	$path .= $path_separator . $installRegistryFile;
	
	if (-f $path){
		if (!unlink ($path)){
			$self->PushError ("Cannot remove file \"$path\": $!");
			return 0;
		}
		else{
			$self->AddMessage ("File \"$path\" deleted");
		}
	}
	return 1;
}

sub syncExtFileList{
    my ($self,$listName) = @_;
    my $found = 0;
    my @files;
    my $file = $self->getProgramPath () . $path_separator . $listName;
    foreach my $package (values %{$self->{packages}}){
        if ($package->getExternalFileListName () eq $listName){
            $found = 1;
            push @files , keys (%{$package->GetFiles()});
        }
        
    }
    if (!$found){
        $file = $self->getProgramPath () . $path_separator . $listName;
        if (-f $file){
            $self->AddMessage ("Deleting external file list '$file'");
            if (!unlink ($file)){
                $self->AddError ("Cannot delete external file list '$file': $!");
            }
        }
    }
    else{
        if (!open (FH, '>'.$file)){
            $self->AddError ("Cannot create external file list '$file', $!");
            return undef;
        }
        if (!$isWin && $> == 0){
            chown ($self->getUID, $self->getGID, $file);
            chmod (0444, $file);
        }
        print FH join ("\n",sort @files) . "\n";
        close (FH);
    }
    return 1;
}



#----------------------------------------------------------------------------

sub Check{
	my ($self,$result, $confirm_handler, $repair) = @_;

	my $rc;

	my $repair_result;	

	my $packages = $self->GetPackages ();
	
	if (defined $self->{progress_handler}){
		$self->{progress_handler}->InitProgress ($self->GetNumberOfPackages,0);
	}

	unless (defined $packages){
		
		#
		# unresolved dependencies
		#
		
		$rc = 0;
		
		#
		#	try to check found packages
		#
		if (defined $self->{packages}){
			$packages = [values %{$self->{packages}}];
		}
		else{
			return $rc;
		}
	}
	
	my $package_result;
	my $package_rc;
	my $package_key;


	foreach my $package (@$packages){
		if (defined $result){
			$package_result = {'num_of_files' => $package->GetNumberOfFiles,
							   'package_name' => $package->GetName
							   };
		}
		else{
			$package_result = undef;
		}

		$package_key = '';
		
		my $msg = $self->AddProgressMessage ('Checking package ' . $package->GetName () . $package_key . '...');
		$package->setMsgLstContext ([$msg->getSubMsgLst()]);
		$package_rc = $package->CheckPackage ($package_result,$repair);
		
		$rc = (defined $package_rc and $rc);
		#$self->AddProgressMessage ('Checking package ' . $package->{data}->{name},$package);
		
		
		if (!defined $package_rc){
			
			$self->PushError ("Checking package $package->{data}->{name}$package_key failed",$package);
			
			if (defined $confirm_handler && $package->IsValid ()){
			 	if(&$confirm_handler ("Checking package $package->{data}->{name}$package_key failed:\n" . $package->GetErrorString ())){
					my $submsg = $self->AddMessage("Setting package $package->{data}->{name}$package_key invalid");
					my $saveContext = $package->setMsgLstContext ([$submsg->getSubMsgLst()]);
					$package->SetValid (0);
					$package->setMsgLstContext ($saveContext);
				}
			}
		}



		if (defined $result){
			if (!defined $package_rc){
				$package_result->{status} = 1;
			}
			else{
				$package_result->{status} = 0;
			}
			$result->{packages}->{$package->GetId} = $package_result;
			$result->{repair_result} = $repair_result;
		}
	}
	



	if (defined $result){
		$result->{status} = $rc;
	}
	
	return $rc;
}

#----------------------------------------------------------------------------


sub Register{
    my ($self) = @_;

    my $regFile = $self->getRegFile();
    if (!defined $regFile){
        return 1;
    }

    my $instconfig = $self->getConfiguration ();

    my $regDir = $self->getRegDir ();
    if (!-d $regDir){
        my $cfg = {'mode' => 0755};
        if (!defined makedir ($regDir, $cfg)){
            $self->AddError ("Cannot create " . $self->getProductName() . " registry", $cfg);
            return undef;
        }
    }

    my @buffer;
    my $create_new = 1;
    if (-f $regFile){
        $create_new = 0;
        if (!open (RG, $regFile)){
            $self->AddError ("Cannot open " . $self->getProductName() . " registry: $!");
            return undef;
        }
        @buffer = <RG>;
        close (RG);

    }
    my ($path, $version);
    my $found = 0;
    my $i;
    my @statbuf = stat $self->{path};
    my $nPath;

    if ($isWin){
        $nPath = normalizePath ($self->{path});
    }

    foreach $i (0 .. $#buffer){
        if ($buffer[$i] =~ /^\s*#/){
            next;
        }
        ($path, $version) = ($buffer[$i] =~ /(.*)=(.*)/);

        if (!defined $path){
            next;
        }

        $path =~ s/^\s*//;
        $path =~ s/\s*$//;
        $version =~ s/\s*$//;
        $version =~ s/^\s*//;
        if (isSameFile ($path, $nPath, \@statbuf)){
            $found = 1;
            $buffer[$i] = "$self->{path}=".$self->GetVersion () . "\n";
        }
    }
    if (!$found){
        push @buffer, "$self->{path}=".$self->GetVersion () . "\n";
    }

    my $tmp_file = $regFile . MD5Str ($self->{path});

    if (!open (RG, ">$tmp_file")){
        $self->AddError ("Cannot create " . $self->getProductName() . " registry ($tmp_file): $!");
        return undef;
    }
    foreach $i (0 .. $#buffer){
        print RG $buffer[$i];
    }
    close (RG);

    if (!$create_new){
        if (!unlink ($regFile)){
            $self->AddError ("Cannot delete origin " . $self->getProductName() . " registry: $!");
            return undef;
        }
    }

    if (!rename ($tmp_file, $regFile)){
        $self->AddError ("Cannot rename " . $self->getProductName() . " registry $tmp_file => $regFile: $!");
        return undef;
    }
    return 1;
}

#----------------------------------------------------------------------------

sub Unregister{
    my ($self) = @_;
    my $regFile = $self->getRegFile();
    if (!defined $regFile){
        return 1;
    }
    if (-f $regFile){
        if (!open (RG, $regFile)){
            $self->AddError ("Cannot open registry '$regFile': $!");
            return undef;
        }
    }
    my @buffer = <RG>;

    close (RG);
    my $msg = $self->AddMessage ('Looking for ' . $self->getProductName() . ' installations...');
    $self->SetFormatInfo ($msg, 'h1', 'Looking for ' . $self->getProductName() . ' installations');
    my ($path, $version);

    my @statbuf = stat ($self->{path});
    my $nPath;
    if ($isWin){
        $nPath = normalizePath ($self->{path});
    }

    my $found = undef;

    foreach my $i (0 .. $#buffer){
        if ($buffer[$i] =~ /^\s*#/){
            next;
        }
        ($path, $version) = ($buffer[$i] =~ /(.*)=(.*)/);

        if (!defined $path){
            next;
        }

        $path =~ s/^\s*//;
        $path =~ s/\s*$//;
        if (@statbuf){
            if (isSameFile ($path,$nPath,\@statbuf)){
                $found = $i;
                last;
            }
        }
        else{
            if (normalizePath ($path) eq normalizePath ($self->{path})){
                $found = $i;
                last;
            }
        }
    }

    if (defined $found){
        if (!open (RG, '>'. $regFile)){
            $self->AddError ("Cannot create registry '$regFile': $!");
            return undef;
        }
        foreach my $i (0 .. $#buffer){
            if ($i == $found){
                next;
            }
            print RG $buffer[$i];
        }
        close (RG);
    }
    return 1;
}

#----------------------------------------------------------------------------

sub PostUninstall{
	return 1;
}

#----------------------------------------------------------------------------

sub FlushRegistry{
	my ($self,$msg) = @_;
	my $msglst;
	if (defined $msg){
		$msglst = $msg->getSubMsgLst()
	}
	else{
		$msglst = $self->getMsgLst ();
	}
	my $writeMsg = $msglst->addMessage ('Writing registry');
	$self->{registry}->setMsgLstContext([$writeMsg->getSubMsgLst ()]);
	if (! defined $self->{registry}->Write ()){
		$self->AddError ('Cannot write install registry', $self->{registry});
		return undef;
	}
	return 1;
}


#----------------------------------------------------------------------------

sub RemovePackageFromRegistry{
	my ($self,$package) = @_;
	$self->{registry}->DeletePackage($package->{id});
}

#----------------------------------------------------------------------------

sub SelectPackage{
	my ($self,$package) = @_;
	
	if (!defined $package){
		return 0;
	}
	
	if(!exists $self->{'packages'}->{$package->{'id'}}){
		$self->AddError ("Package \"$package->{data}->{name}\" not found");
		return 0;
	}
		
	if (exists $package->{'tree_node'}){
		foreach my $child_id (keys %{$package->{tree_node}}){
			if (!$self->SelectPackage ($self->{'packages'}->{$child_id})){
				return 0;
			}
		}
	}
	
	

	$package->{'selected'} = 1;

	return 1;
}

#----------------------------------------------------------------------------

sub UnselectPackage{
	my ($self,$package) = @_;
	
	if (!defined $package){
		return 0;
	}
	
	if(!exists $self->{'packages'}->{$package->{id}}){
		$self->AddError ("Package \"$package->{data}->{name}\" not found");
		return 0;
	}
	
	my $required_packages = $self->GetRequiredPackages ($package);
	
	if (! defined $required_packages){
		return 0;
	}
	
	
	foreach my $required_package (@$required_packages){
		if (!$self->UnselectPackage ($required_package)){
			return 0;
		}
	}

	$package->{'selected'} = 0;

	return 1;
}


#
# registryPath is the directory where INSTREG file is located
# INSTREGISTRY file contains the package meta data of the installed packages
#

#----------------------------------------------------------------------------

sub setRegistryPath{
	$_[0]->{registrypath} = $_[1];
}

#----------------------------------------------------------------------------

sub getRegistryPath{
	return $_[0]->{registrypath};
}

#----------------------------------------------------------------------------

sub setSubstMacros{
	$_[0]->{substMacros} = $_[1];
}

#----------------------------------------------------------------------------

sub getSubstMacros{
	return $_[0]->{substMacros};
}

#----------------------------------------------------------------------------


sub get_persistent_keys{
	my ($self) = @_;
	return ['name', 'path', 'comment'];
}

sub getpersfilename ($){
	my ($self) = @_;
	if (defined $self->{persfile}){
		return $self->{persfile};
	}
	return '';
}


sub store ($){
	my ($self,$file) = @_;

	my $keys = $self->get_persistent_keys ();	

	if (!@$keys){
		return 1;
	}

	if (!defined $file){
		$file = $file = getpersfilename ($self);
	}

	if (-f $file){
			$self->AddError ("File \"$file\" already exists");
			return 0;	
	}

	if (!open (FH, '>' . $file)){
		$self->AddError ("Cannot create file \"$file\": $!");
		return 0;
	}

	foreach my $key (@$keys){
		if (defined $self->{$key}){
			print FH "$key=$self->{$key}\n";
		}
	}
	close (FH);
	return 1;
}


sub load ($){
	my ($self,$file) = @_;
	my $keys = $self->get_persistent_keys ();	

	if (!@$keys){
		return 1;
	}

	if (!defined $file){
		$file = getpersfilename ($self);
		if (!$file){
			$self->AddError ("Cannot detect persfile name");
			return 0;
		}
	}

	if (!-f $file){
		$self->AddError ("Cannot open file \"$file\": $!");
		return 0;
	}
	my $errMsgLst = new SDB::Install::MsgLst ();
	my $data = readini ($file, $errMsgLst);
	if (!defined $data){
		$self->setErrorMessage ("Cannot read ini file '$file'", $errMsgLst);
		return 0;
	}

	my $persdata = $data->{''};
	
	foreach my $key (@$keys){
		if (defined $persdata->{$key}){
			$self->AddMessage ("Setting $key = $persdata->{$key}");
			$self->{$key} = $persdata->{$key};
		}
		else{
			$self->AddMessage ("$key isn\'t defined");
		}
	}
	if (!defined $self->init ()){
		return 0;
	}
	return 1;
}


sub remove{
	my ($self) = @_;
	my $file = getpersfilename ($self);
	
	if (-f $file){
		if (unlink ($file)){
			$self->AddMessage ("File \"$file\" deleted");
		}
		else{
			$self->AddError ("Cannot delete File \"$file\": $!");
			return 0;
		}
	}
	return 1;
}


#----------------------------------------------------------------------------

#
# override PackageManager::GetPackages to get packages even if there are broken dependencies
#

sub GetPackages{
	return $_[0]->SUPER::GetPackages (undef,1);
}

#----------------------------------------------------------------------------

sub RemovePackage{
    my ($self,$package,$is_update) = @_;
    $self->RemovePackageFromRegistry ($package);
    $self->SUPER::RemovePackage ($package,$is_update);
}

#----------------------------------------------------------------------------

sub RemoveDeprecatedPackages{
    my ($self,
        $packageIds
    ) = @_;
    if(not $self->{'create_new'}) {
        # only when we are a software update:
        my @deprecatedPackageIds = @$packageIds;
        if (@deprecatedPackageIds) {
            foreach my $deprecatedPackageId (@deprecatedPackageIds) {
                my $deprecatedPackage = $self->GetPackageById($deprecatedPackageId);
                if(defined $deprecatedPackage) {
                    $self->AddMessage ("Removing deprecated Package '$deprecatedPackage->{data}->{name}'");
                    my $keepFiles = 0;
                    if($self->isServer() && not $isWin) {
                        $keepFiles = 1;
                    }
                    my $removeEmptyDirs = !$keepFiles;
                    $deprecatedPackage->Uninstall($keepFiles, $removeEmptyDirs);
                }
            }
        }
    }
    return 1;
}

#----------------------------------------------------------------------------

sub FreePackages{
	my ($self) = @_;
	$self->SUPER::FreePackages ();
	$self->CloseRegistry ();
	return 1;
}

#----------------------------------------------------------------------------

sub getUID {
	return undef;
}

#----------------------------------------------------------------------------

sub getGID {
    return undef;
}

#----------------------------------------------------------------------------


sub getStartMenuPath{
	my ($self) = @_;
	my $errlst = new SDB::Install::MsgLst();
	my $path = getSystemStartMenuPath ($self->isUserMode (), $errlst);
	if (!defined $path){
		$self->AddError (undef, $errlst);
		return undef;
	}
	return $path;
}


#----------------------------------------------------------------------------

sub getHANAFlavour {
    my ($self) = @_;
    
    my $mf = $self->getManifest ();
    return undef if ! defined $mf || ! $mf->exists();
    
    return $mf->getHANAFlavour();
}

#----------------------------------------------------------------------------

sub DESTROY{
	my ($self) = @_;
	$self->FreePackages();
	$self->SUPER::DESTROY ();
}

#----------------------------------------------------------------------------

1;
