package SDB::Install::LSTFile;
use parent qw(SDB::Install::Base);
use strict;
use warnings;
use IO::File;

my $fileEntryRegex = '\w+\s(X|\s)\s\w+';

sub new {
    my $self = shift->SUPER::new();
    my ($path) = @_;

    $self->{ownPath} = $path;
    return $self;
}

sub getOwnPath {
    my ($self) = @_;
    return $self->{ownPath};
}

sub getFileHash {
    my ($self, $file) = @_;
    if (!$self->_getIsParsed()) {
        if (!$self->_parse()) {
            my $path = $self->getOwnPath();
            $self->getErrMsgLst()->addError("Failed to get hash entry of '$file' from '$path'");
            return undef;
        }
    }
    if (!exists $self->{entries}) {
        return undef;
    }
    return $self->{entries}->{$file};
}

sub _parse {
    my ($self) = @_;
    my $path = $self->getOwnPath();
    $self->getMsgLst()->addMessage("Parsing LST file at '$path'");

    my $lines = $self->_getFileLines();
    if (!$lines) {
        return 0;
    }

    my $headerLine   = shift(@{$lines});
    my $hashTypeLine = shift(@{$lines});

    if ($headerLine ne 'LSS-HASHES' || $hashTypeLine ne 'Hash: SHA256') {
        $self->getErrMsgLst()->addError("Failed to parse LST file '$path'. Did not match LSS file synthax with SHA256 hash");
        return 0;
    }

    for my $line (@{$lines}) {
        next if ($line !~ /$fileEntryRegex/);
        my @entries = split(' ', $line);
        my $shaHash  = $entries[0];
        my $fileName = $entries[-1];
        $self->_addHashedFileEntry($fileName, $shaHash);
    }
    $self->_setIsParsed(1);
    return 1;
}

sub _addHashedFileEntry {
    my ($self, $file, $hash) = @_;
    if (!exists $self->{entries}) {
        $self->{entries} = {};
    }
    $self->{entries}->{$file} = $hash;
}

sub _getFileLines {
    my ($self) = @_;
    my $path = $self->getOwnPath();
    my $file = IO::File->new($path);
    if (!$file) {
        $self->getErrMsgLst()->addError("Failed to open LST file '$file'");
        return undef;
    }

    chomp(my @lines = $file->getlines());

    $file->close();
    return \@lines;
}

sub _setIsParsed {
    my ($self, $isParsed) = @_;
    $self->{_isParsed} = $isParsed;
}

sub _getIsParsed {
    my ($self) = @_;
    return $self->{_isParsed};
}

1;