package SDB::Install::MonitoredTraceDirectory;

use base SDB::Install::BaseLegacy;
use SDB::Install::SysVars qw ($path_separator $isWin);
#use SAPDB::Install::MD5Sum;
use SAPDB::Install::GnuTar;
use strict;

sub new{
    my $self = shift->SUPER::new (); 
    (
        $self->{'fullDirName'},       # mandatory
        $self->{'fullArchiveName'}    # mandatory
    ) = @_;
    if($isWin) {
        $self->{'fullDirName'}     =~ s/\//\\/g;
        $self->{'fullArchiveName'} =~ s/\//\\/g;
    }   
    $self->{'snapshots'} = [];
    return $self;
}

sub setTraceDirPath {
    my ($self, $path) = @_;
    $self->{fullDirName} = $path;
}

sub takeSnapshot {
    my (
        $self,
        $label, # optional; array $self->{'snapshots'} is ordered chronologically anyway
     ) = @_;
     if(!opendir TRACEDIR, $self->{'fullDirName'}) {
        $self->AddError ("could not open directory '$self->{'fullDirName'}'");
        return undef;
     }
     my @statbuf;
     my $snapshot = {};
     my $path;
     my $dir;
     $snapshot->{'label'} = $label;
     $snapshot->{'filedescs'} = {};
     while(defined (my $entry = readdir TRACEDIR)) {
         if($entry =~ /^\.\.?$/) {
             next;
         }
         $dir = $self->{'fullDirName'};
         $dir =~ s/^(\/|\\){1}$//; # eliminate root dir if it is the same as path_separator
         $path = $dir . $path_separator . $entry;
         if (-l $path) {
            # we ignore symlinks.
			next;
         }
         if (-d $path) { # do not combine with -l, we want to use the stat buffer later
            # we ignore subdirs.
			next;
         }
         if(isIgnorableTraceDirEntry($entry)) {
             # this rejects files just by properties of their names, before we had to stat them.
             next;
         }
         @statbuf = stat(_);
         my $filedesc = {
             'size'    => $statbuf[7], 
             'mtime'   => $statbuf[9],
             #'md5sum'  => MD5Sum($path)
         };
         $snapshot->{'filedescs'}->{$entry} = $filedesc;
     }
     push(@{$self->{'snapshots'}}, $snapshot);
     closedir TRACEDIR;
     return 1;
}

sub analyzeDirectoryDeltas {
    my (
        $self,
        $msgList,      # mandatory, to append found error messages to
        $createArchive # 1,0, or undef, optional
    ) = @_;
    if(not $self->takeSnapshot("IMPLICIT_FINAL")) {
        return undef;
    }
    $self->diffCurrentSnapshot($msgList);
    my $snapshotCount = scalar(@{$self->{'snapshots'}});
    my $currentSnapshot = $self->{'snapshots'}->[$snapshotCount-1];
    my $diff = $currentSnapshot->{'diff'};
    my $diffcount = scalar(keys %$diff);
    if($diffcount <= 0) {
        $self->AddWarning("Empty snapshot diff: no errors found.");
        return 1;                                                 
    }
    my $gnutar;
    if($createArchive) {
        $gnutar = new SAPDB::Install::GnuTar();
        my $pmask = 0222;
        if(!defined $gnutar->OpenArchive($self->{'fullArchiveName'}, 'w', $pmask)) {
            $self->AddError("Could not create archive '$self->{'fullArchiveName'}'.");
            return undef;
        }
    }
    foreach my $file (keys %$diff) {
        my $tracefile = $self->{'fullDirName'} . $path_separator . $file;
        # read the new part of the file into a buffer
        my $fh;
        if(!open $fh, "<", $tracefile) {
            $self->AddError ("could not open file '$tracefile' for input");
            return undef;
        }
        my $position = $diff->{$file}->{'oldsize'};
        if (!$isWin && $position > 0) {
            $position -= 1;
        }
        seek($fh, $position, 0);                     ####### heading empty line. why?
        my @buffer;
        while (!eof($fh)) {
            push @buffer, <$fh>;
        }
        close($fh);
        my ($basename) = ($file =~ /([^\/]+$)/);
        my $pattern = getErrorPattern($basename);
        my @msgLines;
        if(defined $pattern) {
            foreach my $line (@buffer) {
                if($line =~ /$pattern/) {
                    { # just for 'local'
                        local $/ = "\r\n";
                        chomp ($line);
                    }
                    chomp ($line);
                    push @msgLines, "$basename: $line";
                }
            }
        }
        # limit number of lines copied from trace file into log file to 50
        if(scalar(@msgLines) > 50) {
            $msgList->AddMessage("$basename: More than 50 relevant messages found, check file.");
        }else{
            foreach my $line (@msgLines) {
                $msgList->AddMessage($line);
            }
        }
        if($createArchive) {
            my $buf = join("", @buffer);
            my $deltaFile = $file;
            if($basename !~ /\./) { # handle file w/o extension
                $deltaFile = $deltaFile.".delta";
            }
            else {
                $deltaFile =~ s/(.*)(\..*)$/$1.delta$2/;
            }
            if (!defined $gnutar->AddScalar($deltaFile, $buf)) {
                $self->AddError("Could not add changes of file '$file' to archive.");
                next;
            }
        }
    }
    if($createArchive) {
        my $rc = $gnutar->CloseArchive();
        if(not defined $rc) {
            $self->AddError("Could not close archive.");
            return undef;
        }
    }
    return 1;
}



####################
# private methods/functions meant to be "not visible from outside":

sub diffCurrentSnapshot {
    my (
        $self,
        $msgList      # mandatory, to append found error messages to
     ) = @_;
     my $snapshotCount = scalar(@{$self->{'snapshots'}});
     if($snapshotCount <= 0) {
        $self->AddError("No snapshot of directory '$self->{'fullDirName'}' available.");
        return undef;
     }
     my $currentSnapshot = $self->{'snapshots'}->[$snapshotCount-1];
     if(defined $currentSnapshot->{'diff'}) {
         return 1; # we have been called before and have computed the diff already.
     }
     my $diff = {};
     if($snapshotCount == 1) {
         my $filedescs = $currentSnapshot->{'filedescs'};
         foreach my $file (keys %$filedescs) {
			my $currEntry = $currentSnapshot->{'filedescs'}->{$file};
			my $filediff = {
				'newsize' => $currEntry->{'size'},
				'oldsize' => 0,
				'newmtime' => $currEntry->{'mtime'},
				'oldmtime' => $currEntry->{'mtime'}
			};
			$diff->{$file} = $filediff;
         }
     }
     else {
         my $previousSnapshot = $self->{'snapshots'}->[$snapshotCount-2];
         my $filedescs = $currentSnapshot->{'filedescs'};
         foreach my $file (keys %$filedescs) {
			my $filediff = {};
			my $currEntry = $currentSnapshot->{'filedescs'}->{$file};
            my $totalMaxmegs = 1000; # we dont process files larger than $totalMaxmegs MB.
            my $deltaMaxmegs = 100;  # we dont process files which have changed in size by more than $deltaMaxmegs MB.
			if (defined $previousSnapshot->{'filedescs'}->{$file}) {
				my $prevEntry = $previousSnapshot->{'filedescs'}->{$file};
				#if (defined $prevEntry->{'md5sum'} && ($prevEntry->{'md5sum'} eq $currEntry->{'md5sum'})) {
				#	next;
				#}
				if($currEntry->{'size'} == $prevEntry->{'size'}) {
				    # we are only interested, when file size has changed.
				    next;
				}
                if($currEntry->{'size'} > $totalMaxmegs*1000000) {
                    # we dont process files larger than $totalMaxmegs MB.
                    $msgList->AddMessage("NOT processing $file: Size is greater than $totalMaxmegs MB.");
                    next;
                }
				if ($currEntry->{'size'} > $prevEntry->{'size'}) {
					$filediff->{'oldsize'} = $prevEntry->{'size'};
				}
				else {
				    # file shrank, probably it was overwritten (with interesting stuff), so take it completely.
					$filediff->{'oldsize'} = 0;
				}
				$filediff->{'oldmtime'} = $prevEntry->{'mtime'};
			}
			else {
				$filediff->{'oldsize'} = 0;
				$filediff->{'oldmtime'} = $currEntry->{'mtime'};
			}
            if($currEntry->{'size'} - $filediff->{'oldsize'} > $deltaMaxmegs*1000000) {
                # we dont process files which have changed in size by more than $deltaMaxmegs MB.
                $msgList->AddMessage("NOT processing $file: Size has changed by more than $deltaMaxmegs MB.");
                next;
            }
 			$filediff->{'newsize'} = $currEntry->{'size'};
			$filediff->{'newmtime'} = $currEntry->{'mtime'};
			$diff->{$file} = $filediff;
         }
     }
     $currentSnapshot->{'diff'} = $diff;
     return 1;
}

sub isIgnorableTraceDirEntry {
    my (
        $path
     ) = @_;

    if ($path =~ /(\.bin|\.mdmp|\.tgz|\.zip|core.*)$/i) { # add more patterns for binary files here
         return 1;
    }
    return 0;
}

sub getErrorPattern {
    my (
        $basename
     ) = @_;
     my $pattern = undef;
     if($basename =~ /\.trc$/) {
         if($basename =~ /crash/i) {
             # e.g. "backupHole_nameserver_ls9263.31701.crashdump.20120816-181932.45323.trc"
             # dont check for errors, if crashdump.
         }
         elsif($basename =~ /rfc\.trc$/) {
             # e.g. "ls9263_trace_dev_rfc.trc"
             $pattern = 'ERROR';
         }
         elsif($basename =~ /loads\..?\.trc$/) {
             # e.g. "ls9263_trace_indexserver_ls9263.31703.unloads.000.trc"
             # dont check for errors, if unload/load log.
         }
         elsif($basename =~ /history\.trc$/) {
             # e.g. "ls9263_trace_nameserver_history.trc"
             # dont check for errors, if history log.
         }
         elsif($basename =~ /recover\.trc$/) {
             # e.g. "ls9263_trace_recover.trc"
             $pattern = 'ERROR';
         }
         elsif($basename =~ /trace_sapstart.*\.trc$/) {
             # e.g. "ls9263_trace_sapstart2.trc"
             $pattern = 'ERROR|[Ee]rror';
         }
         else {
             # "[5344]{0}[0] 2012-12-19 15:17:27.541569 i Memory       AllocatorImpl.cpp(00778) : Allocators activated"
             $pattern = '^[[\]{}/\-\d]+\s+[\d-]+\s+[\d:.]+\s+(e|f)\s+|[Ss][Yy][Ss][Tt][Ee][Mm]\s[Ee][Rr][Rr][Oo][Rr]';
         }
     }
     elsif ($basename =~ /^hdb.*\.log$/) {
         # dont check for errors, if installer log.
     }
     else {
         $pattern = '[Ee][Rr][Rr]|[Ff][Aa][Ii][Ll]|[Cc][Rr][Aa][Ss][Hh]|[Ww][Rr][Oo][Nn][Gg]';
     }
     
     return $pattern;
}

1;
