#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: NewDB installation package (installable archive)


package SDB::Install::Package::Installable;

use SDB::Install::Package;
use SDB::Install::Archive;
use SDB::Install::System qw (copy_file makedir enableWritePermissions);
use SAPDB::Install::MD5Sum;
use SDB::Install::SysVars;
use SDB::Install::Globals qw ($gProductName);
use File::Basename qw (dirname);
use SDB::Install::Version;
use SDB::Install::LSTFile;
use Digest::SHA qw(sha256_hex);

use File::stat;
use Fcntl ':mode';

our @ISA = qw (SDB::Install::Package SDB::Install::Archive);

our $script = 'script.pm';
our $filelist = 'files.lst';

sub new{
	my $self = shift->SUPER::new();
	my ($archive, $lstFilePath, $verifyChecksums) = @_;
	my $lstFile = defined $lstFilePath ? SDB::Install::LSTFile->new($lstFilePath) : undef;
	$self->{verifyChecksums} = $verifyChecksums;
	$self->{archive} = $archive;
	$self->{lstFile} = $lstFile;
	$self->{update}  = 0;
	return $self;
}

sub getArchivePath {
    my ($self) = @_;
    return $self->{archive};
}

sub getPackageLSTFile {
	my ($self) = @_;
	return $self->{lstFile};
}

sub shouldVerifyChecksums {
	my ($self) = @_;
	return $self->{verifyChecksums} && !$self->isArchiveWritableForRootOnly();
}

sub isArchiveWritableForRootOnly {
    my ($self) = @_;
    my $archive = $self->getArchivePath();
    my $stat = File::stat::stat($archive);
    return 0 if (!$stat);
    my $uid = $stat->uid();
    my $gid = $stat->gid();
    my $isWritableForOthers = ($stat->mode() & S_IWOTH) != 0;
    return ($uid == 0 && $gid == 0 && !$isWritableForOthers);
}

#
# open archive + read meta data
#


sub init{
	my ($self, $kitdir) = @_;
	$self->{_kitDir} = $kitdir;
	unless (defined $self->open()){
		$self->AddError ('Cannot open archive', $self);
		return undef;
	}
	
	my $next = $self->next ();
	
	unless (defined $next){
		return undef;
	}
	
	unless ($next->{name} eq 'PACKAGEDATA'){
		$self->AddError ('No meta data found');
		return undef;
	}
	
	my $data = $self->read_meta ();	
	
	unless (defined $data){
		return undef;
	}	

	$self->{meta_data} = $data;
	
	if ($data->{INTERFACE_VERSION} ne '0.5'){
		$self->AddError ('Incompatible software package version ' .
			$data->{INTERFACE_VERSION});
		return undef;	
	}

	my $file = $self->next ();
	
	if ($file->{name} eq $filelist){
		$self->{data}->{files} = $self->read_filelist();	
	}
	else{
		$self->AddError ("Archive is not a valid $gProductName package: file list not found");
		return undef;
	}

	$self->initProperties ();
	
	$self->initDebugPackage ($kitdir);

	$file = $self->next ();
		
	if ($file->{name} eq $script){
		$self->{data}->{script} = $self->get_file_content ();
		if ($self->shouldVerifyChecksums() && !$self->_checkScriptFileSHASignature()) {
			$self->getErrMsgLst()->addError ("Archive is not a valid $gProductName package: failed to validate authenticity of $script file");
			return undef;
		}
	}
	else{
		$self->AddError ("Archive is not a valid $gProductName package: script not found");
		return undef;
	}

	return $self->evaluateScript ();
}


sub ParseRequireStr{
	my ($self,$string) = @_;
	my @requires = ();
	foreach my $requirement (split(',',$string)){
		if ($requirement =~ />|>=|={1,2}/){
			my ($package,$operand,$version,$mode) = ($requirement =~ /(.*\S)\s+([>=]{1,2})\s+(\S+)(.*)/);
			$operand = '==' if $operand eq '=';
			if ($mode =~ /\d\d/){
				($mode) = ($mode =~ /(\d\d)/);
			}
			else{
				undef $mode;
			}
			push @requires, {'id' => $package,
					 'operand' => $operand,
					 'version' => $version,
					 'str' => $requirement,
					 'mode' => $mode
					};
		}
		else{
			push @requires, {'id' => $requirement,
					 'str' => $requirement
					};
		}
	}
	return \@requires;
}

sub initProperties{
	my ($self) = @_;
	return 1 if $self->{initialized};
	my $packagedata = $self->{meta_data};
	
	my $persdata = $self->{data}; 

	$persdata->{name} = $packagedata->{NAME};
	$self->{id} = $packagedata->{ID};
	$persdata->{buildstring} = $packagedata->{BUILD_STRING};
	$persdata->{git_hash} = $packagedata->{GIT_HASH};
	$persdata->{version} = $packagedata->{VERSION};
	$persdata->{desc}  = $packagedata->{DESC};
	$persdata->{debugArchive}  = $packagedata->{DEBUG_ARCHIVE};
    if ($packagedata->{IS_GLOBAL} eq '1'){
        $persdata->{global} = 1;
    }
    if ($packagedata->{IS_CLIENT_PACKAGE} eq '1'){
        $persdata->{isClientPackage} = 1;
    }
	$persdata->{checksum}  = $packagedata->{CHECKSUM};
	if (defined $packagedata->{REQUIRE}){
		$persdata->{requires} = $self->ParseRequireStr ($packagedata->{REQUIRE});
	}
	if (defined $packagedata->{REQUIRE_CONDITIONAL} && ($packagedata->{REQUIRE_CONDITIONAL} =~ /\S/)){
		$persdata->{require_conditional} = $self->ParseRequireStr (
				$packagedata->{REQUIRE_CONDITIONAL});
	}
	$self->{data}->{size} = $packagedata->{SIZE};
	if (defined $packagedata->{TEST_FILE}){
		$persdata->{test_file} = $packagedata->{TEST_FILE};
		$persdata->{test_file_magic} = $packagedata->{TEST_FILE_MAGIC};
		foreach my $key (keys (%$packagedata)){
			if ($key =~ /\./){
				my ($class,$property) = ($key =~ /(.*)\.(.*)/);
				if ($class eq 'SYSINFO'){
					unless (defined $persdata->{sysinfo}){
						$persdata->{sysinfo} = {};
					}
					$property =~ tr/[A-Z]/[a-z]/;
					$persdata->{sysinfo}->{$property} = 
							$packagedata->{$key};
				}
			}
			elsif ($key =~ /compiler/i){
				unless (defined $persdata->{compiler_infos}){
					$persdata->{compiler_infos} = {};
				}
				$key =~ tr/[A-Z]/[a-z]/;
				$key =~ s/_/\ /g; 
				$persdata->{compiler_infos}->{$key} = $packagedata->{$key};
			
			}
		}	
	}
	
	if (defined $packagedata->{REFUSE_SKIP}){
		$self->{refuseSkip} = $packagedata->{REFUSE_SKIP};
	}

	if (defined $packagedata->{DISTRIBUTION}){
			$persdata->{distribution} = $packagedata->{DISTRIBUTION};
	}



	$self->{initialized} = 1;
	delete $self->{meta_data};
}


sub CheckUpdate{
	my ($self, $repair_mode, $force_downgrade) = @_;
	
	my $installed_package = $self->{installation}->{packages}->{$self->{id}};
	
	my $persdata = $self->{data}; 

	unless (defined $installed_package){
		$self->AddError ('Cannot get installed package "'.$persdata->{name}.'"', $self->{installation}->{registry});
		return undef;
	}
	
	$installed_package->{installation} = $self->{installation};
	
	my $progress_handler = $self->GetProgressHandler ();
	
	if ($installed_package->{data}->{valid}){
		
		#
		# version check
		#
		my $version = new SDB::Install::Version (split ('\.', $persdata->{version}));
		my $installed_version = new SDB::Install::Version (split ('\.', $installed_package->{data}->{version}));
		if ($version->isNewerThan ($installed_version)){
		    $self->AddMessage ('Update version check succeeded: '.$persdata->{version}.' > ' . $installed_package->{data}->{version});
		    $self->AddMessage ("Old checksum = $installed_package->{data}->{checksum}");
		}
		elsif ($version->isEqual ($installed_version)){
			$self->AddMessage ('Update version check succeeded: '.$persdata->{version}.' == ' . $installed_package->{data}->{version});
			if ($persdata->{checksum} eq $installed_package->{data}->{checksum}){
				$self->AddMessage ("Same package already installed (checksum = $persdata->{checksum})");
				if (!$repair_mode){
					if ($self->isServerPackage () && !$self->isGlobal () && !-d $self->{installation}->getProgramPath ()){
						$self->AddMessage ('Reinstallation of package ' . $persdata->{name} . ' required due to new version directory');
					}
					elsif(!$force_downgrade){
						$self->AddProgressMessage ('Skipping package ' . $persdata->{name});
						$self->{skip} = 1;
					}
				}
				else{
					my $msg = $self->AddProgressMessage ('Checking installed package '. $installed_package->{data}->{name});
					if (defined $progress_handler){
						$installed_package->SetProgressHandler ($progress_handler);
					}
				
					if (defined $installed_package->CheckPackage ()){
						$self->AddMessage ('Installed package is installed correctly');
						if(!$force_downgrade){
							$self->AddProgressMessage ('Skipping package ' . $persdata->{name});
							$self->{skip} = 1;
						}
					}
					else{
						$self->AddWarning ('Installed package is corrupted and has to be installed again');
					}	
					$self->AddSubMsgLst ($msg,$installed_package);
				}
			}
			else{
				$self->AddMessage ("Update with same version (old checksum = $installed_package->{data}->{checksum})");
			}
		}
		else{
			if (defined $progress_handler){
				$progress_handler->IncValue (0 - $self->GetNumberOfFiles());
			}
			my $msglst = new SDB::Install::MsgLst ();
			
			$msglst->AddMessage ('Version check failed: '.
				$persdata->{version}.' < ' . $installed_package->{data}->{version});
									
			if (!$force_downgrade && $self->canSkip ()){
				$self->AddMessage ('Skipping package '.$self->GetName(), $msglst);
				$self->{skip} = 1;
			}
			else{
				$self->AddError ('Cannot downgrade package ' . $self->GetName(), $msglst);
				if ($force_downgrade){
					$self->AddMessage ("Ignoring error due to command line switch '--ignore'");
					$self->AddMessage ("Old checksum = $installed_package->{data}->{checksum}");
				}
				else{
					return undef;
				}
			}
		}
		
	}

	$self->{installed_package} = $installed_package;
	$self->{installed_package}->{installation} = $self->{installation};
	$self->{update} = 1;
	return 1;
	
}


sub initDebugPackage{
	my ($self,$arch_dir) = @_;
	
	my $dbg_archive = $self->{data}->{debugArchive};
	
	unless (defined $dbg_archive){
		return 1;
	}
	
	unless (-f "$arch_dir/$dbg_archive"){
			$self->AddMessage ("Debug package \"$dbg_archive\" not found");
			return 1;
	}
	
	require SDB::Install::DebugPackage;
	my $dbg_package = new SDB::Install::DebugPackage ("$arch_dir/$dbg_archive");
	
	if ($dbg_package->ErrorState ()){
		$self->AddWarning ('Invalid debug package', $dbg_package);
		return undef;
	}
	$self->AddMessage ("Checking debug package \"$dbg_archive\"", $dbg_package);
	$self->{debugPackage} = $dbg_package;
	return 1;
}



sub IsRunnable{
    my ($self,$sysinfo, $instconfig) = @_;
    my $persdata = $self->{data}; 
    unless (defined $persdata->{test_file}){
        return 1;
    }

    my $msg = $self->AddMessage ("Checking system requirements");

    unless( SAPDB::Install::SysInfo::IsRunnable ($sysinfo, $persdata->{sysinfo}, $persdata->{test_file_magic})){
        my $errlst = $msg->getSubMsgLst ();
        $errlst->AddMessage ('file magic: ' . $persdata->{test_file_magic});
        if (defined $persdata->{sysinfo}){
            my $mmsg = $errlst->AddMessage ('MAKE SYSTEM:');
            my $omsg = $errlst->AddMessage ('YOUR SYSTEM:');
            foreach my $key (keys (%{$persdata->{sysinfo}})){
            	$errlst->AddSubMessage ($mmsg, $key . ' = '  . $persdata->{sysinfo}->{$key});
            	$errlst->AddSubMessage ($omsg, $key . ' = '  . $sysinfo->{$key});
            }
        }
        $self->setErrorMessage ("Software isn\'t runnable on your system", $errlst);
        if (!defined $instconfig || !$instconfig->getIgnore ('check_platform') &&
                !$instconfig->getIgnore ('check_hardware')){
            return undef;
        }
        $self->getMsgLst ()->addMessage ("Ignoring error due to ignore option 'check_platform'");
    }
    return 1;
}

sub CheckDependenciesByKit{
	my ($self,$kit) = @_;


	my $rc = 1;

	return 1 unless defined $self->{data}->{requires};

	unless (defined $kit){
		$kit = $self->{kit};
	}

	foreach my $dependency (@{$self->{data}->{requires}}){
				
	if ($self->CheckDependency ($dependency,$kit->{packages})){
			if (exists $self->{dependencies}->{$dependency->{id}}){
				$self->{dependencies}->{$dependency->{id}}->{resolved_by_kit} = 1;
			}
		}
		else{
			if (exists $self->{dependencies}->{$dependency->{id}}){
				$self->{dependencies}->{$dependency->{id}}->{resolved_by_kit} = 0;
				$rc = 0;
			}
		}
	}
	return $rc;
	
}


sub Install{
	my ($self) = @_;
	
	unless (defined $self->{installed_package}){
		$self->{installed_package} = $self->{installation}->AddInstallable ($self);
		unless (defined $self->{installed_package}){
			$self->AddError ("Cannot persist package",$self->{installation});
			return undef;
		}
	}	
	
	my $submsglist = new SDB::Install::MsgLst();
	
	if (!$self->{installed_package}->SetValid (0, $submsglist)){
		$self->AddError(undef, $self->{installed_package});
		return undef;
	}
	$self->AddMessage(undef,$submsglist);
		
	if (defined $self->GetProgressHandler){
		my $files = $self->GetNumberOfFiles ();
		my $position = $files - $self->GetNumberOfNotExtractedFiles ();
		$self->GetProgressHandler()->InitProgress ($files, $position);
	}
			
	unless (defined $self->Extract ()){
		$self->AddError ('Extraction of package "'. $self->{data}->{name}.'" failed',$self);
		return undef;
	}

	unless (defined $self->Register ()){
		$self->AddError ('Registration of package "'. $self->{data}->{name}.'" failed',$self);
		return undef;
	}
	unless (defined $self->Postinstall ()){
		$self->AddError ('Postinstall step  of package "'. $self->{data}->{name}.'" failed',$self);
		return undef;
	}
	if (!$self->{installed_package}->SetValid (1, $submsglist)){
		$self->AddError(undef, $self->{installed_package});
		return undef;
	}
	$self->createPackageShortcuts();
	$self->AddMessage(undef,$submsglist);
	return 1;
}

sub Update{
	my ($self) = @_;
	
	unless (defined $self->{installed_package}){
		$self->AddError ('installed package isn\'t defined');
		return undef;
		
	}
	my $msg = $self->AddMessage("Uninstalling package " . $self->{installed_package}->GetName () . "...");
	$self->{installed_package}->setMsgLstContext ([$msg->getSubMsgLst ()]);
	$self->{installed_package}->UninstallForUpdate ();
	
	undef $self->{installed_package};
		
	
	unless (defined $self->Install ()){
		$self->AddError ('installation of package '. $self->GetName ().' failed',$self);
		return undef;
	}
	return 1;
}

sub IsUpdate ($){
	return $_[0]->{update};
}

sub canSkip ($){
	if (defined $_[0]->{refuseSkip}){
		return !$_[0]->{refuseSkip};
	}
	if ($_[0]->isServerPackage () && $_[0]->isGlobal ()){
		return 1;
	}
	return 0;
}

sub isSkipped ($){
	defined $_[0]->{skip} && $_[0]->{skip} ? 1 : 0;
}


sub GetFiles ($){
	my ($self) = @_;
	if (defined $self->{debugPackage} && !$self->{debugPackage}->{skip}){
		my %files = (%{$self->{data}->{files}},%{$self->{debugPackage}->{files}});
		return \%files;
	}
	return $self->{data}->{files};
}


sub GetNumberOfFiles{
	my ($self) = @_;
	my $files = scalar keys %{$self->{data}->{files}};
	if (defined $self->{debugPackage} && !$self->{debugPackage}->{skip}){
		$files += scalar keys %{$self->{debugPackage}->{files}};
	}
	return $files;
}


sub GetNumberOfNotExtractedFiles{
	my ($self) = @_;
	my $files;
	if (defined $self->{not_extracted} || (defined $self->{debugPackage} && 
		!$self->{debugPackage}->{skip} && defined $self->{debugPackage}->{not_extracated})){
		$files = (defined $self->{not_extracted}) ? 
					scalar keys (%{$self->{not_extracted}}) : 
					scalar keys (%{$self->{data}->{files}});
		
		if (defined $self->{debugPackage} && !$self->{debugPackage}->{skip}){
				$files += (defined $self->{debugPackage}->{not_extracted}) ? 
							scalar keys (%{$self->{debugPackage}->{not_extracted}}) :
							scalar keys (%{$self->{debugPackage}->{files}});
		}
	}
	else{
		$files = $self->GetNumberOfFiles();
	}
	return $files;
}



sub GetSize{
	my ($self) = @_;
	if (defined $self->{debugPackage} && !$self->{debugPackage}->{skip}){
		return ($self->{data}->{size} + $self->{debugPackage}->{size});	
	}
	return $self->{data}->{size};
}

sub Reset{
	my ($self) = @_;
	if (defined $self->{not_extracted}){
		$self->Rewind();
	}
	$self->{update} = 0;
	$self->{skip} = 0;
	undef $self->{installed_package};
	undef $self->{installation};
}


sub Rewind{
	my ($self) = @_;
	undef $self->{not_extracted};
	$self->rewind();
	$self->next(); # PACKAGEDATA
	$self->next(); # script
	$self->next(); # filelist
	if (defined $self->{debugPackage} && !$self->{debugPackage}->{skip} ){
		$self->{debugPackage}->Reset ();
	}
}


sub getSubstMacros{
	return $_[0]->{installation}->getSubstMacros ();
}



sub substituteMacrosInFile {
	my ($self,$filename) = @_;

	my $substMacros = $self->getSubstMacros ();

	if (!defined $substMacros){
		return 1;
	}

	if (!exists $self->{data}->{files}->{$filename}){
		$self->AddError ("Cannot substitute macros in file: file '$filename' is not registered");
		return undef;
	}

	my $file = $self->GetPath () . $path_separator . $filename;

	if (!-f $file){
		$self->AddError ("Cannot substitute macros in file: file '$file' not found");
		return undef;
	}

	my $backupdir = $self->getOriginFileBackupPath ();

	if (!-d $backupdir){
		my $cfg = {'mode' => 0750};
		if (!$isWin){
			$cfg->{gid} = $self->{installation}->getGID ();
			$cfg->{uid} = $self->{installation}->getUID ();
		}
		if (!defined makedir ($backupdir, $cfg)){
			$self->AddError ("Cannot substitute macros in file: Cannot create backup directory '$backupdir'", $cfg);
			return undef;
		}
	}

	my $parentDir = dirname ($file);
	my $restoreParentMode = enableWritePermissions ($parentDir, undef, $self->getMsgLst());

	my $orig = $backupdir . $path_separator . $self->{data}->{files}->{$filename}->{checksum};
	$self->{data}->{files}->{$filename}->{origin_checksum} = $self->{data}->{files}->{$filename}->{checksum};
	if (!rename ($file, $orig)){
		$self->AddError ("Cannot substitute macros in file: Cannot rename '$file' => '$orig': $!");
		chmod ($restoreParentMode, $parentDir) if (defined $restoreParentMode);
		return undef;
	}

	if (!open (OFH, $orig)){
		$self->AddError ("Cannot substitute macros in file: Cannot open '$orig': $!");
		chmod ($restoreParentMode, $parentDir) if (defined $restoreParentMode);
		return undef;
	}

	if (!open (NFH, '> '. $file)){
		$self->AddError ("Cannot substitute macros in file: Cannot create '$file': $!");
		close (OFH);
		chmod ($restoreParentMode, $parentDir) if (defined $restoreParentMode);
		return undef;
	}
	while (<OFH>){
		my ($newString, $pattern);
		foreach my $key (keys (%$substMacros)){
			$newString = $substMacros->{$key};
			$pattern = quotemeta ($key);
			if (s/$pattern/$newString/g){
				$self->AddMessage ("Replacing text \"$key\" with \"$newString\" in extracted file");
			}
		}
		print NFH $_;
	}
	close (NFH);
	close (OFH);
	chmod ($restoreParentMode, $parentDir) if (defined $restoreParentMode);
	$self->UpdateMd5Sum ($filename);
	return 1;
}


sub _extractDirectory{
    my ($self, $path, $dir) = @_;

    my $dirname = $dir . '/';
     my $absPath = $path .'/'. $dir;
    my $properties = $self->{not_extracted}->{$dirname};

    if (!defined $properties){
        return 1;
    }

    my $msg = $self->AddProgressMessage ('Extracting '.$dirname);
    $msg->getSubMsgLst ()->addMessage (
        sprintf ("dir info: mode = 0%o",
        $properties->{mode} & 07777));
    if (!-e $absPath){
        my $errlst = new SDB::Install::MsgLst ();
        my $cfg = {};
        $errlst->injectIntoConfigHash ($cfg);
        if (!defined makedir ($absPath, $cfg)){
            $self->setErrorMessage ("Cannot extract directory '$dir'", $errlst);
            return undef;
        }
    }
    if (!$isWin && $> == 0){
        if (!chown ($self->getUID (),$self->getGID, $absPath)){
            $self->setErrorMessage ("Cannot chown directory '$dir': $!");
            return undef;
        }
    }
    if (defined $properties->{mode}){
        if (!chmod ($properties->{mode}, $absPath)){
             $self->setErrorMessage ("Cannot chmod directory '$dir': $!");
            return undef;
        }
    }
    delete $self->{not_extracted}->{$dirname};
    return 1;
}

sub Extract {
	my ($self) = @_;
	my $originalUmask = $isWin ? undef : umask(022);
	my $result = $self->_extractPackage();

	if (defined($originalUmask)){
		umask($originalUmask);
	}
	return $result;
}

sub _extractPackage {
	my ($self) = @_;

	if (defined $self->{not_extracted} and !%{$self->{not_extracted}}){
		return 1;
	}

	my $set_setuidroot = 0;

    my $packageName = $self->GetName();
	my $persdata = $self->{data};
	my $files = $persdata->{files};

	unless (defined $self->{archive}){
		$self->AddError ('No archive defined');
		return undef;
	}
	
	unless (defined  $files){
		$self->AddError ('No files defined');
		return undef;
	}

	unless (defined $self->{installation}){
		$self->AddError ('No installation defined');
		return undef;
	}

	# set destination directory
	my $path =  $self->getPath ();
	
	my $archive_handle = $self->GetArchiveHandle ();


	if ($^O !~ /mswin/i){
		$archive_handle->SetOwner ($self->getUID (),
					$self->getGID ());
		$set_setuidroot = 1;
	}
	
	$archive_handle->SetDestDir ($path);

	
	if (!defined $self->{not_extracted}){
		my %not_extracted = %$files;
		$self->{not_extracted} = \%not_extracted; 	
	}
	
	my $file;
	my $msg;
	my $reset_owner_group;
	my $rc;
	my $filename;
	my $packageLstFile = $self->getPackageLSTFile();
	
	# extract version info from archived special files, if necessary.
	# TODO: do this at buildpackage time and pass versions per XML.
	my $specialFileVersions = $self->GetSpecialFileVersions();
	if(not defined $specialFileVersions) {
	    return undef;
	}
	if ($self->shouldVerifyChecksums() && !defined $packageLstFile) {
		$self->setErrorMessage("Failed to detect LST file for package '$packageName'.");
		return undef;
	}
	$self->AddMessage ("Extracting into directory $path");
	while (1){
		
		if (defined $self->{cur_file}){
			$file = $self->{cur_file};
		}else{
			$file = $self->next ();
			$self->{cur_file} = $file;
		}
		# error
		return undef unless (defined $file);
		
		# extract 'special files' i.e. files destined for locations outside global/local program dirs:
		if(exists $file->{name}) {
    		$filename = $file->{name};
    		if( (exists $self->{not_extracted}->{$filename}) && $self->{not_extracted}->{$filename}->{specialFile}) {
    		    my $osfilename = $filename;
    		    if($isWin) {
    		        $osfilename =~ s/\//\\/g;
    		    }
    		    else {
    		        $osfilename =~ s/\\/\//g;
    		    }
                $rc = $self->InstallSpecialFile($osfilename, $specialFileVersions->{$filename});
                if($rc == 0){
                    $self->AddError (join (': ',$archive_handle->GetErr));
                    return undef;
                }
                delete $self->{not_extracted}->{$filename};
                undef $self->{cur_file};
                next;
            }
		}
		
		if (defined $file->{mode} && $file->{mode} & 040000){
			# extract a directory
			if (!-d "$path/$file->{name}"){
				$archive_handle->ExtractFile ();
			}
			undef $self->{cur_file};
			next;
		}
				
		# eof reached
		last if not exists $file->{name};
		$filename = $file->{name};
		if (exists $self->{not_extracted}->{$filename}){
			my $dir = dirname ($filename);
			if ($dir){
				if (!defined $self->_extractDirectory ($path, $dir)){
					return undef;
				}
			}

			$msg = $self->AddProgressMessage ('Extracting '.$filename);
			$reset_owner_group = 0;
			if ($set_setuidroot && $self->{not_extracted}->{$filename}->{setRootAsOwner}){
				$reset_owner_group = 1;
				$archive_handle->SetOwner (0, $self->getGID);
				$self->{not_extracted}->{$filename}->{uid} = 0;
			}
			$rc = $archive_handle->ExtractFile ();
			unless (defined $rc and $rc == 0){
				$self->AddError (join (': ',$archive_handle->GetErr));
				return undef;
			}

            if ($reset_owner_group){
                $archive_handle->SetOwner ($self->getUID (),
                                $self->getGID);
            }
            $self->AddSubMessage(
                $msg, sprintf ("file info: mode = 0%o%s, size = %d bytes",
                     $file->{mode} & 07777,
                    $reset_owner_group ? 
                    ', file owner is root' : '',
                     $file->{size}));

            if ($self->shouldVerifyChecksums() && !$self->checkSHASignature($filename)) {
                return undef;
            }

            if ($self->{not_extracted}->{$filename}->{substituteMacros}){
                if (!defined $self->substituteMacrosInFile ($filename)){
                    return undef;
                }
            }

			delete $self->{not_extracted}->{$filename};
			undef $self->{cur_file};
		}
		else{
			$self->AddWarning ("Skipping file \"$filename\": not in file list");
			undef $self->{cur_file};
		}
	}

	# handle directories
	my ($absPath, $properties);
	foreach my $dirname (keys %{$self->{not_extracted}}){
		my ($dir) = ($dirname =~ /(.+)\/$/);
		if ($dir){
			if (!defined $self->_extractDirectory ($path, $dir)){
				return undef;
			}
		}
	}

	if (%{$self->{not_extracted}}){
		my $msg = $self->AddError ('Could not extract all files: EOF reached');
		my @submsgs;
		foreach my $file (keys (%{$self->{not_extracted}})){	
			push @submsgs, $self->GenMsg ('ERR', "file '$file' not extracted");
		}
		$msg->{submsg} = \@submsgs;
		return undef;
	}

	if (defined $self->{debugPackage} && !$self->{debugPackage}->{skip} ){
		if (!$isWin){
			$self->{debugPackage}->set_owner ($self->{installation}->getUID,
								$self->{installation}->getGID);		
		}
		$self->{debugPackage}->SetProgressHandler ($self->GetProgressHandler ());
		$self->{debugPackage}->Extract ($path);
		$self->AddMessage ('Extracting debug package',$self->{debugPackage});
		#
		# import files
		#
		my $dbg_files = $self->{debugPackage}->{files};
		foreach my $file (keys %$dbg_files){
			$dbg_files->{$file}->{dbg} = 1;
			$files->{$file} =  $dbg_files->{$file};
		}
		$self->{data}->{size} += $self->{debugPackage}->{size};
	}
	return 1;
}

sub checkSHASignature {
    my ($self, $filename) = @_;
    my $filePath = $self->GetPath () . $path_separator . $filename;
    my $rc = 1;

    if (-l $filePath) {
        $rc = $self->_checkSymlinkSHASignature($filename);
    } elsif (-f $filePath) {
        $rc = $self->_checkRegularFileSHASignature($filename);
    } else {
        $self->getMsgLst()->addError("Failed to verify checksum of '$filename': doesn't exist");
        $rc = 0;
    }

    if (!$rc) {
        $self->getErrMsgLst()->addError("Checksum verification for file '$filename' failed");
        if (-e $filePath && !unlink($filePath)) {
            $self->getErrMsgLst()->addError("Failed to delete file '$filename' with invalid checksum: $!");
        }
    }
    return $rc;
}

sub _checkSymlinkSHASignature {
    my ($self, $linkName) = @_;

    my $packageLstFile = $self->getPackageLSTFile();
    my $lstSignature = $packageLstFile->getFileHash($linkName);
    if (!defined $lstSignature) {
        my $lstFilePath = $packageLstFile->getOwnPath();
        $self->getMsgLst()->addError("Failed to get checksum of symlink '$linkName' from '$lstFilePath'");
        return 0;
    }
    my $filePath = $self->GetPath () . $path_separator . $linkName;
    my $linkValue = readlink($filePath);
    if (!defined $linkValue) {
        $self->getMsgLst()->addError("Failed to evaluate the value of symlink '$linkName' to verify its checksum");
        return 0;
    }
    my $linkValueSignature = sha256_hex($linkValue);

    if ($linkValueSignature ne $lstSignature) {
        $self->getMsgLst()->addError("The checksum of the symlink '$linkName' ($linkValueSignature) did not match the one in the LST file ($lstSignature)");
        return 0;
    }
    return 1;
}

sub _checkRegularFileSHASignature {
    my ($self, $filename) = @_;

    my $packageLstFile = $self->getPackageLSTFile();
    my $lstSignature = $packageLstFile->getFileHash($filename);
    if (!defined $lstSignature) {
        my $lstFilePath = $packageLstFile->getOwnPath();
        $self->getMsgLst()->addError("Failed to verify checksum of file '$filename' from '$lstFilePath'");
        return 0;
    }
    my $filePath = $self->GetPath () . $path_separator . $filename;
    my $sha = Digest::SHA->new(256);
    $sha->addfile($filePath);
    my $extractedFileSignature = $sha->hexdigest();

    if ($extractedFileSignature ne $lstSignature) {
        $self->getMsgLst()->addError("The checksum of file '$filename' ($extractedFileSignature) did not match the one in the LST file ($lstSignature)");
        return 0;
    }
    return 1;
}

sub _checkScriptFileSHASignature {
    my ($self) = @_;
    my $scriptContent = $self->{data}->{script} // "";
    my $packageLstFile = $self->getPackageLSTFile();

    if (!defined $packageLstFile) {
        $self->getErrMsgLst()->addError("Failed to check $script signature: package has no LST file");
        return 0;
    }
    my $lstSignature = $packageLstFile->getFileHash($script);
    if (!defined $lstSignature) {
        my $lstFilePath = $packageLstFile->getOwnPath();
        $self->getErrMsgLst()->addError("Failed to get checksum of file '$script' from '$lstFilePath'");
        return 0;
    }
    my $scriptSignature = sha256_hex($scriptContent);

    if ($scriptSignature ne $lstSignature) {
        $self->getErrMsgLst()->addError("The checksum of file '$script' ($scriptSignature) did not match the one in the LST file ($lstSignature)");
        return 0;
    }
    return 1;
}

sub UpdateMd5Sum{
    my (
        $self,
        $file
    ) = @_;
    my $absPath = $self->getPath () . $path_separator . $file;

    if (!-f $absPath){
        $self->AddError ("Cannot update checksum of file  '$absPath': $!");
        return undef;
    }

    $file =~ s/\\/\//g;
    $file =~ s/\/{2,}/\//g;

    if (!exists $self->{data}->{files}->{$file}){
        $self->{data}->{files}->{$file} = {};
    }
    my @statbuf = stat (_);
    my $fileProperties = $self->{data}->{files}->{$file};
    $fileProperties->{checksum} = MD5Sum ($absPath);
    $fileProperties->{mode} = $statbuf[2] & 07777;
    $fileProperties->{size} = $statbuf[7];
    $fileProperties->{mtime} = $statbuf[9];
    return 1;
}

#----------------------------------------------------------------------------


#
#	following methods can/should
#	be overloaded by package scripts
#

sub Preinstall (){
        return 1;
}

sub Postprepare (){
	return 1;
}

sub Postinstall (){
	return 1;
}

sub Register (){
        return 1;
}

sub InstallSpecialFile (){
    return 1;
}

# returns undef on error, otherwise (possibly empty) hash containing
# names of special files as keys, and version strings as values 
# intended use: system libs
#
sub GetSpecialFileVersions() {
    my $retval = {}; 
    return $retval;
}

sub getInstallProgressMessage() {
        return sprintf ('%s package \'%s\'', $_[0]->IsUpdate ? 'Updating' : 'Installing', $_[0]->{'data'}-> {'name'});
}

sub getUID{
    return $_[0]->{installation}->getUID();
}

sub getGID{
    return $_[0]->{installation}->getGID();
}




1;

