#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: Install Registry 

package SDB::Install::Registry;

use SAPDB::Install::DataDumper;
use SAPDB::Install::FileLock;
use SDB::Install::BaseLegacy;
use SAPDB::Install::DataDumper;
use SDB::Install::Package::Installed;
use SDB::Install::System (normalizePath);
use SDB::Install::SysVars;
use SDB::Install::Tools qw (callstack);
use strict;


our @ISA = qw (SDB::Install::BaseLegacy);

# static properties

our $filename = 'INSTREG';
our $legacyID = 'MaxDB InstallRegistry';
our $ID = 'NewDB InstallRegistry';
our $version = '1.1';
our $max_size = 0x400000; # 4096 kb
our $locktab = {};


# methods

sub new{
	my $self = shift->SUPER::new ();
	
	($self->{path}, $self->{open_mode}, $self->{'nolock'},$self->{'uid'},$self->{'gid'})  = @_;
	$self->{fd} = undef;
	$self->{flock} = undef;
	$self->{fpos} = 0;
	
	unless (-d $self->{path}){
		$self->AddError ('Cannot open InstallRegistry file:'.
				' directory "' . $self->{path} .
				'" doesn\'t exist');
		return $self;	
	}
	$self->Open ();	
	return $self;
}

sub InitNewRegistry{
	my ($self) = @_;
	$self->{data} = {'ID' => $legacyID,
			'VERSION' => $version,
			'PACKAGES' => {}
		    };
	$self->{fresh} = 1;
	$self->Write();
}

sub Read{
	my ($self) = @_;
	if (!defined $self->{fd} || !defined fileno ($self->{fd})){
		$self->AddError ('InstallRegistry file is not open');
		return undef;
	}
	
	if ($self->{fpos}){
		seek ($self->{fd},0,0);
		$self->{fpos} = 0;
	}
	($self->{data},$self->{fpos}) = readDump ($self->{fd});
	
	unless ($self->{data}->{ID} eq $legacyID || $self->{data}->{ID} eq $ID){
		$self->AddError ('File "'.$self->{path}.'/'.$filename.
					'" is no valid InstallRegistry');
		$self->{data} = undef;
		return undef;
	}

	unless ($self->{data}->{VERSION} eq $version){
		$self->AddError ('File "'.$self->{path}.'/'.$filename.'
				 has incompatible InstallRegistry version');
		$self->{data} = undef;
		return undef;
	}	
	return $self->{fpos};
}

sub Write{
	my ($self,$truncate) = @_;
	if (!defined $self->{fd} || !defined fileno ($self->{fd})){
		$self->AddError ('InstallRegistry file is not open');
		return undef;
	}	

	if ($self->{fpos}){
		seek ($self->{fd},0,0);
		$self->{fpos} = 0;
	}	
	
	unless ($self->{open_mode}){
		$self->AddError ('InstallRegistry is in read only mode');
		return undef;
	}
	
	my $rc = dumpit ($self->{data},$self->{fd},$truncate);
	
	unless ($rc){
		$self->AddError ('Cannot write InstallRegistry file');
		return undef;
	}
	$self->AddMessage ("Wrote $rc bytes");
	
	if (exists $self->{EOR} and $self->{EOR} > $rc){
		my $len = $self->{EOR} - $rc;
		print {$self->{fd}} (0 x $len);
		$self->{EOR}  = $rc;	
	}
	$self->{fpos} = $rc;	
	return $rc;
}

sub ReserveSpace{
	my ($self) = @_;
	
	if (!defined $self->{fd} || !defined fileno ($self->{fd})){
		$self->AddError ('InstallRegistry isn\'t open');
		return undef;
	}
	
	unless ($self->{open_mode}){
		$self->AddError ('InstallRegistry is in read only mode');
		return undef;
	
	}
	
	my $kb = $max_size / 0x400; 
	
	
	if ($self->{fpos} >= $max_size){
		$self->AddWarning("InstallRegistry file >= $kb kb");
		return undef;
	}
	
	seek($self->{fd},0,1); # reset eof error
	$self->{EOR} = $self->{fpos};
	
	my $cur_pos = tell ($self->{fd});
	my $diff = $cur_pos - $self->{fpos};
	
	if ($diff > 0){
		if ($diff >= $max_size){
			$self->AddMessage("InstallRegistry: $kb kb disk space already reserved");	
			return 1;
		}
	}
	
	if ($self->{fpos} < $max_size){
		my $appendbytes = $max_size - $cur_pos;
		print {$self->{fd}} '0' x $appendbytes;
		my $len = tell ($self->{fd});
		if ($len != $max_size){
			$self->AddError("Can\'t reserve whole needed space ($kb kb), $len bytes reserved: $!");
			return undef;
		}
		$self->AddMessage("InstallRegistry: reserved $kb kb disk space");	
	}
	else{
		$self->AddWarning("InstallRegistry file is larger than $kb kb");
	}
	return 1;
}


sub AddPackage{
	my ($self,$package) = @_;
	
	if (exists $self->{data}->{PACKAGES}->{$package->{id}}){
		$self->AddError ('Package already extists');
		return undef;
	}
	my $pack = new SDB::Install::Package::Installed ($package);
	$self->{data}->{PACKAGES}->{$package->{id}} = $pack->{data};
	return $pack;
}


sub GetPackage{
	my ($self,$id) = @_;
	if (exists $self->{data}->{PACKAGES}->{$id}){
		return new SDB::Install::Package::Installed ($id, $self->{data}->{PACKAGES}->{$id});
	}
	$self->AddError ('Package "'.$id.'" doesn\'t exist');
	return undef;
}


sub GetPackageByName{
	my ($self,$name) = @_;
	foreach my $id (keys (%{$self->{data}->{PACKAGES}})){
		if ($self->{data}->{PACKAGES}->{$id}->{name} eq $name){
			return new SDB::Install::Package::Installed ($id,$self->{data}->{PACKAGES}->{$id});
		}
	}
	$self->AddError ('Package with name "'.$name.'" doesn\'t exist');
	return undef;	
}

sub GetPackageIds{
	my ($self) = @_;
	return keys (%{$self->{data}->{PACKAGES}});
}


sub Open{
	my ($self) = @_;
	my $file = $self->{path} . ($isWin ? '\\' : '/') . $filename;
	
	my @statbuf = stat ($file);
	
	if ($isWin){
		$self->{fid} = normalizePath ($file);
	}
	else{
		$self->{statbuf} = \@statbuf;
	}
	
	if (@statbuf){
		$self->AddMessage ("InstallRegistry file already exists");
		my $mode = $self->{open_mode} ? '+<' : '';
		unless (open ($self->{fd},$mode.$file)){
			$self->AddError ('Cannot open InstallRegistry file "'.
					$file.'": '.$!);
			return undef;
		}
	}
	else{
		unless ($self->{open_mode}){
			$self->AddError ('Cannot open InstallRegistry file "'.
					$file.'": '.$!);
			return undef;
		}
		$self->AddMessage ("Create new InstallRegistry file");
		
		if ($isApple){
			chmod (0750, $self->{path});
		}
		unless (open ($self->{fd}, '>'.$file)){
			$self->AddError ('Cannot open InstallRegistry file "'.
					$file.'": '.$!);
			if ($isApple){
				chmod (0550, $self->{path});
			}
			return undef;
		}
		
		if ($isApple){
			chmod (0550, $self->{path});
		}
        if (defined $self->{uid} && defined $self->{gid}){
            chown ($self->{uid},$self->{gid}, $file);
        }
		@statbuf = stat ($file);
		if (!$isWin){
			$self->{statbuf} = \@statbuf;
		}
		$self->{fresh} = 1;
	}


	if (!$self->{nolock} || $self->{open_mode}){
		if (!$self->Lock ()){
			return undef;
		}
	}

	{
		#
		# switch autoflush on
		#
		
		my $save = select ($self->{fd});
		$| = 1;
		select ($save);
	}

	binmode ($self->{fd});
	
	if ($self->{fresh}){
		$self->InitNewRegistry ();
	}
	else{
		if (!defined $self->Read ()){
			$self->Close();
			return undef;
		}
	}
	
	if ($self->{open_mode}){
		if (!$isWin){
			if ( ($statbuf[2] & 07777) != 0644){
				chmod (0644, $file);
			}	
		}
		$self->ReserveSpace();
	}
	return 1;
}

sub Close{
	my ($self) = @_;

	$self->Unlock ();
	if (defined $self->{fd} && defined fileno ($self->{fd})){
		close ($self->{fd});
		$self->AddMessage ('InstallRegistry closed');
		$self->{fd}  = undef;
	}

	return 1;
}

sub Lock{
	my ($self) = @_;
	if ($isWin){
		
		my $fid = $self->{fid};
		
		if (exists $locktab->{$fid}){
			if ($self->{open_mode}){
				# exclusive
				my $lock_owner;
				if (defined $locktab->{$fid}->[1]){
					$lock_owner =  $locktab->{$fid}->[1];
				}
				else{
					$lock_owner = join (', ', keys %{$locktab->{$fid}->[0]});
				}
				$self->AddError ("InstallRegistry is locked by another installation object ($lock_owner)");
				$self->Close();
				return 0;
			}
			else{
				#shared
				if (exists $locktab->{$fid}->[1]){
					my $lock_owner = $locktab->{$fid}->[1];
					$self->AddError ("InstallRegistry is locked by another installation object ($lock_owner)");
					$self->Close();
					return 0;
				}
			}
		}
	
		if ($self->{open_mode}){
			$locktab->{$fid}->[1] = "$self";
		}
		else{
			$locktab->{$fid}->[0]->{$self} = 1;
		}
		$self->{have_lock} = 1;
	}
	else{
	
		my $statbuf = $self->{statbuf};
		if (exists $locktab->{$statbuf->[0]} and exists $locktab->{$statbuf->[0]}->{$statbuf->[1]}){
			if ($self->{open_mode}){
				# exclusive
				my $lock_owner;
				if (defined $locktab->{$statbuf->[0]}->{$statbuf->[1]}->[1]){
					$lock_owner = $locktab->{$statbuf->[0]}->{$statbuf->[1]}->[1];
				}
				else{
					$lock_owner = join (', ', keys %{$locktab->{$statbuf->[0]}->{$statbuf->[1]}->[0]});
				}
				$self->AddError ("InstallRegistry is locked by another installation object ($lock_owner)");
				$self->Close();
				return 0;
			}
			else{
				#shared
				if (exists $locktab->{$statbuf->[0]}->{$statbuf->[1]}->[1]){
					my $lock_owner = $locktab->{$statbuf->[0]}->{$statbuf->[1]}->[1];
					$self->AddError ("InstallRegistry is locked by another installation object ($lock_owner)");
					$self->Close();
					return 0;
				}
			}
		}
		
		if ($self->{open_mode}){
			$locktab->{$statbuf->[0]}->{$statbuf->[1]}->[1] = "$self";
		}
		else{
			$locktab->{$statbuf->[0]}->{$statbuf->[1]}->[0]->{$self} = 1;
		}
		$self->{have_lock} = 1;
	}	
	
	my $flock = SAPDB::Install::FileLock::new (fileno ($self->{fd}),
							$self->{open_mode});
	my $rc = $flock->Test;
	
	if($rc == -1){
		$self->AddError('Can\'t get flock status of InstallRegistry: '.
				$flock->GetError);
		$self->Close ();
		return 0;
	}
	elsif($rc > 0){
		my $pid = '';
		unless ($isWin){
			$pid = " (pid=$rc)";
		}
		$self->AddError ('InstallRegistry is locked by another '.
				  'process' . $pid);
		$self->Close ();
		return 0;
	}
	unless ($flock->Lock () == 0){
		my $msglst = new SDB::Install::MsgLst();
		callstack ($msglst);
		$self->AddError ('Can\'t lock InstallRegistry: '.
				$flock->GetError () . " MODE = $self->{open_mode}", $msglst);

		$self->Close ();
		return 0;
	}

	$self->{flock} = $flock;
	$self->AddMessage ('InstallRegistry successfully locked');
	return 1;
}

sub Unlock{
	my ($self) = @_;

	my $rc = 1;

	if (defined $self->{flock}){
		if ($self->{flock}->Unlock () != 0){
			$rc = 0;
		}
		$self->{flock} = undef;
	}

	if ($self->{have_lock}){
	
		if ($isWin){
			my $fid = $self->{fid};
			
			if ($self->{open_mode}){
				# exclusive
				delete $locktab->{$fid};
			}
			else{
				# shared
				delete $locktab->{$fid}->[0]->{$self};
				if (!%{$locktab->{$fid}->[0]}){
					delete $locktab->{$fid};
				}
			}
		}
		else{
			my $statbuf = $self->{statbuf};
	
			if ($self->{open_mode}){
				# exclusive
				delete $locktab->{$statbuf->[0]}->{$statbuf->[1]};
			}
			else{
				# shared
				delete $locktab->{$statbuf->[0]}->{$statbuf->[1]}->[0]->{$self};
				if (!%{$locktab->{$statbuf->[0]}->{$statbuf->[1]}->[0]}){
					delete $locktab->{$statbuf->[0]}->{$statbuf->[1]};
				}
			}
	
			if (!%{$locktab->{$statbuf->[0]}}){
				delete $locktab->{$statbuf->[0]};
			}
		}
		$self->{have_lock} = 0;	
	}		

	if ($rc){
		$self->AddMessage ('InstallRegistry successfully unlocked');
	}
	return $rc;
}


sub DeletePackage{
	my ($self,$id) = @_;
	delete $self->{data}->{PACKAGES}->{$id};	
	return 1;
}


sub DESTROY{
	my ($self) = @_;
	if ($self->{open_mode}){
		$self->Write (1);
	}
	$self->Close ();
}


1;
