package SDB::Install::SAPInstance::NameserverUtility;
use parent qw (SDB::Install::Base);

use strict;
use warnings;
use SDB::Common::Utils qw(trim);
use SDB::Install::MsgLst;
use SDB::Install::Globals qw(getServiceNameToComponentDirNameMap);

sub new{
    my ($class,$instance) = @_;
    my $self = {};
    bless($self,$class);
    $self->setSAPInstance($instance);
    return $self,
}

sub setSAPInstance{
    my($self,$hdbInstance) = @_;
    $self->{_hdbInstance} = $hdbInstance;
}

sub run{
    my ($self,$args,$msglst, $ref_output_buffer, $no_user_switch, $ref_input_buffer) = @_;
    my $instance = $self->{_hdbInstance};

    #
    # check whether trace directory exists
    # hdbnsutil will hang (SIGSTOP), when it cannot write its crash dump
    # bug#156863
    #

    my $traceDir = $instance->get_hostNameTraceDir ();
    if (defined $traceDir){
        if (!-e $traceDir){
            $instance->setErrorMessage ("HANA trace directory '$traceDir' doesn't exist.");
            return undef;
        }
        if (!-d _ ){
            $instance->setErrorMessage ("HANA trace directory '$traceDir' exists, but it's not a directory.");
            return undef;
        }
    }

    # Disable Inter-Process Memory Management (IPMM) for hdbnsutil on Windows
    # to prevent hdbnsutil from crashing
    local %ENV = %ENV;
    $ENV{'HDB_IPMM'} = 'd';
    require SDB::Install::OutputHandler::HdbnsutilOutHndlr;
    return $instance->runUtilityInHDBEnv(
        'hdbnsutil',
        $args,
        $msglst,
        $ref_output_buffer,
        $ref_input_buffer,
        undef,
        new SDB::Install::OutputHandler::HdbnsutilOutHndlr(undef, '  hdbnsutil: '),
        $no_user_switch
    );
}

sub checkCanRemoveHost{
    my ($self, $host, $configuration) = @_;
    my $msglst = SDB::Install::MsgLst->new();
    my $args   = ['-removeHost', defined $host ? $host : (),"--test","--sapcontrol=1","--ignoreActiveState"];
    my $rc = $self->run($args, $msglst);
    $self->parseOutput($msglst,$host);
    return 1 if (!$rc && $self->shouldIgnoreErrors($msglst, $configuration));
    return $rc;
}

sub checkCanRemoveRole {
    my ($self, $host, $role) = @_;
    my $msglst = SDB::Install::MsgLst->new();
    my $args = [ "-removeHostRole", defined($host) ? $host : (), "--role=$role", "--sapcontrol=1", "--test" ];
    my $rc = $self->run($args, $msglst);
    $self->parseOutput($msglst, $host);
    return $rc;
}

sub parseOutput{
    my($self, $msglst,$host) = @_;
    my $hostStr = defined $host ? "$host: " : '';
    if(!defined $msglst){
        return;
    }
    my $messages = $msglst->getMessages();
    foreach my $message(@$messages){
        my $msgStr = $message->getMsgString();
        my $out = defined $msgStr ? $$msgStr : '';
        my @lines = split(qr/\n/,$out);
        for my $line(@lines){
            if($line =~ qr/.*SAPCONTROL-OK:.*/){
                $line =~ s/.*SAPCONTROL-OK:(\s*[0-9]*:|\s*:|\s*)\s*;*//g ;
                $self->getMsgLst()->addMessage( $hostStr.trim($line));
            }
            if($line =~ qr/.*SAPCONTROL-ERROR:.*/){
                $line =~ s/.*SAPCONTROL-ERROR:(\s*[0-9]*:|\s*:|\s*)\s*;*//g ;
                $self->getErrMsgLst()->addError( $hostStr.trim($line));
            }
        }
    }
}

# Ignore errors for initialized services of
# components that support auto initialize flag
sub shouldIgnoreErrors{
    my($self, $msglst, $configuration) = @_;
    return 0 if(!defined $msglst);

    my $isSystemInCompatibilityMode = $configuration->isSystemInCompatibilityMode();
    my $autoInitializeServices = $configuration->getValue('AutoInitializeServices');
    return 0 if (!$autoInitializeServices || !$isSystemInCompatibilityMode);

    my $serviceNameToComponentDirNameMap = getServiceNameToComponentDirNameMap();
    my $instance = $self->{_hdbInstance};

    my $shouldIgnoreErrors = 1;
    my $messages = $msglst->getMessages();
    foreach my $message(@$messages){
        my $msgStr = $message->getMsgString();
        my $out = defined $msgStr ? $$msgStr : '';
        my @lines = split(qr/\n/,$out);
        for my $line(@lines){
            if($line =~ qr/.*SAPCONTROL-ERROR:.*/){
                if ($line =~ /.*Perform.*ALTER SYSTEM UNINITIALIZE SERVICE\s+'(\w+)'/) {
                    my $serviceName = $1;
                    my $componentDir = $serviceNameToComponentDirNameMap->{$serviceName};
                    $shouldIgnoreErrors = 0 if (!defined $componentDir); #unknown component
                    $shouldIgnoreErrors = 0 if (!$instance->shouldAutoInitializeService($configuration, $componentDir));
                    next;
                }
                $shouldIgnoreErrors = 0;
            }
        }
    }
    return $shouldIgnoreErrors;
}

sub isSystemInCompatibilityMode{
    my ($self,$msglst) = @_;
    my $out = $self->printTopology("/databases/1/singleTenant", $msglst);
    return ($out && $out =~ qr/.*singleTenant=yes.*/) ? 1 : 0;
}

sub getXsControllerHost {
    my ($self) = @_;
    my $out = $self->printTopology("/host/*/xscontroller") // '';
    my ($topo) = $out =~ /\[(.+)\]/s;
    my @xsControllerHosts = grep { $_ && $_ ne "''" } map { trim($_) } split /\n/, $topo;
    return @xsControllerHosts ? $xsControllerHosts[0] : undef;
}

sub printTopology {
    my ($self, $filter, $msglst) = @_;
    my $args = ['-printTopology'];
    if ($filter) {
        push @{$args}, "--filter=$filter";
    }

    my $rc = $self->run($args, $msglst // SDB::Install::MsgLst->new(), \my $output);
    return $rc ? $output : undef;
}

sub removeHostRole{
    my($self,$host,$role,$force) = @_;
    my $msg    = $self->getMsgLst()->addMessage("Remove host role '$role' on ".(defined $host ? "host $host" : "localhost") );
    my $msglst = $msg->getSubMsgLst ();
    my $saveCtx = $self->getMsgLstContext();
    $self->setMsgLstContext([$msglst]);

    my $output = SDB::Install::MsgLst->new();

    my $args   = ['-removeHostRole', defined $host ? $host : (), "--role=$role", '--sapcontrol=1'];
    if ($force){
        push @$args, '--force';
    }
    my $rc = $self->run($args, $output);
    $self->parseOutput($output, $host);

    if (!$rc){
        $self->setErrorMessage ("Cannot remove host role '$role'", $msglst);
    }
    $self->setMsgLstContext($saveCtx);
    return $rc;
}

sub convertToMultiDB{
    my($self,$systemDBSQLPassword) = @_;

    my $mode = $self->getInstance()->getReplicationMode($self->getMsgLstContext());
    my $args;
    if(!defined $mode || $mode =~ /none/){
        $args   = [ '-convertToMultiDB', '--withBackupCatalog' , '--withLicense', '--phaseConvert' ];

        if (!$self->run($args, $self->getMsgLst())){
            $self->setErrorMessage ("Cannot convert the system to multitenant database containers", $self->getMsgLst());
            return undef;
        }
        $args   = [ '-convertToMultiDB', '--withBackupCatalog' , '--withLicense', '--phaseReinit' ];
        if (!$self->run($args, $self->getMsgLst())){
            $self->setErrorMessage ("Cannot convert the system to multitenant database containers", $self->getMsgLst());
            return undef;
        }
    } else {
        $args = [ '-convertToMultiDB', '--systemReplication' , '--phaseConvert' ];
        if (!$self->run($args, $self->getMsgLst(),undef, undef, [$systemDBSQLPassword])){
            $self->setErrorMessage ("Cannot convert the system to multitenant database containers", $self->getMsgLst());
            return undef;
        }

    }
    #reload cached multiDb parameter
    $self->getInstance()->isMultiDb(1);
    return 1;
}

sub migrateSecureStore {
    my ($self, $target) = @_;
    my $args = [ "-migrateSecureStore", "--target=$target" ];
    return $self->run($args, $self->getMsgLst());
}

sub createOfflineTopology {
    my ($self, $targetFile) = @_;
    my $args = ['-exporttopology', '-createOfflineTopology', $targetFile];
    return $self->run($args, $self->getMsgLst());
}

sub getInstance{
    my $self = shift;
    return $self->{_hdbInstance};
}
1;
