package SDB::Install::SAPSystemContainerHelper::InitPersistenceContainerHelper;

use strict;
use SDB::Install::SAPSystemUtilities;
use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::IOHelper;
use SDB::Install::SysInfo;
use SDB::Install::OSConfig;
use SDB::Install::SAPInstance::TrexInstance;
use SDB::Install::SAPSystem;

our @ISA = qw (SDB::Install::Base SDB::Install::SAPSystemUtilities);


#-------------------------------------------------------------------------------
# renames a HANA host in a docker container
#
# Parameters: $sapSys   SDB::Install::SAPSystem
#             $instance SDB::Install::SAPInstance::TrexInstance
#             $oldHost  string (original host name)
#             $newHost  string (new host name)
#             $sysinfo  SDB::Install::SysInfo
#
# Returns int retCode

sub _renameHost{
    my ($self, $sapSys, $instance, $oldHost, $newHost, $sysinfo) = @_;
    my $msglst = $self->getMsgLst ();
    my $msg = $msglst->addProgressMessage ("Setting host name ($newHost)...");
    $instance->setMsgLstContext ([$msg->getSubMsgLst()]);

    #
    # hwinfo file
    #

    $instance->removeHWInfoFile ($oldHost);
    $instance->{_host} = $newHost;
    $instance->createHWInfoFile ($sysinfo, $newHost);

    #
    # instance profile
    #

    my $oldProfileName = $instance->get_profileName(undef, undef, $oldHost);
    my $newProfileName = $instance->get_profileName(undef, undef, $newHost);

    my $ioHelper = new SDB::Install::IOHelper();

    if (!$ioHelper->rename ($oldProfileName, $newProfileName)){
        $self->setErrorMessage ("Cannot rename instance profile ($oldProfileName => $newProfileName): ". $ioHelper->strerror ());
        return undef;
    }
    return 1;
}

#-------------------------------------------------------------------------------
# initialize a docker container
# This method is called from SDB::Install::App::Console::HdbContainer
#
# Parameters: $app    SDB::Install::App::Console
#             $sapSys SDB::Install::SAPSystem
#             $sysinfo  SDB::Install::SysInfo (currenly not required due to unsupported host rename)
#
# Returns int retCode


sub performContainerAction {
    my ($self, $instconfig, $sapSys, $sysinfo) = @_;

    my $hostname      = $instconfig->getValue('HostName'); # <<<<< HOST RENAME NOT SUPPORTED YET
    my $systemPassword= $instconfig->getValue('SystemUserPassword');
    my $instance      = $instconfig->getOwnInstance();
    my $msglst        = $self->getMsgLst();
    my $msg;
    my $oldHost       = $instance->get_host();
    if (defined $hostname && lc($hostname) ne lc($oldHost)){
        if (!defined $self->_renameHost ($sapSys, $instance, $oldHost, $hostname, $sysinfo)){
            return undef;
        }
    }

    $instance->setMsgLstContext([$msglst]);
    $sapSys->SelectAllPackages();
    $instance->killtrex($sapSys);

    $msg = $msglst->addProgressMessage ("Installing initial topology");
    $msg->setEndTag('Install initial topology');
    my $submsglst = $msg->getSubMsgLst ();
    my $initTopologyArgs = [];

    push @$initTopologyArgs, '--no_initial_tenant';

    my $do_not_change_password = $systemPassword eq 'manager';
    my $systemUserPWInputBuffer = undef;
    if(!$do_not_change_password) {
        push @$initTopologyArgs, '--set_user_system_pw';
        $systemUserPWInputBuffer = ["$systemPassword", "$systemPassword"];
    }
    if (!defined $instance->installTopology ($initTopologyArgs , $submsglst, $systemUserPWInputBuffer)){
        $self->setErrorMessage ("Cannot create initial topology", $submsglst);
        return undef;
    }

    return 1;
}



1;
