package SDB::Install::SAPSystemXS2;
use parent qw(SDB::Install::ServiceManager);
use strict;

use SDB::Install::SAPSystem;
use SDB::Install::SAPControl;
use SDB::Install::LayeredConfig;
use SDB::Install::SAPInstance::TrexInstance;
use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::Globals qw ( $gHostRoleXS2Standby $gHostRoleXS2Worker $gProductNameXS2 getXS2ServiceNames );


sub new {
    my $class = shift;
    my $self  = {};
    bless ($self, $class);
    return $self;
}


sub stopXS2 {
    my (
        $self,
        $trexInstance,
        $instconfig
    ) = @_;
    my $start = 0;
    return $self->_stopOrStartXS2($trexInstance, $instconfig, $start);
}

sub startXS2 {
    my (
        $self,
        $trexInstance,
        $instconfig
    ) = @_;
    my $start = 1;
    return $self->_stopOrStartXS2($trexInstance, $instconfig, $start);
}

sub _getXSHostsInLandscape {
    my ($self, $trexInstance, $reverseOrder) = @_;
    my $allHosts = $trexInstance->get_allhosts() || [];
    my $xsHosts = [];

    for my $hostName (@{$allHosts}){
        my $hostRoles = $trexInstance->getHostRolesByIniFile($hostName) || [];
        my $hostHasXSRole = grep {$_ eq $gHostRoleXS2Standby || $_ eq $gHostRoleXS2Worker } @{$hostRoles};

        push(@{$xsHosts}, $hostName) if($hostHasXSRole);
    }
    return $reverseOrder ? [ reverse(@{$xsHosts}) ] : $xsHosts;
}

sub _stopOrStartXS2 {
    my ($self, $trexInstance, $instconfig, $start) = @_;
    if (!defined $trexInstance) {
        my $action = ($start) ? 'start' : 'stop';
        $self->getErrMsgLst()->addMessage("Could not initiate instance object. Failed to reconfigure system and $action services.");
        return undef;
    }
    my $serviceNames = getXS2ServiceNames();
    my $reverseOrder = $start ? 0 : 1;
    my @hosts = @{$self->_getXSHostsInLandscape($trexInstance, $reverseOrder)};

    foreach my $host (@hosts) {
        my $daemon_ini = $self->_getHostDaemonIni($instconfig, $host);

        my $relevantSections = $self->_getRelevantDaemonIniSections($daemon_ini, $serviceNames);
        next if (! scalar(@$relevantSections));

        $self->getMsgLst()->addMessage ("Adjusting daemon.ini for host $host");
        return undef if (!$self->_manageDaemonIniEntries($daemon_ini, $relevantSections, $start));

    }
    return undef if (!$self->_reconfigureSystem($trexInstance));
    return undef if ($start && !$self->_startNecessaryXsHosts($trexInstance, $instconfig));
    return $self->startStopViaSapcontrol($instconfig, \@hosts, $start, $serviceNames);
}

sub _startNecessaryXsHosts {
    my ($self, $trexInstance, $instconfig) = @_;
    my @hosts = @{$self->_getXSHostsInLandscape($trexInstance)};
    my $hostsToStart = $self->_getXsWorkerHostsToStart($trexInstance, \@hosts);
    my $userName = $trexInstance->get_user()->getSidAdmName();
    my $password = $instconfig->getValue('Password');
    my $useHttps = $instconfig->isUseHttps();
    my $ssoCert = $instconfig->getValue('SSOCertificate');
    my $saveContext = $trexInstance->setMsgLstContext($self->getMsgLstContext());
    my $rc = $trexInstance->startHosts($userName, $password, \@hosts, undef, $useHttps, $ssoCert);
    $trexInstance->setMsgLstContext($saveContext);
    return $rc;
}

sub _manageDaemonIniEntries {
    my ($self, $daemon_ini, $serviceNames, $startFlag) = @_;
    my $hasAlteredDaemonIni = 0;
    foreach my $service (@{$serviceNames}) {
        my $instancesValue = $self->_getDaemonIniValue($daemon_ini, $service, 'instances', CFG_LAYER_HOST);
        # Check if service is stopped by us
        my $wasStoppedByUs = (1 == $self->_getDaemonIniValue($daemon_ini, $service, 'hplm_stopped', CFG_LAYER_HOST));
        if($startFlag) {
            if(defined $instancesValue && $instancesValue == 0 && $wasStoppedByUs) {
                $hasAlteredDaemonIni = 1;
                $daemon_ini->setValue(CFG_LAYER_HOST, $service, 'instances', 1);
                $daemon_ini->removeKey(CFG_LAYER_HOST, $service, 'disabled');
                $daemon_ini->removeKey(CFG_LAYER_HOST, $service, 'hplm_stopped'); # Remove mark that service is stopped by us
            }
        }
        else {
            if(defined $instancesValue && $instancesValue > 0) {
                $hasAlteredDaemonIni = 1;
                $daemon_ini->setValue(CFG_LAYER_HOST, $service, 'hplm_stopped', 1); # Mark that service is stopped by us
                $daemon_ini->setValue(CFG_LAYER_HOST, $service, 'instances', 0);
            }
        }
    }
    if($hasAlteredDaemonIni && !defined $daemon_ini->write()) {
        $self->getErrMsgLst()->addError (undef, $daemon_ini);
        return undef;
    }
    $self->getMsgLst()->appendMsgLst($daemon_ini->getMsgLst());
    return 1;
}

sub _getXsWorkerHostsToStart {
    my ($self, $instance, $hosts) = @_;
    my @result = ();
    for my $host (@$hosts) {
        my $hostRoles = $instance->getHostRolesByIniFile($host);
        push(@result, $host) if (grep {$_ eq 'xs_worker'} @$hostRoles);
    }
    return \@result
}

1;
