package SDB::Install::SSH::Libssh2Session;

use strict;
use warnings;

use Fcntl qw(:DEFAULT :mode);
use File::Spec;
use File::Basename qw (dirname);
use IO::File;
use parent qw(SDB::Install::SSH::BaseSession);

our $can_exit_signal;


sub new {
    my $self = shift->SUPER::new();
    $can_exit_signal = shift;
    $self->{session} = $self->_createSession();
    return $self;
}

sub connect {
    my ($self, $hostname) = @_;
    $self->setHostname($hostname);
    return $self->{session}->connect($hostname);
}

sub sessionError {
    my ($self) = @_;
    if (!defined $self->{session}) {
        return undef;
    }
    my $msg = join(', ', grep { $_ ne '' } $self->{session}->error());
    return $msg;
}

sub sftpError {
    my ($self) = @_;
    if (!defined $self->getSftp()) {
        return undef;
    }
    my ($code, $string, $desc) = $self->getSftp()->error();
    $desc = ($desc) ? " - $desc" : '';
    $string .= $desc if(defined($string));
    return $string;
}

sub sftpErrorRaw {
    my ($self) = @_;
    if (!defined $self->getSftp()) {
        return undef;
    }
    return $self->getSftp()->error();
}

sub channelError {
    my ($self) = @_;
    if (!defined $self->{channel}) {
        return undef;
    }
    return $self->{channel}->error();
}

sub openSftp {
    my ($self) = @_;
    $self->{session}->blocking(1);
    if (!defined $self->getSftp()) {
        $self->setSftp($self->{session}->sftp());
        if (!defined $self->getSftp() && $self->reconnect()) {
            $self->{session}->blocking(1);
            $self->setSftp($self->{session}->sftp());
        }
    }
    return defined($self->getSftp());
}

sub openChannel {
    my ($self) = @_;
    if (!$self->authOk()) {
        $self->appendErrorMessage("Host '$self->{hostname}' is not authorized");
        return undef;
    }
    if (!defined $self->{channel}) {
        $self->{channel} = $self->_createChannel();
        if (!defined $self->{channel}) {
            $self->reconnect();
            $self->{channel} = $self->_createChannel();
        }
    }
    return defined($self->{channel});
}

sub _createSession {
    my $self = @_;
    my $session = Net::SSH2->new();
    $session->method('LIBSSH2_METHOD_KEX', 'diffie-hellman-group14-sha1');
    return $session;
}

sub _createChannel {
    my ($self) = @_;
    $self->{session}->blocking(1);
    my $channel = $self->{session}->channel();
    if ($channel) {
        $channel->ext_data('merge');
    }
    return $channel;
}

sub authPassword {
    my ($self, $username, $password, $installSSHKey, $keystr) = @_;
    if ($self->authOk()) {
        return 1;
    }
    $self->setUsername($username);
    $self->setPassword($password);

    if (!defined $self->getUsername()) {
        $self->appendErrorMessage("User name not specified");
        return 0;
    }

    if (!defined $self->getPassword()) {
        $self->appendErrorMessage("Password not specified");
        return 0;
    }

    my $host = $self->getHostname();
    my $session = $self->{session};
    my $result;
    if ($self->_canAuthPassword()) {
        $result = $session->auth_password($self->getUsername(), $self->getPassword());
    }
    else {
        $result = $session->auth_keyboard($self->getUsername(), $self->getPassword());
    }

    if (!$result) {
        $self->appendErrorMessage("Password authorization on host '$host' failed: " . $self->sessionError());
        return 0;
    }
    $self->getMsgLst()->addMessage("Connection to host '$host' established");
    if ($installSSHKey && $keystr) {
        $self->getMsgLst()->addMessage("Distributing public rsa key to host '$host'");
        if (!defined $self->stat('.ssh')) {
            $self->mkdir ('.ssh', 0700);
        }
        my $file = $self->openFile('.ssh/authorized_keys', O_CREAT|O_WRONLY|O_APPEND, 0600);
        if ($file) {
            $file->seek($file->stat ()->{'size'});
            if (!defined $file->write($keystr)) {
                $self->appendErrorMessage("Cannot distribute public rsa key: write to '.ssh/authorized_keys' failed (host '$host')");
            }
        }
        else {
            $self->appendErrorMessage("Cannot distribute public rsa key: cannot open'.ssh/authorized_keys' (host '$host')");
        }
    }
    return 1;
}

sub _canAuthPassword {
    my ($self) = @_;
    if (!defined $self->{_canAuthPassword}) {
        my $tmpSocket = $self->_createSession();
        if (!$tmpSocket->connect($self->{hostname})) {
            return undef;
        }
        $self->{_canAuthPassword} =
            (grep {$_ eq 'password'} $tmpSocket->auth_list()) ?
            1 : 0;
    }
    return $self->{_canAuthPassword};
}

sub authKey {
    my ($self, $ignorePubKeyMissing, $user, $publicKey, $privateKey, $passphrase) = @_;
    if ($self->authOk()) {
        return 1;
    }
    $self->setUsername($user);
    $self->setPublicKey($publicKey);
    $self->setPrivateKey($privateKey);
    $self->setPassphrase($passphrase);

    if (!defined $self->getUsername()) {
        $self->appendErrorMessage("User name unkown");
        return 0;
    }
    if (!defined $self->getPublicKey()) {
        my $error = "No public rsa key";
        $self->_appendAuthKeyErrorMsg($error, $ignorePubKeyMissing);
        return 0;
    }
    if (!defined $self->getPrivateKey()) {
        my $error = "No private rsa key";
        $self->_appendAuthKeyErrorMsg($error, $ignorePubKeyMissing);
        return 0;
    }
    $self->{session}->auth_publickey($self->getUsername(), $self->getPublicKey(), $self->getPrivateKey(), $self->getPassphrase());
    if (!$self->authOk()) {
        $self->_appendAuthKeyErrorMsg($self->sessionError(), $ignorePubKeyMissing);
        return 0;
    }
    return 1;
}

sub authOk {
    return $_[0]->{session}->auth_ok();
}

sub mkdir {
    my ($self, $dir, $mode) = @_;
    my $sftp = $self->getSftpConnection();
    if (!defined $sftp) {
        return undef;
    }
    return $sftp->mkdir($dir, $mode);
}

sub stat {
    my ($self, $path) = @_;
    my $sftp = $self->getSftpConnection();
    if (!defined $sftp) {
        return undef;
    }
    return $sftp->stat($path);
}

sub rmdir {
    my ($self, $dir) = @_;
    my $sftp = $self->getSftpConnection();
    if (!defined $sftp) {
        return undef;
    }
    return $sftp->rmdir($dir);
}

sub unlink {
    my ($self, $path) = @_;
    my $sftp = $self->getSftpConnection();
    if (!defined $sftp) {
        return undef;
    }
    return $sftp->unlink($path);
}

sub openFile {
    my ($self, $file, $flags, $mode) = @_;
    my $sftp = $self->getSftpConnection();
    if (!defined $sftp) {
        return undef;
    }
    return $sftp->open($file, $flags, $mode);
}

sub openDir {
    my ($self, $dir) = @_;
    my $sftp = $self->getSftpConnection();
    if (!defined $sftp) {
        return undef;
    }
    return $sftp->opendir($dir);
}

sub isDirectory {
    my ($self, $stat) = @_;
    return S_ISDIR($stat->{mode});
}

sub isRegularFile {
    my ($self, $stat) = @_;
    return S_ISREG($stat->{mode});
}

sub copyFile {
    my ($self, $localFilePath, $remoteFilePath, $statbuf) = @_;
    my $localFh = IO::File->new();
    if (!$localFh->open($localFilePath)){
        $self->appendErrorMessage("Cannot open local file '$localFilePath': $!");
        return 1;
    }
    my $bufSize = $statbuf->[11] || 32768;
    my ($buffer, $written, $offset, $len);
    my $file = $self->openFile($remoteFilePath, O_CREAT|O_WRONLY, $statbuf->[2]);
    if (!defined $file) {
        my ($code, $str, $dsc) = $self->sftpErrorRaw();
        $self->appendErrorMessage("Cannot create remote file '$remoteFilePath'"
                . (defined $str ? ": $str" : '')
                . (defined $dsc ? "- $dsc" : ''));
        return 1;
    }
    $offset = 0;
    $localFh->sysseek(0, SEEK_SET);
    while ($len = $localFh->sysread($buffer, $bufSize)) {
        if (!defined $len) {
            next if $! =~ /^Interrupted/;
            $self->appendErrorMessage('Read failure: ' . $!);
            return 1;
        }
        $file->seek($offset);
        $written = $file->write($buffer);
        if (!defined $written) {
            $self->appendErrorMessage('Write failure: ' . $!);
            return 1;
        }
        $len -= $written;
        $offset += $written;
    }
    return 0;
}

sub deltree {
    my ($self, $path, $msglst) = @_;
    my $rc = 0;
    my ($code, $str, $dsc);
    my $stat = $self->stat($path);
    if (!defined $stat) {
        # already gone
        return 0;
    }
    if ($self->isDirectory($stat)) {
        my $dir = $self->openDir($path);
        if (!defined $dir) {
            $self->_appendSftpError("Cannot open directory '$path'", $msglst);
            return 1;
        }
        my $entry;
        while ($entry = $dir->read()) {
            next if $self->isEntryToSkip($entry);
            $rc ||= $self->deltree(File::Spec->catfile($path, $entry->{name}), $msglst);
        }
        if ($rc == 0) {
            $self->rmdir($path);
            ($code, $str, $dsc) = $self->sftpErrorRaw();
            if ($code != 0) {
                $self->_appendSftpError("Cannot delete directory '$path'", $msglst);
                $rc = 1;
            }
        }
    }
    else {
        $self->unlink($path);
        ($code, $str, $dsc) = $self->sftpErrorRaw();
        if ($code != 0){
            $self->_appendSftpError("Cannot delete file '$path'", $msglst);
            $rc = 1;
        }
    }
    return $rc;
}

sub existsFile {
    my ($self, $path, $isDirectory, $msgLst, $outStatError) = @_;
    my $hostname = $self->getHostname();
    if (!defined $msgLst) {
        $msgLst = $self;
    }

    my $sftp = $self->getSftpConnection();
    return 0 if !defined($sftp);

    my $stat = $sftp->stat($path);
    if (!defined $stat) {
        $stat = $sftp->stat($path);
    }
    if (!defined($stat)) {
        my @statError = $self->sftpErrorRaw();
        if ($statError[0] == 2) {
            $msgLst->appendErrorMessage("File '$path' does not exist on host '$hostname': $statError[1]");
        }
        elsif ($statError[0] == 3) {
            $msgLst->appendErrorMessage("File '$path' cannot be accessed on host '$hostname': $statError[1]");
        }
        else {
            $msgLst->appendErrorMessage("stat ('$path') on host '$hostname' failed: " .
                join (', ', @statError));
        }
        if (defined $outStatError) {
            push @$outStatError, [@statError, $self->{hostname}];
        }
        return 0;
    }

    return $self->checkFileType($path, $stat, $isDirectory, $msgLst);
}

sub executeCommand {
    my ($self, $cmd, $inputData, $blocking) = @_;
    if (!$self->authOk()) {
        $self->appendErrorMessage("Host '". $self->getHostname() ."' is not authorized");
        return 2;
    }
    if (!defined($self->{channel})) {
        if (!$self->openChannel()) {
            $self->appendErrorMessage("Cannot execute command because a channel cannot be opened");
            return 16;
        }
    }
    $blocking //= 0;
    $self->{channel}->blocking($blocking);
    $self->{channel}->exec($cmd);
    if (defined $inputData) {
        $self->setStdinBuffer($inputData);
        $self->write();
    }
    return 0;
}

sub _writeToChannel {
    my ($self, $inputData) = @_;
    my $dataLen = length($inputData);
    my $written = $self->{channel}->write($inputData);
    if (defined $written && $written == $dataLen) {
        return 1;
    }
    return 0;
}

sub sendEof {
    my ($self) = @_;
    return $self->{channel}->send_eof();
}

sub hasChannel {
    my ($self) = @_;
    return defined($self->{channel});
}

sub getOutput {
    my ($self) = @_;
    if (!$self->hasChannel()) {
        $self->appendErrorMessage("There isn't an open channel for reading on host '". $self->getHostname() ."'");
        return undef;
    }
    my $result = $self->{channel}->read2();
    $self->appendToOutputBuffer($result);
    return $result;
}

sub finishedExecution {
    my ($self) = @_;
    if (!$self->hasChannel()) {
        return 1;
    }
    my $rc = $self->{channel}->eof();
    return $rc;
}

sub exitChannel {
    my ($self) = @_;
    if (!$self->hasChannel()) {
        return 1;
    }

    my $channel = $self->{channel};
    $channel->close();
    $channel->wait_closed();

    my ($signal, $rc);
    if ($can_exit_signal) {
        $signal = $channel->exit_signal();
    }
    if ($signal) {
        $rc = 130;
    }
    else {
        $rc = $channel->exit_status();
    }
    $self->{channel} = undef;
    $self->deleteOutputBuffer();
    return ($rc, $signal);
}

sub disconnect {
    my ($self) = @_;
    return $self->{session}->disconnect();
}

sub generateSSHKeys {
    my ($self, $privateKeyPath, $msglst) = @_;

    my $sshKeygenPath  = File::Spec->catfile('', 'usr', 'bin', 'ssh-keygen');
    if (! -x $sshKeygenPath) {
        $msglst->addWarning("'$sshKeygenPath' cannot be executed");
    }
    my $publicKeyPath  = File::Spec->catfile(dirname($privateKeyPath), 'id_rsa.pub');
    my $args = ['-m', 'PEM', '-t', 'rsa', '-b', '4096', '-N', '', '-f', $privateKeyPath];

    my $rc = SDB::Install::System::exec_program ($sshKeygenPath, $args, $msglst);
    return (defined $rc && $rc == 0);
}

1;
