package SDB::Install::SSH::LibsshSession;

use strict;
use warnings;

use POSIX;
use File::Spec;
use File::Basename qw (dirname);
use IO::File;
use parent qw(SDB::Install::SSH::BaseSession);

use constant {
    SSH_DIR => '.ssh',
    SSH_AUTHORIZED_KEYS => '.ssh/authorized_keys',
};

sub new {
    my $self = shift->SUPER::new();
    $self->{session} = $self->_createSession();
    return $self;
}

sub _createSession {
    my $session = Libssh::Session->new();
    return $session;
}

sub connect {
    my ($self, $hostname) = @_;
    my $session = $self->{session};
    $self->setHostname($hostname);
    if ($session->options(host => $hostname, logverbosity => $session->SSH_LOG_NOLOG) != $session->SSH_OK) {
        return 0;
    }
    if ($session->connect() != $session->SSH_OK) {
        return 0;
    }
    return 1;
}

# After finishing SFTP operation all references to the sftp object should be removed, otherwise Bug 254992 can occur
sub openSftp {
    my ($self) = @_;
    my $session = $self->{session};
    $session->set_blocking(blocking => 1);
    my $sftp = Libssh::Sftp->new(session => $session);
    if (!defined($sftp) && $self->reconnect()) {
        $session = $self->{session};
        $session->set_blocking(blocking => 1);
        $sftp = Libssh::Sftp->new(session => $session);
    }
    $self->setSftp($sftp);
    return defined($sftp);
}

sub authKey {
    my ($self, $ignorePubKeyMissing, $user, $publicKey, $privateKey, $passphrase) = @_;

    if ($self->authOk()) {
        return 1;
    }
    $self->setUsername($user);
    $self->setPassphrase($passphrase);

    if (!defined $self->getUsername()) {
        $self->appendErrorMessage("User name unkown");
        return 0;
    }
    my $session = $self->{session};
    if ($session->options(user => $self->getUsername()) != $session->SSH_OK) {
        $self->_appendAuthKeyErrorMsg($self->sessionError(), $ignorePubKeyMissing);
        return 0;
    }
    if ($session->auth_publickey_auto(passphrase => $self->getPassphrase()) != $session->SSH_AUTH_SUCCESS) {
        $self->_appendAuthKeyErrorMsg($self->sessionError(), $ignorePubKeyMissing);
        return 0;
    }
    return 1;
}

sub authPassword {
    my ($self, $username, $password, $installSSHKey, $keystr) = @_;
    if ($self->authOk()) {
        return 1;
    }
    $self->setUsername($username);
    $self->setPassword($password);

    if (!defined $self->getUsername()) {
        $self->appendErrorMessage("User name unkown");
        return 0;
    }
    if (!defined $self->getPassword()) {
        $self->appendErrorMessage("Password unkown");
        return 0;
    }

    my $host = $self->getHostname();
    my $session = $self->{session};
    my $result = $session->SSH_AUTH_ERROR;
    if ($session->options(user => $self->getUsername()) != $session->SSH_OK) {
        $self->_appendAuthPassErrorMsg($self->sessionError());
        return 0;
    }
    if ($self->_canAuthPassword()) {
        $result = $session->auth_password(password => $self->getPassword());
    }
    elsif ($self->_canAuthKeyboard()) {
        $result = $self->authKeyboardInteractive($self->getUsername(), $self->getPassword());
    }
    if ($result != $session->SSH_AUTH_SUCCESS) {
        $self->_appendAuthPassErrorMsg($self->sessionError());
        return 0;
    }
    $self->getMsgLst()->addMessage("Connection to host '$host' established");
    if ($installSSHKey && $keystr) {
        $self->getMsgLst()->addMessage("Distributing public key to host '$host'");
        if (!defined($self->stat(SSH_DIR))) {
            $self->mkdir(SSH_DIR, 0700);
        }
        if (!$self->writeToFile(SSH_AUTHORIZED_KEYS, $keystr, 0600)) {
            $self->appendErrorMessage("Cannot distribute public key (host '$host')");
        }
    }
    return 1;
}

sub authKeyboardInteractive {
    my ($self, $username, $password) = @_;
    my $index = 0;
    my $session = $self->{session};
    my $result = $session->auth_kbdint();
    return $result if ($result != $session->SSH_AUTH_INFO ||
                       $session->auth_kbdint_setanswer(index => $index, answer => $password) != 0);

    $result = $session->auth_kbdint();
    if ($result == $session->SSH_AUTH_INFO) {
        if ($session->auth_kbdint_getnprmopts() > 0) {
            # The ssh server has prompts again, meaning that the password was invalid.
            return $session->SSH_AUTH_DENIED;
        }
        # The status needs to be refreshed.
        $result = $session->auth_kbdint();
    }
    return $result;
}

sub stat {
    my ($self, $path) = @_;
    my $sftp = $self->getSftpConnection();
    return undef if (!defined($sftp));

    my $result = $sftp->stat_file(file => $path);
    $self->setSftp(undef);
    return $result;
}

sub mkdir {
    my ($self, $dir, $mode) = @_;
    my $sftp = $self->getSftpConnection();
    return undef if (!defined($sftp));

    my $result = $sftp->mkdir(dir => $dir, mode => $mode) == $sftp->SSH_OK;
    $self->setSftp(undef);
    return $result;
}

sub writeToFile {
    my ($self, $filePath, $data, $mode, $accessType) = @_;
    $accessType //= (O_CREAT|O_WRONLY|O_APPEND);
    my $sftp = $self->getSftpConnection();
    return undef if (!defined($sftp));

    my $result = 1;
    my $file = $sftp->open(file => $filePath, accesstype => $accessType, mode => $mode);
    if (defined($file) && $sftp->write(handle_file => $file, data => $data) != $sftp->SSH_OK) {
        $self->appendErrorMessage("Cannot write to file '$file': " . $sftp->error());
        $result = 0;
    }
    $self->setSftp(undef);
    return $result;
}

sub existsFile {
    my ($self, $path, $isDir, $msglst, $statErrors) = @_;
    $msglst //= $self;
    my $sftp = $self->getSftpConnection();
    return 0 if (!defined($sftp));

    my $hostname = $self->getHostname();
    my $stat = $sftp->stat_file(file => $path);
    my $result = 1;

    if (!defined($stat)) {
        my $error = "'$path' is not accessible on host '$hostname'";
        $msglst->appendErrorMessage($error);
        if (defined($statErrors)) {
            # for backward compatibility with libssh2 statError handling use error code 2
            push(@$statErrors, [2, $hostname]);
        }
        $result = 0;
    }
    $self->setSftp(undef);
    return $result ? $self->checkFileType($path, $stat, $isDir, $msglst) : $result;
}

sub copyFile {
    my ($self, $localFilePath, $remoteFilePath, $statbuf) = @_;
    my $sftp = $self->getSftpConnection();
    return 1 if (!defined($sftp));

    my $mode = $statbuf->[2] || undef;
    my $chunk = $statbuf->[11] || undef;
    my $timeout = 300;
    my $result = 0;
    if ($sftp->copy_file(src => $localFilePath, dst => $remoteFilePath, mode => $mode, chunk => $chunk, timeout => $timeout) != 0) {
        $self->_appendSftpError("Error copying file '$localFilePath' to host " . $self->getHostname());
        $result = 1;
    }
    $self->setSftp(undef);
    return $result;
}

sub isDirectory {
    my ($self, $stat) = @_;
    return $stat->{type} == Libssh::Sftp->SSH_FILEXFER_TYPE_DIRECTORY;
}

sub isRegularFile {
    my ($self, $stat) = @_;
    return $stat->{type} == Libssh::Sftp->SSH_FILEXFER_TYPE_REGULAR;
}

sub deltree {
    my ($self, $path, $msglst) = @_;
    $msglst //= $self;
    my $stat = $self->stat($path);
    return 0 if(!defined($stat));

    my $sftp = $self->getSftpConnection();
    return 1 if(!defined($sftp));

    if (!$self->isDirectory($stat)) {
        if ($sftp->unlink(file => $path) != $sftp->SSH_OK) {
            $self->_appendSftpError("Error deleting '$path'", $msglst);
            return 1;
        }
        $self->setSftp(undef);
        return 0;
    }
    my $dir = $sftp->opendir(dir => $path);
    if (!defined($dir)) {
        $self->_appendSftpError("Cannot open directory '$path'", $msglst);
        return 1;
    }
    my $rc = 0;
    while (!$sftp->dir_eof(handle_dir => $dir)) {
        my $entry = $sftp->readdir(handle_dir => $dir);
        next if (!defined($entry) || $self->isEntryToSkip($entry));
        $rc |= $self->deltree(File::Spec->catfile($path, $entry->{name}), $msglst);
    }
    $sftp->closedir(handle_dir => $dir);
    if ($rc == 0 && $sftp->rmdir(dir => $path) != $sftp->SSH_OK) {
        $self->_appendSftpError("Error deleting '$path'", $msglst);
        return 1;
    }
    $self->setSftp(undef);
    return $rc;
}

sub authOk {
    my ($self) = @_;
    return $self->{session}->is_authenticated();
}

sub _getAuthList {
    my ($self) = @_;
    my $session = $self->{session};
    if (!defined($self->{auth_list}) && defined($session)) {
        $session->auth_none();
        $self->{auth_list} = $session->auth_list();
    }
    return $self->{auth_list};
}

sub _canAuthPassword {
    my ($self) = @_;
    my $session = $self->{session};
    my $auth_list = $self->_getAuthList();
    my $result = $auth_list & $session->SSH_AUTH_METHOD_PASSWORD;
    return $result;
}

sub _canAuthKeyboard {
    my ($self) = @_;
    my $session = $self->{session};
    my $auth_list = $self->_getAuthList();
    my $result = $auth_list & $session->SSH_AUTH_METHOD_INTERACTIVE;
    return $result;
}

sub getChannelId {
    my ($self) = @_;
    return $self->{_channelId};
}

sub setChannelId {
    my ($self, $value) = @_;
    $self->{_channelId} = $value;
}

sub hasChannel {
    my ($self) = @_;
    return defined($self->getChannelId());
}

sub finishedExecution {
    my ($self) = @_;
    my $channelId = $self->getChannelId();
    if (!defined $channelId) {
        return 1;
    }
    my $channel = $self->{session}->get_channel(channel_id => $channelId);
    if (!defined $channel) {
        return 1;
    }
    my $rc = $self->{session}->channel_get_exit_status(channel => $channel);
    my $eof = $self->{session}->channel_is_eof(channel => $channel);
    if ($rc != -1 && $eof != 0) {
        return 1;
    }
    return 0;
}

sub getCallbackHash {
    my ($self) = @_;
    my $callback = $self->{session}->{store_no_callback};
    if (defined($callback)) {
        return $callback->[scalar(@$callback) - 1];
    }
    return undef;
}

sub deleteCallbackHash {
    my ($self) = @_;
    $self->{session}->{store_no_callback} = [];
}

sub exitChannel {
    my ($self) = @_;
    my $channelId = $self->getChannelId();
    if (!defined $channelId) {
        return 1;
    }
    my $channel = $self->{session}->get_channel(channel_id => $channelId);
    if (!defined $channel) {
    } else {
        $self->{session}->execute_read_channel(channel_id => $channelId);
    }
    my $callbackHash = $self->getCallbackHash();
    my $exitCode = (defined $callbackHash) ? $callbackHash->{exit_code} : 1;

    $self->deleteCallbackHash();
    $self->deleteOutputBuffer();
    $self->setChannelId(undef);
    return ($exitCode, undef); # LibsshSession doesn't return any signals
}

sub executeCommand {
    my ($self, $cmd, $stdin, $blocking) = @_;
    if (!$self->authOk()) {
        $self->appendErrorMessage("Host '". $self->getHostname() ."' is not authorized");
        return 2;
    }
    $stdin = join("\n", @$stdin) if defined($stdin);
    my $channelId = $self->{session}->add_command_internal(
        command => {cmd => $cmd, input_data => $stdin}
    );
    if (!defined $channelId && $self->reconnect()) {
        $channelId = $self->{session}->add_command_internal(
            command => {cmd => $cmd, input_data => $stdin}
        );
    }
    if (!defined($channelId)) {
        $self->appendErrorMessage("Failed to start execution of command '$cmd'");
        return 16;
    }
    $self->setChannelId($channelId);
    $self->{session}->set_blocking(blocking => $blocking // 0);
    return 0;
}

# These additional methods for writing stdin to the channel are relevant only for libssh2
sub write {
    return 1;
}

sub isEmptyInputBuffer {
    return 1;
}

sub getOutput {
    my ($self) = @_;
    my $channelId = $self->getChannelId();
    if (!defined $channelId) {
        return '';
    }
    my $stdout = '';
    my $channel = $self->{session}->get_channel(channel_id => $channelId);
    if (!defined $channel) {
        $stdout = $self->getOutputFromCallback();
    } else {
        my $exitStatus = $self->{session}->channel_get_exit_status(channel => $channel);
        my $isEof = $self->{session}->channel_is_eof(channel => $channel);
        if ($exitStatus == -1 || $isEof == 0) { # not finished
            $self->{session}->execute_read_channel(channel_id => $channelId);
            $stdout = $self->getOutputFromChannelSlots();
            if (!$stdout) {
                $stdout = $self->getOutputFromCallback()
            }
        }
    }
    $stdout = $self->resizeOutput($stdout);
    $self->appendToOutputBuffer($stdout);
    return $stdout;
}

sub getOutputFromChannelSlots {
    my ($self) = @_;
    my $channelId = $self->getChannelId();
    my $stdout = $self->{session}->{slots}->{$channelId}->{stdout};
    my $stderr = $self->{session}->{slots}->{$channelId}->{stderr};
    my $result = '';
    $result .= $stdout if defined($stdout);
    $result .= $stderr if defined($stderr);
    return $result;
}

sub getOutputFromCallback {
    my ($self) = @_;
    my $callback = $self->getCallbackHash();
    my $stdout = '';
    if (defined($callback)) {
        $stdout .= $callback->{stdout} if defined($callback->{stdout});
        $stdout .= $callback->{stderr} if defined($callback->{stderr});
    }
    return $stdout;
}

sub resizeOutput {
    my ($self, $output) = @_;
    $output //= '';
    my $outputLength = length($output);
    my $outputBufferLength = length($self->getOutputBuffer());
    my $diff = $outputLength - $outputBufferLength;
    my $result = '';
    if ($diff >= 0) {
        $result = substr($output, $outputBufferLength);
    }
    return $result;
}

sub disconnect {
    my ($self) = @_;
    $self->{session}->disconnect() if defined($self->{session});
}

sub generateSSHKeys {
    my ($self, $privateKeyPath, $msglst) = @_;
    my $sshKeygenPath  = File::Spec->catfile('', 'usr', 'bin', 'ssh-keygen');
    if (! -x $sshKeygenPath) {
        $msglst->addWarning("'$sshKeygenPath' cannot be executed");
    }
    my $publicKeyPath  = File::Spec->catfile(dirname($privateKeyPath), 'id_rsa.pub');
    my $args = ['-b', '4096', '-N', '', '-f', $privateKeyPath];

    my $rc = SDB::Install::System::exec_program($sshKeygenPath, $args, $msglst);
    return (defined $rc && $rc == 0);
}

sub sessionError {
    my ($self) = @_;
    my $session = $self->{session};
    if (!defined($session)) {
        return undef;
    }
    return $session->error(GetErrorSession => 1) // '';
}

sub sftpError {
    my ($self) = @_;
    my $error;
    if (defined($self->getSftp())) {
        $error = $self->getSftp()->error();
    }
    return $error // Libssh::Sftp::error() // '';
}

1;
