package SDB::Install::Sql::Hana2PersistenceMigrationHelper;


use base SDB::Install::Base;

use LCM::FileUtils qw(readFile);
use SDB::Install::Sql::SqlConnectionProvider;
use SDB::Install::SAPInstance::TrexInstance;
use strict;

sub executeHana2PersistenceMigrationHelper {
	my (
	    $self,
	    $sqlScriptLocation,
	    $user_store_key,
	    $sqlUser,     # optional, defaults to 'SYSTEM'
	    $sqlPassword,
	    $host,
	    $instanceNumber,
	    $isMultiDb,
	    $trexinstance # optional, only needed for SSL connections
	) = @_;
    my $msglst = $self->getMsgLst();
    $sqlUser //= 'SYSTEM';
    my $migMsg = $msglst->addMessage ("Migration of table persistencies for HANA2:");
    my $subMsglst = $migMsg->getSubMsgLst();
    $subMsglst->addMessage ("Reading SQL script '$sqlScriptLocation'...");
    my $fileContents = readFile($sqlScriptLocation, $subMsglst);
    if(not defined $fileContents) {
        $subMsglst->addWarning("Cannot read sql script '$sqlScriptLocation'");
        return undef;
    }
    $subMsglst->addMessage ("SQL script read.");
    $subMsglst->addMessage ("Establishing SQLDBC connection...");
    my $sqlScriptBuf = join("", @{$fileContents});

    my $connectionProvider = new SDB::Install::Sql::SqlConnectionProvider();
    $connectionProvider->setMsgLstContext([$subMsglst]);
    my $sql = $connectionProvider->getSqlConnection(
                $isMultiDb ? 'SYSTEMDB' : undef,
                $trexinstance,
                $sqlUser,
                $sqlPassword,
                $user_store_key);

    if (!defined $sql){
        if ($connectionProvider->isECONNREFUSED){
            return 2;
        }
        return undef;
    }
    $subMsglst->addMessage ("SQLDBC connection established.");
    $subMsglst->addMessage ("Creating and executing migration procedure...");
    my $dropError = undef;
    if (!defined $sql->execute('DROP PROCEDURE HANA2_PERSISTENCE_MIGRATION_HELPER')){
        $dropError = "Deletion of procedure HANA2_PERSISTENCE_MIGRATION_HELPER failed: " . $sql->GetErrorString();
    }
    if (!defined $sql->execute($sqlScriptBuf)){
        if(defined $dropError) {
            $subMsglst->addWarning($dropError);
        }
        $subMsglst->addWarning("Creation of procedure HANA2_PERSISTENCE_MIGRATION_HELPER failed: " . $sql->GetErrorString());
        return undef;
    }
    if (!defined $sql->execute('call HANA2_PERSISTENCE_MIGRATION_HELPER(1,?)')){
        $subMsglst->addWarning("Execution of procedure HANA2_PERSISTENCE_MIGRATION_HELPER failed: " . $sql->GetErrorString());
        return undef;
    }
    my $result = $sql->getResultSet();
    if (!defined $result){
        $subMsglst->addWarning("getResultSet failed: " . $sql->GetErrorString());
        return undef;
    }
    $subMsglst->addMessage ("Migration procedure executed.");
    if(not @$result) {
        $subMsglst->addMessage ("No tables needed to be migrated.");
        return 1;
    }
        # process the result (i.e. the schemas and names of the migrated tables and, if existing,
        # the migration error codes and error texts),
        # as of 11/2016 it has this form:
        #
        # TABLE (schema_name nvarchar(256),
        #        table_name nvarchar(256),
        #        error_code integer,
        #        error_message nvarchar(5000))
        #
    my $successRows = [];
    my $failRows = [];
    foreach my $row (@$result){
        if($row->[2] == 0) {
            push(@$successRows, $row);
        }
        else {
            push(@$failRows, $row);
        }
    }
    if(@$successRows) {
        my $successResultMsg = $subMsglst->addMessage("The following tables have been migrated sucessfully:");
        my $successResultMsglst = $successResultMsg->getSubMsgLst();
        foreach my $row (@$successRows) {
            my $schema = $row->[0];
            my $table = $row->[1];
            $successResultMsglst->addMessage("$schema.$table");
        }
    }
    else {
        $subMsglst->addWarning("No tables have been migrated sucessfully.");
    }
    if(@$failRows) {
        my $failResultMsg = $subMsglst->addWarning("The following tables could not be migrated:");
        my $failResultMsglst = $failResultMsg->getSubMsgLst();
        foreach my $row (@$failRows) {
            my $schema = $row->[0];
            my $table = $row->[1];
            my $errorCode = $row->[2];
            my $errorText = $row->[3];
            $failResultMsglst->addMessage("$schema.$table: error code: $errorCode, error text: $errorText");
        }
        return undef;
    }
    return 1;
}

#--------------------------------------------------------------------------------------------------

# this main method is for testing/debugging  the above method.
# the sql script in he comment below can be used to simulate the
# result of the actual migration helper script.
#
sub main{
    local @ARGV = @_;
    require Getopt::Long;
    import Getopt::Long;
    my $usage = 'usage: -f <persMigrationHelper sql script> -u <system user> -p <system password> -h <host> -n <instance number> -m <multidb:true|false>'."\n".
                '       -d <instance dir> -s <sid> -k <user store key>';
    my ($opt_f, $opt_u, $opt_p, $opt_h, $opt_n, $opt_m, $opt_d, $opt_s, $opt_k);
    if (!GetOptions (
        'f=s' => \$opt_f,
        'u=s' => \$opt_u,
        'p=s' => \$opt_p,
        'h=s' => \$opt_h,
        'n=s' => \$opt_n,
        'm=s' => \$opt_m,
        'd=s' => \$opt_d,
        's=s' => \$opt_s,
        'k=s' => \$opt_k
    )){
        print "$usage\n";
        return 1;
    }
    my $trexinstance = new SDB::Install::SAPInstance::TrexInstance( $opt_d, $opt_s );
    $trexinstance->{_nr} = $opt_n;
    my $helper = new SDB::Install::Sql::Hana2PersistenceMigrationHelper();
    $helper->executeHana2PersistenceMigrationHelper(
        $opt_f,
        $opt_k, # user store key
        $opt_u,
        $opt_p,
        $opt_h,
        $opt_n,
        ($opt_m eq 'true') ? 1 : 0,
        $trexinstance
    );
    print "done\n";
    return 1;
}

#
#DROP PROCEDURE HANA2_PERSISTENCE_MIGRATION_HELPER;
#CREATE PROCEDURE HANA2_PERSISTENCE_MIGRATION_HELPER (IN _doMigration TINYINT DEFAULT 0,
#                                                     OUT _outTable TABLE (schema_name nvarchar(256),
#                                                                          table_name nvarchar(256),
#                                                                          error_code integer,
#                                                                          error_message nvarchar(5000)) )
#LANGUAGE SQLScript SQL SECURITY INVOKER
#AS
#BEGIN
#    CREATE LOCAL TEMPORARY TABLE #_tableStatus (schema_name nvarchar(256), table_name nvarchar(256), error_code integer, error_message nvarchar(5000));
#    INSERT INTO #_tableStatus VALUES ('GoodSchema', 'GoodTable01', 0, 'AllOK');
#    INSERT INTO #_tableStatus VALUES ('GoodSchema', 'GoodTable02', 0, '');
#    INSERT INTO #_tableStatus VALUES ('BadSchema', 'Badtable01', 37, 'SNAFU');
#    INSERT INTO #_tableStatus VALUES ('BadSchema', 'Badtable02', 38, 'OMG');
#    INSERT INTO #_tableStatus VALUES ('BadSchema', 'Badtable03', 39, '');
#    INSERT INTO #_tableStatus VALUES ('GoodSchema', 'GoodTable03', 0, 'WTF');
#    _outTable = SELECT DISTINCT schema_name, table_name, error_code, error_message FROM #_tableStatus;
#    DROP TABLE #_tableStatus;
#END;

#--------------------------------------------------------------------------------------------------

1;
