package SDB::Install::Sql::SSLConnectProperties;

use strict;

use base 'SDB::Install::Base';
use SDB::Install::SysVars;
use File::Basename qw (dirname);
use File::stat;
use SDB::Install::Globals qw($gBoolTruePattern);

our $iniFileSection         = 'communication';
our $providerKey            = 'sslCryptoProvider';
our @storeValueNames        = qw (sslKeyStore sslTrustStore);
our $sslValidateCertificate = 'sslValidateCertificate';
our $sapcliPseName          = 'sapcli.pse';

our @openSSLKeys = (@storeValueNames, $sslValidateCertificate);
our @CCLKeys = (@storeValueNames, $sslValidateCertificate);


sub findSSLProperties{
    my ($self, $hdbInstance, $hostname) = @_;
    my $properties = {'ENCRYPT' => '1'};
    my $global_ini = $hdbInstance->getGlobalIni ();
    my $sslCryptoProvider = $global_ini->getValue ($iniFileSection, $providerKey);
    if (!$sslCryptoProvider){
        $self->setErrorMessage ("No $providerKey defined");
        return undef;
    }
    $properties->{$providerKey} = $sslCryptoProvider;
    $self->getMsgLst ()->addMessage ("$providerKey '$sslCryptoProvider' detected");
    if ($sslCryptoProvider eq 'openssl'){
        return $self->findOpenSSLProperties ($global_ini, $properties);
    }
    elsif ($sslCryptoProvider eq 'commoncrypto'){
        return $self->findCCLProperties ($global_ini, $hdbInstance, $hostname, $properties);
    }
    else{
        $self->setErrorMessage ("$providerKey '$sslCryptoProvider' is not supported by installer yet");
        return undef;
    }
}

sub findOpenSSLProperties{
    my ($self, $global_ini, $properties) = @_;
    my $msglst = $self->getMsgLst ();
    my $value;
    foreach my $openSSLKey (@openSSLKeys){
        $properties->{$openSSLKey} = $global_ini->getValue ($iniFileSection, $openSSLKey);
    }
    return [$properties];
}

sub _changeServerSpecificKeyStoreProperties{
    my ($self, $sslProperties, $msglst) = @_;
    foreach my $keyStoreValueName (@storeValueNames){
        if (exists $sslProperties->{$keyStoreValueName}){
            $msglst->addMessage("Using '$keyStoreValueName' '$sapcliPseName'");
            $sslProperties->{$keyStoreValueName} = $sapcliPseName;
        }
    }
    return 1;
}


sub findCCLProperties{
    my ($self, $global_ini, $hdbInstance, $hostname, $properties) = @_;

    my $env = $self->getCCLEnvironment ($hdbInstance, $hostname);

    if (!defined $env){
        return undef;
    }

    my $msglst = $self->getMsgLst ();

    foreach my $CCLKey (@CCLKeys){
        $properties->{$CCLKey} = $global_ini->getValue ($iniFileSection, $CCLKey);
    }

    if (defined $properties->{$sslValidateCertificate} &&
        ($properties->{$sslValidateCertificate} =~ /$gBoolTruePattern/i)){
            $msglst->addMessage("Server-side $sslValidateCertificate is enabled");
            my $secuDir = $env->{SECUDIR};
            my $sapcli_pse = "$secuDir/$sapcliPseName";
            my $stat = File::stat::stat($sapcli_pse);
            if (defined $stat){
                $msglst->addMessage("$sapcli_pse found");
                if (-f $stat){
                    my $msg = $msglst->addMessage("Resetting server specific key store properties");
                    $self->_changeServerSpecificKeyStoreProperties($properties, $msg->getSubMsgLst());
                }
                else{
                    $msglst->addMessage("'$sapcli_pse' is not a regular file");
                }
            }
            else{
                $msglst->addMessage("Cannot access '$sapcli_pse': $!");
            }
    }
    #disable client-side cert validation
    $properties->{$sslValidateCertificate} = 'false';
    return [$properties, $env];
}

sub getCCLEnvironment{
    my ($self, $hdbInstance, $hostname) = @_;

    my $secDir = $hdbInstance->get_usrSapHostNameDir ($hostname) . $path_separator . 'sec';

    if (!-d $secDir){
        $self->setErrorMessage ("Cannot access key store directory '$secDir': $!");
        return undef;
    }

    my $env = {
        'SECUDIR' => $secDir,
        'USER' => $hdbInstance->getSidAdmName (),
        'SID' => $hdbInstance->get_sid(),
        'INSTANCE_NUMBER'  => $hdbInstance->get_nr()
    };

    if (!defined $hdbInstance->{_localUsrSapSid}){
        my $instPath = dirname ($hdbInstance->get_globalSidDir());
        $env->{HANA_INSTALLATION_PATH} = $instPath;
    }
    else{
        $env->{HANA_INSTALLATION_PATH} = '/usr/sap';
    }
    return $env;
}


1;
