package SDB::Install::VersionDependency;

use SDB::Install::DebugUtilities qw (dumpThings);

use base SDB::Install::BaseLegacy;
use strict;

sub new{
	# Parameters:
	#  minimum (required): 1 = check minimum constraint, 0 = check maximum constraint
	#  inclusive (required): 1 = range of compatible versions includes the boundaries (<=, >= instead of <, >),
	#                        0 = range of compatible versions excludes the boundaries
	#  release (required): release in the format [\d]+.[a-z\d]+
	#  SP (optional): support package in the format [\d]+
	#  PL (optional): patch level in the format [\d]+
	#  CL (optional): patch level in the format [\d]+


	my $self = shift->SUPER::new ();
	($self->{minimum}, $self->{inclusive}, $self->{release}, $self->{SP}, $self->{PL}, $self->{CL}) = @_;
	if (!defined $self->{minimum} || !defined $self->{inclusive} || !defined $self->{release}) {
		return undef;
	}
	return $self;
}

sub isCompatible{
	my ($self, $checkRelease, $checkSP, $checkPL, $checkCL) = @_;

	my @ownVersionSegments = split(/\./, $self->getVersionString);
	my @targetVersionSegments = split(/\./, "$checkRelease.$checkSP.$checkPL.$checkCL");

	for (0 .. $#ownVersionSegments) {
		my $ownSegment = $ownVersionSegments[$_];
		my $targetSegment = $targetVersionSegments[$_];

		my $nextOwnSegment = $ownVersionSegments[$_ + 1];
		my $nextTargetSegment = $targetVersionSegments[$_ + 1];

		if (defined $ownSegment && defined $targetSegment) {
			my $inclusive = ($self->{inclusive} || (defined $nextOwnSegment && defined $nextTargetSegment)) ? 1 : 0;
			return 0 if (!$self->fits($ownSegment, $targetSegment, $inclusive));

			if ($self->_areSegmentsEqual($ownSegment, $targetSegment)) {
				next;
			} else {
				return 1;
			}
		} else {
			return 1;
		}
	}
	return 1;
}

sub _areSegmentsEqual {
	my ($self, $firstSegment, $secondSegment) = @_;

	my $areCharacterSegments = ($firstSegment =~ /[a-z]/i || $secondSegment  =~ /[a-z]/i) ? 1 : 0;

	return 1 if (!$areCharacterSegments && $firstSegment == $secondSegment);
	return 1 if ($areCharacterSegments && $firstSegment eq $secondSegment);
	return 0;
}


sub fits{
	# Check if a part of a version (i.e. a release, SP, PL, or CL) meets the current constraint (<, <=, >, or >=)
	# Parameters:
	#  own (required): a part of the constraint
	#  other (required): a part of the version to be checked
	#  $inclusive (required): 1 = check includes constraint boundaries (<=, >= instead of <, >),
	#                         0 = check does not include constraint boundaries

	my ($self, $own, $other, $inclusive) = @_;
# Can't compare versions with characters in them with
# pure digit ones
	return 0 if (($own =~ /[a-z]/i && $other !~ /[a-z]/i) ||
				 ($own !~ /[a-z]/i && $other =~ /[a-z]/i));

	my $isCharComparison = ($own =~ /[a-z]/i) ? 1 : 0;

	if ($self->{minimum}) {
		# we are checking a minimum
		if ($inclusive) {
			# $other must be greater or equal
			if ($isCharComparison) {
				return $other ge $own;
			} else {
				return $other >= $own;
			}
		} else {
			# $other must be greater
			if ($isCharComparison) {
				return $other gt $own;
			} else {
				return $other > $own;
			}
		}
	} else {
		# we are checking a maximum
		if ($inclusive) {
			# $other must be less or equal
			if ($isCharComparison) {
				return $other le $own;
			} else {
				return $other <= $own;
			}
		} else {
			# $other must be less
			if ($isCharComparison) {
				return $other lt $own;
			} else {
				return $other < $own;
			}
		}
	}
}


sub getVersionString{
    my ($self) = @_;
    my $version = $self->{release};
    for my $versionElement (qw(SP PL CL)) {
        last if (!$self->{$versionElement});
        $version .= ".$self->{$versionElement}";
    }
    return $version;
}

sub getDependencyVersionString{
    my ($self) = @_;
    my $string;
    if ($self->{minimum}){
        $string = $self->getVersionString () . ' <';
        if ($self->{inclusive}){
            $string .= '=';
        }
    }
    else{
        $string = '<';
        if ($self->{inclusive}){
            $string .= '=';
        }
        $string .= ' ' . $self->getVersionString ();
    }
    return $string
}


1;
