#=========================================
# Specifies the custom_input methods for
# the AddHostRoles and HostRoleProperties parameters
#=========================================
package LCM::App::AddHostRolesUtils;
use strict;

use base qw(Exporter);
use SDB::Install::MsgLst;
use SDB::Install::Tools qw(askConfirmation printTableToArrayOfLines);
use SDB::Install::Globals qw(getHumanReadableRolesString GetHostRoleProperties GetAllRoles $gProductNameSystem);
our @EXPORT = qw (readInteractiveAddHostRolesInput getPossibleRolesForExistingHosts);

sub readInteractiveAddHostRolesInput {
    my ($console, $config) = @_;
    my $existingHostToValidRolesMap = getPossibleRolesForExistingHosts($config);
    if (! scalar(keys %$existingHostToValidRolesMap)) {
        $config->PushError("There are no available roles that can be assigned to the $gProductNameSystem host(s).");
        return undef;
    }
    my $invalidHostsMessage = $config->getInvalidHostsForSelectionMessage($existingHostToValidRolesMap);

    my $oldContext = $config->setMsgLstContext([new SDB::Install::MsgLst(), new SDB::Install::MsgLst()]);
    my @hostsValidForAdditionalRoles = sort(keys(%$existingHostToValidRolesMap));

    my $shouldUseDefaultsOnly = 0;
    my @hostsSelected = ();
    my $TABLE_LENGTH_WITH_ONE_VALUE = 3;
    while (1) {
        my $table = _buildExistingHostsTable($config, \@hostsValidForAdditionalRoles);
        my $hasOnlySingleEntry = (scalar @$table == $TABLE_LENGTH_WITH_ONE_VALUE) ? 1 : 0;
        my $defaultIndex = $hasOnlySingleEntry ? "[1]" : "";
        my $action = sprintf("Enter comma-separated list of selected indices %s: ", $defaultIndex);
        
        my $dummyParam = {type => 'csv', valid_values => \@hostsValidForAdditionalRoles};
        if ($config->getUseDefault('AddRoles') && $hasOnlySingleEntry) {
            my $defaultHost = $console->convertFromIndexToValue($dummyParam, 1);
            my $numberOfAdditionalRoles = scalar(@{$existingHostToValidRolesMap->{$defaultHost}});
            $shouldUseDefaultsOnly = ($numberOfAdditionalRoles == 1);
        }

        my $input = '';
        if ($shouldUseDefaultsOnly) {
            $input = 1;
        } else {
            print("$invalidHostsMessage\n") if (defined $invalidHostsMessage); #_TODO_ test if this is actually useless
            print("Select hosts to which you would like to assign additional roles\n\n");
            _printTable($table);
            print($action);
            chomp($input = <STDIN>);
        }
        $input = 1 if ($input eq '' && $hasOnlySingleEntry);
        my $csvHosts = $console->convertFromIndexToValue($dummyParam, $input);
        next if (!defined $csvHosts);
        if ($csvHosts eq '') {
            print "Please select at least one host.\n";
            next;
        }
        @hostsSelected = split(',', $csvHosts);
        print "\n";
        last;
    }

    for my $host (@hostsSelected) {
        while (1) {
            my $table = _buildSelectAdditionalRolesTable($config, $existingHostToValidRolesMap->{$host});
            my $hasOnlySingleEntry = (scalar @$table == $TABLE_LENGTH_WITH_ONE_VALUE) ? 1 : 0;
            my $defaultIndex = $hasOnlySingleEntry ? "[1]" : "";
            my $action = sprintf("Enter comma-separated list of additional roles for host '$host' %s: ", $defaultIndex);

            my $input = '';
            if ($shouldUseDefaultsOnly) {
                $input = 1;
            } else {
                print "\nSelect additional host roles for host '$host'\n\n";
                _printTable($table);
                print($action);
                chomp($input = <STDIN>);
            }
            my $dummyParam = {type => 'csv', valid_values => $existingHostToValidRolesMap->{$host}};
            $input = 1 if ($input eq "" && $hasOnlySingleEntry);
            my $csvRoles = $console->convertFromIndexToValue($dummyParam, $input);
            next if (!defined $csvRoles);
            if ($csvRoles eq '') {
                print "Please select at least one host role.\n";
                next;
            }
            if (!$config->setMapValueItem('AddRoles', $host, $csvRoles)) {
                print($config->getErrorString() . "\n");
                $config->resetError();
                next;
            }
            last;
        }
    }
    $config->setMsgLstContext($oldContext);
    return 1;
}

sub getPossibleRolesForExistingHosts {
    my ($config) = @_;

    my $oldContext = $config->setMsgLstContext([new SDB::Install::MsgLst(), new SDB::Install::MsgLst()]);
    my %existingHostToValidRolesMap = ();
    my $instance = $config->getOwnInstance();
    my $allHosts = $instance->get_allhosts();
    my $allRoles = GetAllRoles();
    for my $possibleRole (@$allRoles) {
        for my $host (@$allHosts) {
            my $canAssignRole = $config->checkAddRoles($host, $possibleRole);
            next if (! $canAssignRole);
            $existingHostToValidRolesMap{$host} = [] if (! exists $existingHostToValidRolesMap{$host});
            push(@{$existingHostToValidRolesMap{$host}}, $possibleRole);
        }
    }
# checkAddRoles above just filters the valid roles and should
# not unskip the ASE passwords that's why we re-skip them
    _skipAcceleratorPasswords($config);
    $config->setMsgLstContext($oldContext);
    return \%existingHostToValidRolesMap;
}

sub _skipAcceleratorPasswords {
    my ($config) = @_;
    $config->setSkip('AcceleratorUser');
    $config->setSkip('AcceleratorPassword');
    $config->setSkip('SystemUser');
    $config->setSkip('SQLSysPasswd');
}

sub _buildExistingHostsTable {
    my ($config, $hostsValidForAdditionalRoles) = @_;
    my $roleProperties = GetHostRoleProperties();
    my $instance = $config->getOwnInstance();
    my $hostsWithRolesHash = $instance->getHostRolesInfo();
    my @hostRolesColumn = ();
    for my $host (@$hostsValidForAdditionalRoles) {
        my $hostRoles = $hostsWithRolesHash->{$host};
        my $humanReadableRoles = getHumanReadableRolesString(split(' ', $hostRoles));
        push(@hostRolesColumn, $humanReadableRoles);
    }
    my $columnsArray= [$hostsValidForAdditionalRoles, \@hostRolesColumn];
    return _buildTable($columnsArray, ['Index', 'System host', 'Roles']);
}

sub _buildSelectAdditionalRolesTable {
    my ($config, $validRoles) = @_;
    my $roleProperties = GetHostRoleProperties();

    my @roles = ();
    my @roleDescription = ();
    for my $role (@$validRoles) {
        push @roles, $role;
        push @roleDescription, $roleProperties->{$role}->{str};
    }
    my $columnsArray = [\@roles, \@roleDescription];
    return _buildTable($columnsArray, ['Index', 'Additional Role', 'Role Description']);
}

# Args:
# $colmunsArray => [[entry1.1, entry2.1], [entry1.2, entry2.2], ...]
#   where in 'entryX.Y' 'X' means 'row' and 'Y' means 'column'.
#
# $columnNames => ['HeaderForColumn1', 'HeaderForColmun2' ...]
#
sub _buildTable {
    my ($columnsArray, $columnNames) = @_;

    my @table = ();
    push @table, $columnNames;
    my $numOfColumns = @$columnsArray;
    my $numOfEntries = @{$columnsArray->[0]};

    my $index = 1;
    while ($index <= $numOfEntries) {
        my $row = [$index];
        for (my $columnNumber = 0; $columnNumber < $numOfColumns; $columnNumber++) {
            my $currentColumn = $columnsArray->[$columnNumber];
            push (@$row, $currentColumn->[$index - 1]);
        }
        push @table, $row;
        $index++;
    }
    return printTableToArrayOfLines(\@table, ' | ', 1);
}

sub _printTable {
    my ($indexTable) = @_;
    print '  ' . join ("\n  ", @$indexTable) . "\n\n";
}

1;
