package LCM::Component::Installable::InstallationKitChecker;

use SDB::Install::SysVars qw($path_separator);
use LCM::FileUtils qw(readFile);
use LCM::VerificationError;
use File::Basename;
use File::Spec;
use Fcntl ':mode';
use SDB::Install::System;
use File::stat;
use SDB::Install::Tools qw(isMountPoint);
use strict;

sub new { return bless({}, $_[0]); }

sub isKitOwnedByRoot {
	my ($self, $rootDirectory, $filelistFileName) = @_;
	
	my $filelistFilePath = File::Spec->catfile($rootDirectory, $filelistFileName);
	my $filelistLines = readFile($filelistFilePath);
	return 1 if(!defined($filelistLines)); 
# We could not read the contents of filelist.* so we assume, that this is undetermined situation
# and proceed like the kit is owned by the root user. This behaviour should be clarified with Ivan and Rocco
# and we have to make sure that we handle the error correctly. Same applies when we discover that a file from
# the filelist.* is missnig (first line in the for loop few lines bellow).
	
	chomp(@{$filelistLines});
	my ($rc, $filesWithWrongPermissions) = (1, []);
	for my $path (@{$self->_getListOfFilesForChecking($rootDirectory, $filelistLines)}){
		next if(!$self->_checkExistingPath($path));
		return 0 if(!$self->_checkPathPermissions($path));
	}
	return 1;
}

sub _getListOfFilesForChecking {
	my ($self, $rootPath, $filelist) = @_;
	my %result = ("$rootPath" => 1);
	for my $file (@{$filelist}){
		my $path = $rootPath;
		for my $subPath (File::Spec->splitdir($file)){
			$path = File::Spec->catfile($path, $subPath);
			$result{$path} = 1;
		}
	}
	return [keys(%result)];
}

sub _checkExistingPath {
	my ($self, $path) = @_;
	return -e $path;
}

sub _checkPathPermissions {
	my ($self, $path) = @_;
	my $stat = File::stat::stat($path);
	my $isWritableForGroup = ($stat->mode() & S_IWGRP) != 0;
	my $isWritableForOthers = ($stat->mode() & S_IWOTH) != 0;

	return 0 if($stat->uid() != 0); # Owner is not root - SHA check will fail
	return 0 if($stat->gid() != 0 && $isWritableForGroup); # Too loose permissions - SHA check will fail
	return 0 if($isWritableForOthers); # Too loose permissions - SHA check will fail
	return 1;
}

sub isKitNotWritableForOthers{
    my ($self, $installPath, $errorMsgLst) = @_;
    my $checkFilePermissionsWalker = $self->getDirectoryWalker(\&checkFileNotWritableForOthers, $self);
    $checkFilePermissionsWalker->setMsgLstContext([undef, $errorMsgLst]);
    $checkFilePermissionsWalker->findAndProcess($installPath);
    my $rc = !$checkFilePermissionsWalker->errorState();
    $rc = $self->checkDirectoryStructureNotWritableForOthers($installPath, $errorMsgLst) && $rc;
    return $rc;
}

sub checkFileNotWritableForOthers{
    my ( $self,$errlst,$msglst,$userData,$globalRootDir,$relentry,$file,$isDir,$statbufRef,$isSymlink,$lstatbufRef) = @_;
    my $fullpath = File::Spec->catfile($globalRootDir, $relentry);
    my $mode = $statbufRef->[2];
    my $isWritableForOthers = ($mode & S_IWOTH);
    if($isWritableForOthers){
        $errlst->addError("File $fullpath is world-writable; too loose permissions.") if (defined $errlst);
        return undef;
    }

    return 1;
}

sub _getParentDir{
    my($self,$path) = @_;
    my $parentdir = undef;
    if(SDB::Install::System::isLink($path)){
        my $linkpath = SDB::Install::System::readLink($path);
        $parentdir = dirname($linkpath);
        if(!File::stat::stat($parentdir)){
            return undef;
        }
    }  else {
        $parentdir = dirname($path);
    }
    return $parentdir;
}

sub checkDirectoryStructureNotWritableForOthers{
    my ($self,$installPath,$errorMsgLst) = @_;
    my $rootDir = File::Spec->rootdir();
    my $rootStat = File::stat::stat($rootDir);
    my $rc = 1;
    my $parentdir = $installPath;
    while( $parentdir ne $rootDir ) {
        if(SDB::Install::Tools::isMountPoint($parentdir)){ # is a mountpoint - skip further checks
            $rc = 1;
            last;
        }
        my $stat_obj = File::stat::stat($parentdir);
        if ($stat_obj->mode() & S_IWOTH){
             $errorMsgLst->addError("Directory $parentdir is world-writable; too loose permissions.") if( defined($errorMsgLst));
             $rc = undef;
             last;
        }
        my $nextParentdir = $self->_getParentDir($parentdir);
        if(! defined $nextParentdir){
             $errorMsgLst->addError("Cannot get parent of $parentdir.") if( defined($errorMsgLst));
             $rc = undef;
             last;
        }
        $parentdir = $nextParentdir;
     }
     return $rc;
}

sub scanForPipes {
    my ($self, $componentPath, $errMsgLst) = @_;
    $errMsgLst->initMsgLst();
    my $dirWalker = $self->getDirectoryWalker(\&checkFileIsPipe, $self);
    $dirWalker->setMsgLstContext([undef, $errMsgLst]);
    $dirWalker->findAndProcess($componentPath);
    return !$dirWalker->errorState();
}

sub checkFileIsPipe {
    my ($self, $errorList, undef, $userData, $rootDir, $relentry, $entry, $isDir) = @_;
    my $fullPath = File::Spec->catfile($rootDir, $relentry);
    my $file = File::stat::stat($fullPath);
    return 1 if (!defined $file); # handle broken symlinks

    my $mode = $file->mode();
    if(S_ISFIFO($mode)) {
        $errorList->addError("File '$fullPath' is a named pipe.");
    }
    return 1;
}

sub getDirectoryWalker {
    my ($self, $actionSub, $actionSubObj) = @_;
    return SDB::Install::DirectoryWalker->new(undef,
                                              undef,
                                              undef,
                                              undef,
                                              undef,
                                              undef,
                                              $actionSub,
                                              $actionSubObj,
                                              undef,
                                              0,  # depth-first
                                              1,  # dont collect list
                                              1); # follow symlinks
}

1;