package LCM::Component::Installable::ReferenceData;

use File::Basename;
use SDB::Install::SysVars qw ($isWin $path_separator);
use SDB::Install::System qw (isAdmin);
use SDB::Install::Globals qw ($gLogDir);
use LCM::ProcessExecutor;
use LCM::Component qw ( ONLINE_PHASE );

use strict;

use parent qw (LCM::Component::Installable Exporter);
our @EXPORT = qw ( INSTALLATION_PATH_MODE );

use constant INSTALLATION_PATH_MODE => "0750";

sub installComponent {
	my ( $self, $configuration, ) = @_;

	my $msg = $self->getMsgLst ()->addProgressMessage ($self->getProgressMsg() . '...');
	my $messageContext = $self->setMsgLstContext([$msg->getSubMsgLst ()]);
	$self->initProgressHandler ();
	$self->_setMessage($msg);
	$self->_setContext($messageContext);

	my $message = sprintf("%s %s", $self->isUpdate ? 'Update' : 'Install', $self->getComponentName());

    if(!$self->isUpdate() && !$self->resetReferenceDataPathParam($configuration)){
        my $refdataPathDescr = $configuration->getParamDesc($configuration->{params}->{ReferenceDataPath});
        my $errMessage = "Value of $refdataPathDescr could not be reset in scriptserver.ini file";
        $self->setErrorMessage($errMessage);
        $msg->endMessage (undef, $message);
        return 0;
    }

    my $exitCode = $self->_runWrappedCommand($configuration, 1);
    $msg->endMessage (undef, $message);
	$self->setMsgLstContext($messageContext);
	return $exitCode;
}

sub updateComponent {
	my $self = shift;
	return $self->installComponent(@_);
}

sub _runWrappedCommand {
    my ($self,$configuration, $isVerbose) = @_;
    my $command = $self->getExecutable();
    my $arguments = $self->_buildArgs($configuration);
    my $stdinPassword = [ $configuration->getValue('SQLSysPasswd') ];
    my $sid = $configuration->getSID();
    my $installer = new LCM::Installer();
    my $runtimeDirectory = $installer->getRuntimeDir();
    my $wrapperCommand = File::Spec->catfile(dirname($runtimeDirectory), 'hdblcm');
    my $wrappedArguments = [ '--main', 'LCM::EnvironmentWrapper::main', $sid, $command, @{$arguments} ]; # LCM::EnvironmentWrapper will switch to sidadm if root

    if(!$isWin){ # Ensure that '$command' has execute permissions
       my (undef, undef, $mode) = stat($command);
       chmod(($mode & 0777) | 0111, $command);
    }

    my $executor = new LCM::ProcessExecutor($wrapperCommand, $wrappedArguments, $stdinPassword);
    $executor->setOutputHandler($self->getMsgLst()->getProgressHandler()) if($isVerbose);
    $executor->setMsgLstContext($self->getMsgLstContext());

    my $exitCode = $executor->executeProgram();
    $self->setLogLocation($self->parseLogFileLocation($executor->getOutputLines()));

    if (!defined $exitCode || $exitCode){
       my $errorMessageList = $self->getHdbInstallerErrorMessages( $executor->getOutputLines() );
       my $message = sprintf("%s of %s failed", $self->isUpdate ? 'Update' : 'Installation', $self->getComponentName());
       $self->setErrorMessage ( $message, $errorMessageList->isEmpty() ? $executor->getErrMsgLst() : $errorMessageList);
   }
    return defined($exitCode) && $exitCode == 0 ? 1 : undef;
}

sub getExecutable {
    my ($self) = @_;
    return File::Spec->catfile($self->getInstallerDir(), 'refdatainst.sh');
}

sub getNumberOfExpectedOutputLines{
	return 30;
}

sub getDefaultSelection{
	my ($self, $stackUpdate) = @_;

	if (!$stackUpdate) {
		return 0;
	}

	my $installedComponent = $self->getInstalledComponent();
	if (!$installedComponent) {
		return 0;
	}

	my $newVersion = new SDB::Install::Version(split('\.', $self->getVersion()));
	my $installedVersion = new SDB::Install::Version(split('\.', $installedComponent->getVersion()));
	return ($installedVersion->isNewerThan($newVersion)) ? 0 : 1;
}

sub _buildArgs {
	my ($self, $configuration) = @_;

	my $sapSystem = $self->getSAPSystem ( $configuration );
	my $instanceNumber = $sapSystem->getNewDBInstanceNr();
	my $installationDirectory = $sapSystem->get_globalSidDir();
	
	my @arguments = ();
	push( @arguments, '-sid');
	push( @arguments, $configuration->getSID ());
	push( @arguments, '-inst');
	push( @arguments, $instanceNumber);
	push( @arguments, '-path');
	push( @arguments, $configuration->getValue('ReferenceDataPath'));
	push( @arguments, '-hdbhost');
	push( @arguments, $configuration->getOwnInstance()->getActiveMaster());
	if ( ! $self->_shouldActivate( $configuration ) ) {
		push( @arguments, '-no_activate');
	}

	my $user = $configuration->getValue ('SystemUser');
	if (defined $user ) {
		push( @arguments, '-hdbuser');
		push( @arguments, $user);
	}

	push( @arguments, '-log_dir');
	push( @arguments, $gLogDir);

	push @arguments, @{$self->SUPER::_buildArgs($configuration)};
	return \@arguments;
}

sub _shouldActivate {
    my ($self, $configuration) = @_;
    my $keyname = $self->getComponentKeyName();
    my @remainingRefDataForUpdate = grep{$_ ne $keyname} @{$configuration->{references_list}};
    $configuration->{references_list} = \@remainingRefDataForUpdate;
    my $remainingRefDataCount = scalar(@remainingRefDataForUpdate);
    return ($remainingRefDataCount == 0) ? 1 : 0;
}

sub getSAPSystem {
	my ( $self, $configuration, $msg, $saveCntxt) = @_;

	my $systems = $configuration->CollectSAPSystems (undef, 1);
	my $sid = $configuration->getValue("SID");
	my $sapSystem = $systems->{$sid};

	if (not defined $sapSystem) {
		$self->setErrorMessage ("No SAP system '$sid' found");
		$self->_getMessage()->endMessage( undef, 'Install ' . $self->getComponentName());
		$self->setMsgLstContext($self->_getContext());
		return undef;
	}

	return $sapSystem;
}

sub getSlppLogFileName {
    my ($self) = @_;
    my $manifest = $self->getManifest();
    my $componentKey = $manifest->getComponentKey();
    return "refdatainst_$componentKey.log"
}


sub resetReferenceDataPathParam {
    my ($self, $configuration) = @_;
    my $instance = $configuration->getOwnInstance();
    my $refDataPath = $instance->getReferenceDataPath();

    my $rc = 1;
    if($refDataPath && $refDataPath =~ /\$\(.*\)/){
        $rc = $instance->setReferenceDataPath('');
    }
    return $rc;
}

sub _setMessage{
	$_[0]->{_message} = $_[1];
}

sub _getMessage {
	return $_[0]->{_message};
}

sub _setContext{
	$_[0]->{_context} = $_[1];
}

sub _getContext {
	return $_[0]->{_context};
}

sub requiresSqlSysPasswd{
    return 1;
}

1;
