package LCM::Component::Installed::CockpitStack;

use strict;
use base 'LCM::Component::Installed';
use SDB::Install::System qw(deltree);
use SDB::Install::Globals qw($gProductNameEngine
                             $gKeynameXS2
                             $gShortProductNameXS2
                             $gShortProductNameCockpit
                             $gProductNameCockpit
                             $gEvenShorterProductNameXSA);
use SDB::Install::MsgLst;


sub uninstallComponent{
    my ($self, $instconfig) = @_;
    my $compName = $self->getComponentName();
    my $msg = $self->getMsgLst()->addProgressMessage("Uninstalling $compName...");
    my $saveCtx = $self->setMsgLstContext([ $msg->getSubMsgLst() ]);
    $self->getMsgLst()->getProgressHandler()->incrementIndentationDepth();

    my $rc = $self->_uninstallCockpitStack($instconfig);

    $self->getMsgLst()->getProgressHandler()->decrementIndentationDepth();
    $msg->endMessage(undef, "Uninstall $compName");
    $self->setMsgLstContext($saveCtx);
    return $rc;
}

sub _uninstallCockpitStack {
    my ($self, $instconfig) = @_;
    my $cmpName = $self->getComponentName();
    my $scm = $instconfig->getSystemComponentManager();

    if ($scm->isHDBServerComponentSelected()) {
        $self->getMsgLst()->addMessage("$cmpName will be uninstalled during the uninstallation of $gProductNameEngine.");
        return 1;
    }

    my $rc = 1;
    if ($instconfig->isCockpitOnPlatformAction()) {
        $rc &&= $self->_deleteCockpitSpace($instconfig);
        $rc &&= $self->_deleteCockpitTenant($instconfig);
# Logout here because it is possible that the tenant db will be unregistered from XSA before deletion
# and this will avoid the need to make a second xs login call
        $instconfig->getXSClient()->logout();
    }

    $rc &&= $self->_deleteCockpitComponent();
    return $rc;
}

sub _deleteCockpitTenant {
    my ($self, $instconfig) = @_;
    my $systemUsername = $instconfig->getValue('SystemUser');
    my $systemPassword = $instconfig->getValue('SQLSysPasswd');
    my $tenantDatabase = $instconfig->getValue('CockpitTenant');
    if ($instconfig->getValue('KeepCockpitTenant')) {
        return 1;
    }

    $self->getMsgLst()->addProgressMessage("Deleting tenant database '$tenantDatabase'...");
    my ($rc, $sql) = $instconfig->createSQLConnection($systemUsername, $systemPassword, 'SYSTEMDB');
    if (!$rc) {
        $self->setErrorMessage("Failed to connect to SYSTEMDB.", $sql->getErrMsgLst());
        return 0;
    }

    my @statements = (
        "ALTER SYSTEM STOP DATABASE $tenantDatabase",
        "DROP DATABASE $tenantDatabase"
    );

    for my $sqlStmt (@statements) {
        $self->getMsgLst()->addMessage("Executing '$sqlStmt'.");
        if (!$sql->execute($sqlStmt)) {
            $self->setErrorMessage("Failed to execute '$sqlStmt'.", $sql->getErrMsgLst());
            return 0;
        }
    }

    $self->_deleteUserConfigEntry($instconfig);
    $self->_unregisterTenant($instconfig);
    return 1;
}

sub _unregisterTenant {
    my ($self, $configuration) = @_;
    my $xsClient = $configuration->getXSClient();
    $xsClient->setMsgLstContext($self->getMsgLstContext());
# Listing tenant databases somehow magically clears the xsa binding with already
# deleted tenants, which otherwise break the XSA update
    return $xsClient->listTenantDatabases();
}

sub _deleteUserConfigEntry {
    my ($self, $instconfig) = @_;
    my $sapSys = $instconfig->getSAPSystem();
    my $configFile = $sapSys->getUserConfigFile();
    $self->getMsgLst()->addMessage("Removing $gShortProductNameCockpit tenant database entry from user config file '$configFile'...");

    if (!$sapSys->saveConfig(undef, ['cockpit_tenant_db'])) {
        $self->getMsgLst()->addWarning("Failed to remove the $gShortProductNameCockpit tenant database name from the user config file.");
    }
    return 1;
}

sub _determineXSSpace {
    my ($self, $configuration) = @_;
    my $xsClient = $configuration->getXSClient();
    my $spaceGuid = $configuration->getValue('CockpitSpaceGUID');
    return $spaceGuid ? $xsClient->retrieveSpaceNameByGUID($spaceGuid) : $configuration->getValue('CockpitXSSpace');
}

sub _deleteCockpitSpace {
    my ($self, $instconfig) = @_;
    if ($instconfig->getSystemComponentManager->isComponentSelected($gKeynameXS2)) {
        $self->getMsgLst()->addMessage("Skipping deletion of the $gShortProductNameXS2 space. It will be deleted together with $gShortProductNameXS2.");
        return 1;
    }

    my $xsClient = $instconfig->getXSClient();
    my $msg = $self->getMsgLst()->addProgressMessage("Deleting $gShortProductNameXS2 space for $gShortProductNameCockpit ...");
    my $errorList = SDB::Install::MsgLst->new();
    $xsClient->setMsgLstContext([ $msg->getSubMsgLst(), $errorList ]);

    if (!$xsClient->login()) {
        $self->setErrorMessage("Failed to login into $gShortProductNameXS2.", $errorList);
        return 0;
    }

    my $xsSpace = $self->_determineXSSpace($instconfig);
    if (!$xsSpace) {
        my $guid = $instconfig->getValue('CockpitSpaceGUID') || '<not-defined>';
        $self->setErrorMessage("Failed to retrieve $gEvenShorterProductNameXSA space ($guid)");
        return 0;
    }

    if (!$xsClient->stopSpace($xsSpace)) {
        $self->setErrorMessage("Failed to stop the applications in the $gShortProductNameXS2 space '$xsSpace'.", $errorList);
        return 0;
    }

    if (!$xsClient->deleteSpace($xsSpace)) {
        $self->setErrorMessage("Failed to delete $gShortProductNameXS2 space '$xsSpace'.", $errorList);
        return 0;
    }

    return 1;
}

sub _deleteCockpitComponent {
    my ($self) = @_;
    my $compName = $self->getComponentName();
    my $installationPath = $self->getInstallationPath();
    my $errorList = SDB::Install::MsgLst->new();
    $self->getMsgLst()->addProgressMessage("Deleting $compName...");

    if (!deltree($installationPath, undef, $errorList)) {
        $self->setErrorMessage("Failed to delete $compName", $errorList);
        return 0;
    }

    return 1;
}

sub getComponentName {
	return "$gProductNameCockpit Stack";
}

sub getSlppLogFileName {
    return 'cockpit_stack.log';
}

sub getNumberOfExpectedOutputLines {
    return 24;
}

1;
