package LCM::Component::Installed::ResidentInstaller;

use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::Globals qw($gDirNameResidentInstaller);
use SDB::Install::System qw(deltree copy_tree);
use SDB::Common::BuiltIn;
use LCM::Component qw (COMPONENT_NAME_INSTALLER);
use File::Spec;
use File::Basename qw(dirname);
use File::stat;
use strict;

use base 'LCM::Component::Installed';

sub uninstallComponent{
    my ( $self, $instconfig ) = @_;
    my $compName = $self->getComponentName();
    my $mesg = "Uninstalling $compName...";
    my $imsg = $self->getMsgLst ()->addProgressMessage($mesg);
    my $tgtDir = $self->getInstSidPath($instconfig);
    if (!defined $tgtDir) {
        return undef;
    }
    my $hdblcmDir = File::Spec->catfile($tgtDir, 'hdblcm');
    my $errorList = SDB::Install::MsgLst->new();
    if (!deltree($hdblcmDir, undef, $errorList)) {
        $self->getErrMsgLst()->addError("Failed to uninstall $compName", $errorList);
        return undef;
    }

    $imsg->endMessage(undef, "Uninstall $compName");
    return 1;
}

sub getComponentName {
    return COMPONENT_NAME_INSTALLER;
}

# Override
sub getInstallationPath {
    my ($self) = @_;
    return dirname($self->getPath());
}

sub getBackupPath {
    my ($self) = @_;
    my $instPath = dirname($self->getPath());
    return File::Spec->catfile(dirname($instPath), "${gDirNameResidentInstaller}_old");
}

sub hasBackup {
    my ($self) = @_;
    my $backupPath = $self->getBackupPath();
    return defined(File::stat::stat($backupPath));
}

sub removeBackup {
    my ($self) = @_;
    return 1 if !$self->hasBackup();

    my $backupPath = $self->getBackupPath();
    return deltree($backupPath, undef, $self->getErrMsgLst());
}

sub backupInstallation {
    my ($self) = @_;
    my $installationPath = $self->getInstallationPath();
    my $backupPath = $self->getBackupPath();
    my $cmpName = $self->getComponentName();
    $self->getMsgLst()->addProgressMessage("Creating backup of $cmpName in '$backupPath'...");

    if ($self->hasBackup() && !$self->removeBackup()) {
        $self->getErrMsgLst()->addError("Failed to remove existing backup path '$backupPath'.");
        return undef;
    }

    my $copyMsgLst = SDB::Install::MsgLst->new();
    if (!copy_tree($installationPath, $backupPath, $copyMsgLst, undef, 0)) {
        my @errors = grep { $_->isError() } @{$copyMsgLst->getMessages()};
        $self->getErrMsgLst()->appendMsg($_) for @errors;
        return undef;
    }

    return 1;
}

sub restoreBackup {
    my ($self) = @_;
    my $cmpName = $self->getComponentName();
    $self->getMsgLst()->addProgressMessage("Restoring $cmpName from backup...");

    if (!$self->hasBackup()) {
        $self->getErrMsgLst()->addError("Backup of $cmpName does not exist.");
        return 0;
    }

    my $backupPath = $self->getBackupPath();
    my $targetPath = $self->getInstallationPath();
    return SDB::Common::BuiltIn->get_instance()->rename($backupPath, $targetPath);
}

sub canBeSelectedForUninstall {
    return 0;
}

1;
