package LCM::ComponentManager::InstalledXSApplicationsManager;

use strict;
use warnings;
use parent 'SDB::Install::Base';
use SDB::Common::Utils qw(createXMLParser);
use SDB::Install::Globals qw($gShortProductNameXS2
                             $gEvenShorterProductNameXSA
                             $gXSASpaceSAP);
use LCM::Manifests::XS2ApplicationManifest::InstalledXS2ApplicationManifest;
use LCM::FileUtils qw(listDirectory);
use File::Basename qw (basename);
use File::Spec;
use File::stat qw(); # Do not import anything explicitly

my $cmpRegistryRegex = qr/^cmp_registry_([\w-]+)\.xml$/;


sub new {
    my ($class) = @_;
    my $self = $class->SUPER::new();
    $self->setXSComponentRegistry('sap'); # Default registry
    $self->{xsApps} = {};

    return $self;
}

sub setXSComponentRegistry {
    my ($self, $registry) = @_;
    $self->{registry} = $registry
}

sub getXSComponentRegistry {
    my ($self) = @_;
    return $self->{registry};
}

sub getAppsInCurrentRegistry {
    my ($self) = @_;
    my $registry = $self->getXSComponentRegistry();
    return $registry ? $self->getAppsInRegistry($registry) : [];
}

sub getAppsInRegistry {
    my ($self, $registry) = @_;
    return [ values %{$self->{xsApps}->{$registry}} ];
}

sub getAllRegistries {
    my ($self) = @_;
    return [ keys %{$self->{xsApps}} ];
}

# Returns an array, containing apps, detected by all component registries
sub getAllXSApps {
    my ($self) = @_;
    my $registries = $self->getAllRegistries();
    my @allApps = map { @{$self->getAppsInRegistry($_)} } @{$registries};
    return \@allApps;
}

# Tries to retrive an app by key in the current component registry
sub getXsApp {
    my ($self, $key) = @_;
    my $registry = $self->getXSComponentRegistry();
    my $appsInSpace = $registry ? $self->{xsApps}->{$registry} : {};
    return $appsInSpace->{lc($key)};
}

sub addXs2Application {
    my ($self, $app) = @_;
    my $manifest = $app->getManifest();
    my $registryID = $app->getRegistryID();
    my $keyname = $manifest->getKeyname();
    if (!$keyname) {
        $self->appendErrorMessage ("Manifest does not contain 'keyname'");
        return 0;
    }

    $self->{xsApps}->{$registryID}->{lc($keyname)} = $app;
    return 1;
}

sub detectInstalledXsComponents {
    my($self, $targetDir) = @_;
    my $cmpRegistryDir = File::Spec->catfile($targetDir, 'xs', 'cmp-registry');
    my @registryCandidates = grep { $_ =~ $cmpRegistryRegex } @{listDirectory($cmpRegistryDir) // []};

    for my $registryFile (@registryCandidates) {
        my $fullPath = File::Spec->catfile($cmpRegistryDir, $registryFile);
        $self->_detectXSComponentsByRegistry ($fullPath);
    }

    return 1;
}

sub _detectXSComponentsByRegistry {
    my ($self, $registryFile) = @_;
    my $message = $self->getMsgLst()->addMessage("Detecting installed $gEvenShorterProductNameXSA applications from registry file '$registryFile'...");
    my $statObject = File::stat::stat($registryFile);
    if (! -f $statObject) {
        $message->getSubMsgLst()->addMessage("No installed XS applications found");
        return 1;
    }

    my $xmlTree = undef;
    eval {
        my $xmlParser = createXMLParser();
        $xmlTree = $xmlParser->parse_file($registryFile);
    };

    if ($@ || !$xmlTree) {
        $message->getSubMsgLst()->addWarning("Cannot detect the already installed $gShortProductNameXS2 components");
        $message->getSubMsgLst()->addWarning("Cannot parse component registry file '$registryFile': $@") if $@;
        return 1;
    }

    my $componentNodes = $xmlTree->getElementsByTagName('component') || [];
    for my $componentNode (@{$componentNodes}){
        my $manifest = LCM::Manifests::XS2ApplicationManifest::InstalledXS2ApplicationManifest->new($componentNode);
        my $xsApp = LCM::Component::Installed->new(undef, $manifest, $self, $self->{instconfig});
        my $xsAppName = $xsApp->getComponentName();
        my $xsAppVersion = $xsApp->getVersion();
        my ($registryID) = basename($registryFile) =~ $cmpRegistryRegex;
# The component registry files have the format cmp_registry_<guid>.xml
# Only the sap space registry has the format cmp_registry_sap.xml.
        $xsApp->setRegistryID($registryID);

        my $appMessage = $message->getSubMsgLst()->addMessage("Detected '$xsAppName' version $xsAppVersion");
        if (!$self->addXs2Application($xsApp)) {
            $appMessage->getSubMsgLst()->addWarning("Couldn't add the detected XS Application '$xsAppName'");
        }
    }

    return 1;
}


1;