package LCM::ComponentManager::MediumComponentManager;

use strict;
use File::Basename qw (dirname);
use LCM::Mid::DUParser;
use LCM::Component::Installable;
use SDB::Install::SysVars qw($path_separator $isWin $currentPlatformName);
use SDB::Install::System qw(isRegularFile);
use SDB::Install::DirectoryWalker;
use LCM::Component; # just to import component ids
use LCM::Manifests::ManifestFactory;
use base qw (LCM::ComponentManager Exporter);
use SDB::Install::DebugUtilities qw(dumpThings);
use File::Spec;
use File::stat qw();
use LCM::FileUtils qw(listDirectory);

use Exporter;
use SDB::Install::Globals qw ($gDirNameEs
                              $gDirNameStreaming
                              $gDirNameXS2
                              $gProductNameInstaller
                              $gProductNameCockpit
                              $gDirNameReferenceData
                              $gKeynameEngine
                              $gKeynameClient
                              $gKeynameStudio
                              $gKeynameAFL
                              $gKeynameLCA
                              $gKeynameHLM
                              $gKeynameSDA
                              $gKeynameStreaming
                              $gKeynameEs
                              $gKeynameAccelerator
                              $gKeynameXS
                              $gKeynameXS2
                              $gKeynameOfficeClient
                              $gKeynameInstaller
                              $gKeynameCockpitStack
                              $gKeynameLSS
                              $gFlavourPlatform
                              $gFlavourCockpit);

our @EXPORT = qw (@componentDirNames);

#es must be before the server in this list !!!
our @componentDirNames = ('packages',
                          $gDirNameEs,
                          $gDirNameStreaming,
                          $gDirNameXS2,
                          $gDirNameReferenceData,
                          'asehana',
                          'server',
                          'client',
                          'studio');

my $SUCCESS = 0;
my $ERR_MISSING_MANIFEST = 1;
my $ERR_DUPLICATE_COMPONENTS = 2;
my $ERR_COMPILING_COMPONENT_LIST = 3;
my $ERR_ONLY_SERVER_DETECTED = 4;
my $ERR_OTHER_PLATFORM_COMPONENT_DETECTED = 5;

sub new {
    my ($class, $instconfig) = @_;
    my $self = $class->SUPER::new ($instconfig);
    $self->{dvdComponents} = [];
    $self->{rootComponents} = [];
    $self->{directoryComponents} = [];
    $self->{defaultLocation} = [];
    return $self;
}

# Override
sub clone {
    my ($self, $config) = @_;
    my $copy = ref($self)->new($config);

    # Copy the actual components lists
    for my $property (qw(dvdComponents rootComponents directoryComponents defaultLocation)) {
        for my $cmp (@{$self->{$property}}) {
            my $manifestCopy = ref($cmp->getManifest())->new($cmp->getManifest()->getFileName());
            my $cmpCopy = ref($cmp)->new($cmp->getPath(), $manifestCopy, $copy, $config, $cmp->isSidAdmUserExecution());
            push @{$copy->{$property}}, $cmpCopy;
            $copy->addComponent($cmpCopy);
        }
    }
    $copy->_compileComponentList();
    $copy->setSystemComponentManager($self->getSystemComponentManager());
    return $copy;
}

sub detectComponentsByDVDRoot {
	my ($self,$dvdRootPath) = @_;
    my $app = LCM::App::ApplicationContext::getInstance ()->getApplication ();
    $dvdRootPath = $app->getAbsPath ($dvdRootPath);
    $dvdRootPath = File::Spec->canonpath($dvdRootPath);

    my $msg = $self->getMsgLst ()->addMessage ("Checking installation medium '$dvdRootPath'");
    my $saveCntxt = $self->setMsgLstContext ([$msg->getSubMsgLst(), undef]);
    my $parser = LCM::Mid::DUParser->new($dvdRootPath);
    eval{
        $parser->parse();
    };
    if ($@){
        $self->setErrorMessage ("Error reading installation medium: $@");
        return 0;
    }
    my $data_units = $parser->getDataUnits();

    my ($manifestDir,$unitPath, $subDir);
    OUTER: for (keys %$data_units){
        $self->getMsgLst ()->addMessage ("Found data unit '$_'");
        $unitPath = $data_units->{$_}->{'path'};
        for $subDir ( @componentDirNames  ) {
            $manifestDir = $unitPath . $path_separator . $subDir;
            if (-d $manifestDir){
                my $isComponentDirs = 0;
                $self->detectComponentByManifestDir($manifestDir, $self->{dvdComponents}, $isComponentDirs);
                next OUTER;
            }
        }
        $self->_detectXS2ApplicationArchives($unitPath, $self->{dvdComponents});
    }

    $self->setMsgLstContext ($saveCntxt);
    $self->_compileComponentList();
    return 1;
}

sub _compileComponentList {
	my ($self) =  @_;
	my $msg = $self->getMsgLst()->addMessage ("Merging detected components");
	my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst(), undef]);

	#reset the components list
	$self->{components} = {};
	my @mergeOrder = ( $self->{directoryComponents}, $self->{rootComponents}, $self->{dvdComponents}, $self->{defaultLocation} );
	my $warningMessage = "";
	$self->{isOnlyServerDetected} = 1;
	
	for my $componentList (@mergeOrder) {
		for my $component (@$componentList) {
			if ( $component->isServer() ) {
				$self->{isServerDetected} = 1;
			} else {
				$self->{isOnlyServerDetected} = 0;
			}
			my $additionSuccessfull = $self->addComponent($component);
		    if (!$additionSuccessfull){
	    	    $self->setErrorMessage ('Cannot add component', $self->getErrMsgLst());
	        	$self->setMsgLstContext ($saveCntxt);
	    	} else {
	    	    my $warning = $self->hasDuplicatedComponentWarning($component);	
	    	    if($warning) {
	    	    	$warningMessage .= $warning;
	    	    }
	    	}
		}
	}

    my $detectedServerComponent = $self->getComponentByKeyName($gKeynameEngine);
    if(defined($detectedServerComponent)){
        my $appContext = LCM::App::ApplicationContext::getInstance();
        $appContext->setServerManifest($detectedServerComponent->getManifest());
    }
    if($warningMessage) {
    	$self->setWarningMessage($warningMessage);
    }
    
    my $rc;
	if ($self->hasComponents()){
        if (!$self->_addResidentInstallerComponent()) {
            $rc = $ERR_COMPILING_COMPONENT_LIST;
        } else {
            $rc = $SUCCESS;
        }
    } else {
    	if ( ! $self->handleDetectionOfServer() ) {
            $rc = $ERR_ONLY_SERVER_DETECTED;
    	} else {
            $rc = $ERR_COMPILING_COMPONENT_LIST;
    	}
    }

    $self->printDetectedComponents();
    return $rc;
}

sub setPrintAllDetectedComponents {
    my($self, $value) = @_;
    $self->{_printNoCache} = $value;
}

sub printDetectedComponents {
    my($self) = @_;
    my @componentList = sort { $a->getComponentKeyName() cmp $b->getComponentKeyName() } (@{$self->getExternalDetectedComponents()});
    for my $handler (@{$self->{_outHandlerCollection}}){
        $handler->printDetectedComponents($self->getConfiguration(), \@componentList, $self->{_printNoCache});
    }

}

# Override
sub addComponent{
    my ($self, $component) = (shift(), shift());
    my $componentKey = $component->getComponentBatchKey();
    my $blackListedComponentBatchKeyes = ['xsac_shine', 'xsac_xmla_int', 'xsac_ha_adp'];
    my $isComponentInBlackList = grep { $_ eq $componentKey } @{$blackListedComponentBatchKeyes};

    # Exclude SHINE app from the detected XSA components. See Bug 123462 for details.
    # Exclude XSAC_XMLA_INT and XSAC_HA_ADP. See TIPHANA15IT10-2378 for details.

    my $skipMsg = sprintf('Skipping component %s', $component->getComponentName());
    if($isComponentInBlackList){
        $self->getMsgLst()->addMessage($skipMsg);
        return 1;
    }

    my $defaultFlavour = $self->getDefaultFlavour();
    if (defined $defaultFlavour && $component->isServer() && $component->getFlavour() ne $defaultFlavour) {
        $self->getMsgLst()->addMessage("$skipMsg because it doesn't match installation kit's flavour");
        return 1;
    }

    return $self->SUPER::addComponent($component, @_);
}

sub handleDetectionOfServer {
	my ( $self ) = @_;
	my $rc = 1;
	if ( $self->{isServerDetected} && $self->{isOnlyServerDetected} && $self->isResidentComponentManager() ) {
        $rc = 0;
	}
	return $rc;
	
}

sub setWarningMessage() {
	my ($self, $warning) = @_;
	$self->resetError();
    my $warningMessage = "The following components were found in more than one location and will be ignored\n". $warning;
    $self->setErrorMessage ($warningMessage);
}

sub hasDuplicatedComponentWarning() {
	my ($self, $component) = @_;
	my $message = undef;
	my $errorMessage = $self->getErrorString();
	if($errorMessage =~/was already found in directory/) {
		$message = "Component " . $component->getComponentName () . "(" . $component->getVersion () . ") in " . $component->getPath () . ".\n";
		$self->resetError();
	}
	return $message;
	
}

sub _addResidentInstallerComponent {
    my ($self) = @_;
    
    # detect resident installer:
    my $stackKitPath = new SDB::Install::Installer()->GetRuntimeDir();

    if($isWin) {
        $stackKitPath =~ s/(.*)\\.+\\?/$1/;
    } else {
        $stackKitPath =~ s/(.*)\/.+\/?/$1/;
    }
    return $self->addResidentInstallerComponent($stackKitPath);
}

sub addResidentInstallerComponent {
    my ($self, $installerPath) = @_;
    my $residentInstallerManifestDir = File::Spec->catfile($installerPath, 'instruntime');
    my $compName = COMPONENT_NAME_INSTALLER;
    my $msg = $self->getMsgLst()->addMessage("Checking directory '$residentInstallerManifestDir' for manifest of '$compName'.");
    my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst(), undef]);
	my $residentInstallerManifest = $self->getResidentInstallerManifestByDir($residentInstallerManifestDir, $saveCntxt);

	if (! defined $residentInstallerManifest) {
		return 0;
	}

    # Installable acts as a factory for its subclasses:
    my $residentInstallerComponent = LCM::Component::Installable->new($residentInstallerManifestDir, $residentInstallerManifest, $self, undef, $self->isSidAdmUserExecution());
    if($residentInstallerComponent->errorState){
        $self->appendErrorMessage("Component '$compName' invalid", $residentInstallerComponent->getErrMsgLst());
        $self->setMsgLstContext($saveCntxt);
        return 0;
    }
    $residentInstallerComponent->addComponentInfosToMsgLst($msg->getSubMsgLst);
    my $additionSuccessfull = $self->addComponent($residentInstallerComponent);
    if(!$additionSuccessfull){
        $self->setErrorMessage("Cannot add component '$compName'.", $self->getErrMsgLst());
        $self->setMsgLstContext($saveCntxt);
        return 0;
    }

    $self->setMsgLstContext($saveCntxt);
}

sub getResidentInstallerManifestByDir {
	my ($self, $residentInstallerManifestDir, $saveCntxt) = @_;
	my $residentInstallerManifestFile = $residentInstallerManifestDir.$path_separator.'manifest';
	my $compName = COMPONENT_NAME_INSTALLER;

    my $residentInstallerManifestStatbuf = File::stat::stat($residentInstallerManifestFile);
	if(!$residentInstallerManifestStatbuf || ! -f $residentInstallerManifestStatbuf) {
		$self->appendErrorMessage("Directory '$residentInstallerManifestDir' does not contain a manifest of '$compName'.");
		$self->setMsgLstContext($saveCntxt) if defined $saveCntxt;
		return undef;
	}

	my $residentInstallerManifest = LCM::Manifests::ManifestFactory::createComponentManifest($residentInstallerManifestFile);
	if($residentInstallerManifest->errorState) {
		$self->appendErrorMessage("Invalid manifest of '$compName'.", $residentInstallerManifest->getErrMsgLst());
		$self->setMsgLstContext($saveCntxt) if defined $saveCntxt;
		return undef;
	}

    my $isPlatformSupported = $residentInstallerManifest->isPlatformSupported($currentPlatformName);
    if(!$isPlatformSupported){
        $self->appendErrorMessage("HDBLCM does not support current OS platform '$currentPlatformName'");
        $self->setMsgLstContext($saveCntxt) if defined $saveCntxt;
        return undef;
    }

	return $residentInstallerManifest;
}

sub clearDetectedComponents {
	my ($self) =  @_;
	$self->SUPER::clearDetectedComponents();
	
	$self->{directoryComponents} = [];
	$self->{rootComponents} = [];
	$self->{dvdComponents} = [];
	$self->{defaultLocation} = [];
}

sub isComponentDir{
	my $directoryName = $_[0];
	return grep ( /^$directoryName$/, @componentDirNames );
}

# returns 0 : success
#         1 : failed to detect by manifest in a directory
#         2 : duplicate components  
#         3 : _compileComponentList failed
#         4 : _compileComponentList failed, Only server is selected
sub detectComponentsByFsRoot {
    my ($self,$rootPath, $isComponentDirs) = @_;
    my $app = LCM::App::ApplicationContext::getInstance ()->getApplication ();
    $rootPath = $app->getAbsPath ($rootPath);
   	my $manifestDirectories = $self->_findManifestDirectories($rootPath);
   	return $self->detectComponentsByDirectoryList( $manifestDirectories ,$self->{rootComponents}, $isComponentDirs );
}

sub _findManifestDirectories {
    my ($self,$unitPath) = @_;
    my @manifestDirectories = ();
    my $dirWalker = SDB::Install::DirectoryWalker->new(
        sub {   # actionMatcher callback, see comments in DirectoryWalker.pm for full signature
            my $name = $_[5];
            my $isDir = $_[6];
            my $relativePath = $_[4];
            if( !$isDir && $name eq 'manifest' ){ #visit only manifest files
                my @pathSegments = File::Spec->splitdir($relativePath);
                return ( ( scalar @pathSegments ) > 1 && isComponentDir( $pathSegments[-2] ) );
            }
            if( !$isDir && $name =~ /\.zip$/i ) { #possible XS2 Application archive
                my $candidateArchivePath = File::Spec->catfile($unitPath, $relativePath);
                my $manifest = LCM::Manifests::ManifestFactory::createComponentManifest($candidateArchivePath);
                return $manifest->isXS2Application() ? 1 : 0;
            }
            return 0;
        },
        undef, undef, undef, undef, undef,
        sub {
            my $traversalRoot=$_[3];
            my $relativePath = $_[4];
            my $manifestFilename=$_[5];
            $relativePath =~ s/\Q$manifestFilename\E//;
            my $manifestDir = $traversalRoot.$path_separator.$relativePath;
            push(@manifestDirectories, $manifestDir);
            return 1;
        },
        undef,undef,
        0,     # no breadth-first traversal, so we do it depth-first.
        1,    # dont collect list
        1     # follow symlinks
    );
    $dirWalker->findAndProcess($unitPath);

    my @fileList = grep { isRegularFile($_) } @{listDirectory($unitPath) // []};
    my $hasXSAppArchive = 0;
    for my $fileName (@fileList) {
        my $candidateArchivePath = File::Spec->catfile($unitPath, $fileName);
        my $manifest = LCM::Manifests::ManifestFactory::createComponentManifest($candidateArchivePath);
        if($manifest->isXS2Application()){
            $hasXSAppArchive = 1;
            last;
        }
    }

    # DirectoryWalker does not process root of tree
    if (-f File::Spec->catfile($unitPath, 'manifest') || $hasXSAppArchive) {
        push @manifestDirectories, $unitPath . $path_separator;
    }
    return \@manifestDirectories;
}

sub detectDefaultLocation {
    my ($self,$dirList, $isComponentDirs) = @_;
    return $self->detectComponentsByDirectoryList( $dirList ,$self->{defaultLocation}, $isComponentDirs );
}

# returns 0 : success
# 		  1 : failed to detect by manifest in a directory
#		  2 : duplicate components	
#		  3 : _compileComponentList failed
#         4 : _compileComponentList failed, Only server is selected
#         5 : detected component for another platform in components directory list
sub detectComponentsByDirectoryList {
    my ($self, $dirList, $componentList, $isComponentDirs) = @_;
    my $errorState = $SUCCESS;
    $isComponentDirs = defined($isComponentDirs) ? $isComponentDirs : 1;
    $componentList = defined($componentList) ? $componentList : $self->{directoryComponents};

    $self->resetMsgLstContext();
    my $msg = $self->getMsgLst()->addMessage ("Detecting components from directories");
    my ($messageList, $errorList) = @{$self->setMsgLstContext ([$msg->getSubMsgLst(), undef])};

    if(scalar(@{$dirList}) == 0){
        $self->appendErrorMessage("No components were found in the specified location");
        return $ERR_MISSING_MANIFEST;
    }
    my $app = LCM::App::ApplicationContext::getInstance ()->getApplication ();
    foreach my $dir (@$dirList){
        $dir = $app->getAbsPath ($dir);
        if (! -d $dir && !$dir =~ /\.zip$/i) {
            $errorState = $ERR_MISSING_MANIFEST;
            $self->appendErrorMessage ( "Directory $dir doesn't exist" ) ;
            next;
        }
        $dir = File::Spec->canonpath($dir);

        my $temporaryErrorList = new SDB::Install::MsgLst();
        $self->setMsgLstContext ([undef, $temporaryErrorList]);

        my $detectedComponent = $self->detectComponentByManifestDir ( $dir, $componentList, $isComponentDirs );
        my $successfulDetection = defined $detectedComponent ? 1 : 0;
        if ( ! $successfulDetection ) {
            my @componentSubDirectories = $self->searchSubDirectories ( $dir, $componentList, $isComponentDirs, \$detectedComponent );

            if ( @componentSubDirectories == 1 ) {
               $successfulDetection = 1;
            } elsif ( @componentSubDirectories > 1 ) {
               $self->generateErrorMessage(\@componentSubDirectories, $componentList, $dir, $errorList);
            } else {
               my $detectedXs2ApplicationArchives = $self->_detectXS2ApplicationArchives($dir, $componentList);
               $successfulDetection = $detectedXs2ApplicationArchives > 0 ? 1 : 0;
            }
        }

        if(!$successfulDetection){
            $errorList->appendMsgLst($temporaryErrorList);
            $errorState= $ERR_MISSING_MANIFEST;
        } elsif(defined $detectedComponent && $isComponentDirs && !$self->checkDetectedComponentPlatform($detectedComponent,$errorList)){
            $errorState = $ERR_OTHER_PLATFORM_COMPONENT_DETECTED;
        }
    }
    $self->setMsgLstContext ([$messageList, $errorList]);
    return $errorState if $errorState;

    $errorState = $self->_checkDuplicatedComponents ( $componentList );

    return $errorState if $errorState;
    return $self->_compileComponentList();
}

sub checkDetectedComponentPlatform {
    my ($self, $detectedComponent, $errorList) = @_;
    my $manifest = $detectedComponent->getManifest();
    my $isKnownComponent = $manifest->isKnownComponent();
    if($isKnownComponent && !$detectedComponent->isCurrentPlatformSupported()){
        my $supportedPlatform = join(", ",@{$manifest->getSupportedPlatforms()});
        my $message = sprintf("The location '%s' contains component %s for another platform %s.", $detectedComponent->getPath(), $detectedComponent->getComponentName(), $supportedPlatform);
        $errorList->addError($message);
        return 0;
    }
    return 1;
}

sub generateErrorMessage() {
	my ( $self, $componentSubDirs, $components, $dir, $errorList) = @_;
	my %seen = ();
	foreach my $componentSubDir ( @$componentSubDirs ) {
		$seen{$componentSubDir} = $componentSubDir;
	}
	$errorList->addError("More than one component found in directory $dir.");
	foreach my $component ( @$components ) {
		next if(!exists $seen{$component->getPath()});
		my $message = sprintf("Component %s(%s) was found in %s.", $component->getComponentName(), $component->getVersion(), $component->getPath());
		$errorList->addError($message);
	}
}

sub searchSubDirectories() {
	my ( $self, $parentDir, $componentList, $isComponentDirs, $detectedComponent ) = @_;
	my @componentDirs = ();
	for my $subDir ( @componentDirNames ) {
		my $manifestDir = $parentDir . $path_separator . $subDir;
		if ( -d $manifestDir ) {
			my $component = $self->detectComponentByManifestDir ( $manifestDir, $componentList, $isComponentDirs );
			if ( $component) {
				push @componentDirs, $manifestDir;
                ${$detectedComponent} = $component;
			}
		}
	}
	return @componentDirs;
}

sub _checkDuplicatedComponents() {
	my ( $self, $componentList ) = @_;

	my %componentNameMap = ();

	$self->resetError ();
	my $result = $SUCCESS;
	foreach my $component ( @$componentList ) {
		my $manifest = $component->getManifest ();
        my $componentName = $component->getComponentName ();
		#take into account components only for our platform
		next if (!$manifest->isPlatformSupported($currentPlatformName));
		
		if ( exists $componentNameMap{$componentName} ) {
			my $comonentFromMap = $componentNameMap{$componentName};
			if ( $component->getPath () eq $comonentFromMap->getPath () ) {
				next;
			}
			my $message =
				"The following components were found in more than one location $componentName ("
				. $component->getVersion () . ") in "
				. $component->getPath () . " and in " . $comonentFromMap->getPath ();
			$result = $ERR_DUPLICATE_COMPONENTS;
			$self->appendErrorMessage ( $message );
			last;
		} else {
			$componentNameMap{$componentName} = $component;
		}
	}

	return $result;
}

sub detectComponentByManifestDir{
    my ($self, $manifestDir, $componentCollection, $isComponentDirs ) = @_;
    my ($manifest, $manifestFile, $component);
    $manifestDir = File::Spec->canonpath($manifestDir);

    $manifestFile = $manifestDir . $path_separator . 'manifest';
    my $msg = $self->getMsgLst ()->addMessage ("Checking component directory $manifestDir");
    my $saveCntxt = $self->setMsgLstContext ([$msg->getSubMsgLst(), undef]);
    if (! -f $manifestFile){
        $self->appendErrorMessage ("No component was found in directory '$manifestDir'");
        $self->setMsgLstContext ($saveCntxt);
        return undef;
    }
    $manifest = LCM::Manifests::ManifestFactory::createComponentManifest($manifestFile);
    if ($manifest->errorState){
        $self->appendErrorMessage ('Invalid manifest', $manifest->getErrMsgLst ());
        $self->setMsgLstContext ($saveCntxt);
        return undef;
    }
    if ( $manifest->isServer() ) {
        my $appContext = LCM::App::ApplicationContext::getInstance();
        $appContext->setServerManifest($manifest);
    } elsif ( $manifest->isHLM() && $isComponentDirs ) {
        my $SAPNoteNumber = "2068452";
        $self->appendErrorMessage ("The $gProductNameInstaller component is no longer supported. For more information see SAP Note $SAPNoteNumber.");
        $self->setMsgLstContext ($saveCntxt);
        return undef;
    }
    $component = new LCM::Component::Installable ($manifestDir, $manifest, $self, undef, $self->isSidAdmUserExecution()); # Installable acts as a factory for its subclasses.
    if (!$manifest->isKnownComponent()){
        $self->getMsgLst()->addWarning("Unknown commponent was found in directory '$manifestDir'");
        return $component;
    }
    if ($component->errorState){
        $self->appendErrorMessage ("Invalid component", $component->getErrMsgLst());
        $self->setMsgLstContext ($saveCntxt);
        return undef;
    }
	
    if ( defined($component->getWarnings()) && scalar(@{$component->getWarnings()}) ){
        my @warnings = @{$component->getWarnings()};
        push(@{$self->{instconfig}->{warnings}}, @warnings);
        for my $warning (@{$component->getWarnings()}) {
            $self->getMsgLst()->addWarning($warning);
        }
    }

    $component->addComponentInfosToMsgLst ($msg->getSubMsgLst);
    push @$componentCollection, $component;
    $self->setMsgLstContext ($saveCntxt);
    return $component;
}

# Fix for Bug 166247: manifestPath could be a directory or an archive
sub _detectXS2ApplicationArchives {
	my ($self, $manifestPath, $componentCollection) = @_;
	my $numberOfComponentsFound = 0;
	my $msg = $self->getMsgLst ()->addMessage ("Checking component directory $manifestPath");
	my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst(), undef]);

	my $manifestDir;
	my $fileList;
	if($manifestPath =~ /\.zip$/i){
		my (undef,$directory,$fileName) = File::Spec->splitpath($manifestPath);
		$manifestDir = $directory;
		$fileList = [$fileName];
	} else {
		$manifestDir = $manifestPath;
		$fileList = listDirectory($manifestPath);
	}

	return $numberOfComponentsFound if(!defined($fileList));

	for my $fileName (@{$fileList}){
		next if($fileName !~ /\.zip$/i);

		my $candidateArchivePath = File::Spec->catfile($manifestDir, $fileName);
		my $manifest = LCM::Manifests::ManifestFactory::createComponentManifest($candidateArchivePath);

		next if (!$manifest->isXS2Application());

		my $component = new LCM::Component::Installable($candidateArchivePath, $manifest, $self, undef, $self->isSidAdmUserExecution());
		if ($component->errorState()){
			$self->appendErrorMessage ("Invalid component", $component->getErrMsgLst());
			next;
		}
		push(@{$componentCollection}, $component);
		++$numberOfComponentsFound;
	}
	$self->setMsgLstContext ($saveCntxt);
	return $numberOfComponentsFound;
}

sub isStackUpdate {
	my $server = $_[0]->getComponentByKeyName($gKeynameEngine);
	return defined $server ? $server->isUpdate() : 0;
}

sub getSystemComponentManager{
    my ($self) = @_;
    return $self->{systemComponentManager};
}

sub setSystemComponentManager{
    my ($self, $scm) = @_;
    $self->{systemComponentManager} = $scm;
}

sub getSortedComponentKeynames{
	my ($self) = @_;
	my @sortedComponentKeynames;
	my @serverPlugins = @{$self->getServerPluginComponents()};
	my @referenceDataComponents = @{$self->getReferenceDataComponents()};
	if ($self->isStackUpdate()) {
# Install/update LSS before the plugins because during the installation of the plugins,
# their binaries should be imported in LSS. (Bug: 215524)
        push @sortedComponentKeynames, $gKeynameLSS;

		for my $component( @serverPlugins ) {
			push (@sortedComponentKeynames, $component->getComponentKeyName());
		}
		push (@sortedComponentKeynames, $gKeynameEngine, $gKeynameCockpitStack, $gKeynameInstaller);
	} else {
		push (@sortedComponentKeynames, $gKeynameEngine, $gKeynameLSS, $gKeynameCockpitStack, $gKeynameInstaller);

		for my $component( @serverPlugins ) {
			push (@sortedComponentKeynames, $component->getComponentKeyName());
		}
	}

	my @refetenceDataComponentsKeynames = ();
	push (@refetenceDataComponentsKeynames, $_->getComponentKeyName() ) for ( @referenceDataComponents );

	push (@sortedComponentKeynames,
		$gKeynameClient,
		$gKeynameStudio,
		$gKeynameOfficeClient,
		$gKeynameSDA,
		$gKeynameStreaming,
		$gKeynameEs,
		$gKeynameAccelerator,
		$gKeynameXS2,
		@refetenceDataComponentsKeynames);
	return \@sortedComponentKeynames;
}

sub isHanaOptionSelected {
    my ($self) = @_;

    if ( !$self->isComponentSelected($gKeynameStreaming) &&
         !$self->isComponentSelected($gKeynameEs) &&
         !$self->isComponentSelected($gKeynameAccelerator) &&
         !$self->isComponentSelected($gKeynameXS2) ) {
            return 0;
    }

    return 1;
}

sub getHANAFlavour {
    my($self) = @_;
    my $flavour = $self->SUPER::getHANAFlavour();
    return $flavour if defined ($flavour);

    return $gFlavourCockpit if($self->getComponentByKeyName($gKeynameCockpitStack));

    my $scm = $self->getSystemComponentManager();
    return $gFlavourPlatform unless defined $scm;

    $flavour = $scm->getHANAFlavour() if defined($scm);
    return $flavour // $gFlavourPlatform;
}

sub areDetectedOnlyOtherPlatformComponents {
    my ($self) = @_;
    my $componentsForCurrentPlattform = $self->getAllComponents();
    my $componentsForOtherPlatform = $self->getOtherPlatformComponents();

    if(!@{$componentsForCurrentPlattform} && @{$componentsForOtherPlatform} > 0){
        my $error = "No components were detected for the current platform ($currentPlatformName). Only components for another platform were detected.";
        $error.=" Check if you have provided right components and software locations.";
        $self->appendErrorMessage($error);
        return 1;
    }
    return 0;
}

sub canAddDuplicateComponents {
    my ($self, $foundComponent, $newComponent, $keyName) = @_;
    return 1 if($keyName eq $gKeynameInstaller);
    return $self->SUPER::canAddDuplicateComponents($foundComponent, $newComponent, $keyName);
}

sub getDefaultFlavour {
    my ($self) = @_;
    if (!defined $self->{defaultFlavour}) {
        my $defaultServer = (grep {$_->isServer()} @{$self->{defaultLocation}})[0];
        return undef if (! defined $defaultServer);
        $self->{defaultFlavour} = $defaultServer->getFlavour();
    }
    return $self->{defaultFlavour};
}

sub resetRootComponents {
    my ($self) = @_;
    $self->{rootComponents} = [];
}

sub willInstallComponent {
    my ($self, $componentKeyname, $config) = @_;
    if (!$self->isComponentSelected($componentKeyname)) {
        return 0;
    }
    my $mcmComponent = $self->getComponentByKeyName($componentKeyname);
    my $persFile = $mcmComponent->getPersistenceXMLObject($config);
    if (defined $persFile) {
        return 1 if ($persFile->getActionString() eq 'install');
        return 0;
    }
    my $scm = $self->getSystemComponentManager();
    return !defined $scm || !defined $scm->getComponentByKeyName($componentKeyname);
}

sub addComponentsOutputHandler {
    my($self, $outputHandler) = @_;
    $self->{_outHandlerCollection} //= [];
    push(@{$self->{_outHandlerCollection}}, $outputHandler);
}

sub removeOutputHandler {
    my($self, $hanlder) = @_;
    my $handlers = $self->{_outHandlerCollection};
    my @foundIndex = grep { @$handlers[$_] eq $hanlder } 0..$#$handlers;
    return if(!@foundIndex);
    splice @$handlers, $foundIndex[0], 1;
}

1;
